/*
 * $Id: lm-applet.c,v 1.19 2004/08/15 01:15:57 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <gnome.h>
#include <panel-applet-gconf.h>
#include "lm-applet.h"
#include "lm-util.h"
#include "lm-preferences.h"
#include "lm-sockets.h"
#include "lm-host.h"
#include "lm-icmp.h"

/*** cpp *********************************************************************/

#define ICON_PADDING			1
#define BAR_PADDING			3

#define BAR_SPACING			3
#define BAR_THICKNESS			6

#define CONF_HOSTS			"hosts"
#define CONF_DELAY			"delay"
#define CONF_TIMEOUT			"timeout"
#define CONF_SCALE			"scale"
#define CONF_PREFERENCES_WIDTH		"preferences_width"
#define CONF_PREFERENCES_HEIGHT		"preferences_height"

#define KEY_PROGRESS_BAR		"lm-applet:progress-bar"

/* keep in sync with link-monitor-applet.schemas */
#define DEFAULT_PREFERENCES_WIDTH	-1
#define DEFAULT_PREFERENCES_HEIGHT	387

/* keep in sync with preferences.glade */
#define MIN_SCALE			1

/*** types *******************************************************************/

enum {
  ACTIVATE,
  ACTIVATE_PREFERENCES,
  LAST_SIGNAL
};
    
enum {
  PROP_0,
  PROP_HOSTS,
  PROP_DELAY,
  PROP_TIMEOUT,
  PROP_SCALE,
  PROP_PREFERENCES_WIDTH,
  PROP_PREFERENCES_HEIGHT
};
    
struct _LMAppletPrivate
{
  GSList		*watches;
  GSList		*hosts;
  unsigned int		delay;
  unsigned int		timeout;
  unsigned int		scale;
  int			preferences_width;
  int			preferences_height;
  GtkTooltips		*tooltips;
  GtkWidget		*about;
  unsigned int		reconfigure_timeout_id;
};

typedef struct
{
  GIOChannel		*channel;
  LMSocket		*s;
  LMApplet		*applet;
  unsigned int		source_id;
} WatchInfo;

/*** variables ***************************************************************/

static GObjectClass *parent_class = NULL;
static unsigned int applet_signals[LAST_SIGNAL] = { 0 };

/* functions needed by menu_verbs */
static void lm_applet_preferences_cb (BonoboUIComponent *component,
				      gpointer user_data,
				      const char *cname);
static void lm_applet_help_cb (BonoboUIComponent *component,
			       gpointer user_data,
			       const char *cname);
static void lm_applet_about_cb (BonoboUIComponent *component,
				gpointer user_data,
				const char *cname);

static const BonoboUIVerb menu_verbs[] = {
  BONOBO_UI_VERB("LinkMonitorPreferences", lm_applet_preferences_cb),
  BONOBO_UI_VERB("LinkMonitorHelp", lm_applet_help_cb),
  BONOBO_UI_VERB("LinkMonitorAbout", lm_applet_about_cb),
  BONOBO_UI_VERB_END
};

/*** functions ***************************************************************/

static void lm_applet_class_init (LMAppletClass *class);
static void lm_applet_init (LMApplet *applet);
static void lm_applet_finalize (GObject *object);

static void lm_applet_set_property (GObject *object,
				    guint prop_id,
				    const GValue *value,
				    GParamSpec *pspec);
static void lm_applet_get_property (GObject *object,
				    unsigned int prop_id,
				    GValue *value,
				    GParamSpec *pspec);

static void lm_applet_activate (LMApplet *applet);
static void lm_applet_activate_preferences (LMApplet *applet);

static gboolean lm_applet_factory_cb (PanelApplet *applet,
				      const char *iid,
				      gpointer data);

static gboolean lm_applet_socket_read_cb (GIOChannel *source,
					  GIOCondition condition,
					  gpointer data);

static void lm_applet_reconfigure (LMApplet *applet);
static gboolean lm_applet_reconfigure_timeout_cb (gpointer data);

static gboolean lm_applet_button_press_event_h (GtkWidget *widget,
						GdkEventButton *event,
						gpointer user_data);

static void lm_applet_configure (LMApplet *applet);

static void lm_applet_sync_gconf_hosts (LMApplet *applet);
static unsigned int lm_applet_get_unique_seq (LMApplet *applet);

static LMHost *lm_applet_add_host_internal (LMApplet *applet,
					    const char *name);
static LMHost *lm_applet_replace_host_internal (LMApplet *applet,
						LMHost *old,
						const char *new);
static void lm_applet_swap_hosts_internal (LMApplet *applet,
					   LMHost *host1,
					   LMHost *host2);
static void lm_applet_remove_host_internal (LMApplet *applet, LMHost *host);

static void lm_applet_host_notify_h (GObject *object,
				     GParamSpec *pspec,
				     gpointer user_data);

static void lm_applet_configure_progress_bar (LMApplet *applet, LMHost *host);
static void lm_applet_update_progress_bar (LMApplet *applet, LMHost *host);
static void lm_applet_update_tooltip (LMApplet *applet);

/*** implementation **********************************************************/

GType
lm_applet_get_type (void)
{
  static GType applet_type = 0;
  
  if (! applet_type)
    {
      static const GTypeInfo applet_info = {
	sizeof(LMAppletClass),
	NULL,
	NULL,
	(GClassInitFunc) lm_applet_class_init,
	NULL,
	NULL,
	sizeof(LMApplet),
	0,
	(GInstanceInitFunc) lm_applet_init,
      };
      
      applet_type = g_type_register_static(PANEL_TYPE_APPLET,
					   "LMApplet",
					   &applet_info,
					   0);
    }
  
  return applet_type;
}

static void
lm_applet_class_init (LMAppletClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  GtkBindingSet *binding_set;

  parent_class = g_type_class_peek_parent(class);

  object_class->finalize = lm_applet_finalize;
  object_class->set_property = lm_applet_set_property;
  object_class->get_property = lm_applet_get_property;

  class->activate = lm_applet_activate;
  class->activate_preferences = lm_applet_activate_preferences;

  g_object_class_install_property(object_class,
                                  PROP_HOSTS,
                                  g_param_spec_pointer("hosts",
						       _("Hosts"),
						       _("The list of hosts"),
						       G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_DELAY,
                                  g_param_spec_uint("delay",
						    _("Delay"),
						    _("The amount of time to wait between sending each echo request to a particular host"),
						    LM_HOST_MIN_DELAY,
						    G_MAXUINT,
						    LM_HOST_MIN_DELAY,
						    G_PARAM_WRITABLE | G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_TIMEOUT,
                                  g_param_spec_uint("timeout",
						    _("Timeout"),
						    _("The delay after which a host is considered dead if no reply was received"),
						    LM_HOST_MIN_TIMEOUT,
						    G_MAXUINT,
						    LM_HOST_MIN_TIMEOUT,
						    G_PARAM_WRITABLE | G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_SCALE,
                                  g_param_spec_uint("scale",
						    _("Scale"),
						    _("The round-trip time a full bar represents"),
						    MIN_SCALE,
						    G_MAXUINT,
						    MIN_SCALE,
						    G_PARAM_WRITABLE | G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_PREFERENCES_WIDTH,
                                  g_param_spec_int("preferences-width",
						   _("Preferences width"),
						   _("The width of the preferences dialog"),
						   -1,
						   G_MAXINT,
						   DEFAULT_PREFERENCES_WIDTH,
						   G_PARAM_WRITABLE | G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_PREFERENCES_HEIGHT,
                                  g_param_spec_int("preferences-height",
						   _("Preferences height"),
						   _("The height of the preferences dialog"),
						   -1,
						   G_MAXINT,
						   DEFAULT_PREFERENCES_HEIGHT,
						   G_PARAM_WRITABLE | G_PARAM_READABLE));

  applet_signals[ACTIVATE] = g_signal_new("activate",
					  LM_TYPE_APPLET,
					  G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					  G_STRUCT_OFFSET(LMAppletClass, activate),
					  NULL,
					  NULL,
					  g_cclosure_marshal_VOID__VOID,
					  G_TYPE_NONE,
					  0);
  applet_signals[ACTIVATE] = g_signal_new("activate-preferences",
					  LM_TYPE_APPLET,
					  G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					  G_STRUCT_OFFSET(LMAppletClass, activate_preferences),
					  NULL,
					  NULL,
					  g_cclosure_marshal_VOID__VOID,
					  G_TYPE_NONE,
					  0);

  binding_set = gtk_binding_set_by_class(class);

  gtk_binding_entry_add_signal(binding_set, GDK_Return, 0, "activate", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_ISO_Enter, 0, "activate", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_KP_Enter, 0, "activate", 0);

  gtk_binding_entry_add_signal(binding_set, GDK_Return, GDK_MOD1_MASK, "activate-preferences", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_ISO_Enter, GDK_MOD1_MASK, "activate-preferences", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_KP_Enter, GDK_MOD1_MASK, "activate-preferences", 0);
}

static void
lm_applet_init (LMApplet *applet)
{
  applet->priv = g_new0(LMAppletPrivate, 1);
}

static void
lm_applet_finalize (GObject *object)
{
  LMApplet *applet = LM_APPLET(object);
  GSList *l;

  if (applet->priv->reconfigure_timeout_id)
    g_source_remove(applet->priv->reconfigure_timeout_id);

  LM_LIST_FOREACH(l, applet->priv->hosts)
    g_object_unref(l->data);
  g_slist_free(applet->priv->hosts);

  LM_LIST_FOREACH(l, applet->priv->watches)
    {
      WatchInfo *info = l->data;

      g_source_remove(info->source_id);
      g_io_channel_unref(info->channel);
      g_free(info);
    }
  g_slist_free(applet->priv->watches);
  
  g_object_unref(applet->priv->tooltips);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
lm_applet_set_property (GObject *object,
			guint prop_id,
			const GValue *value,
			GParamSpec *pspec)
{
  LMApplet *applet = LM_APPLET(object);
  GSList *l;

  switch (prop_id)
    {
    case PROP_DELAY:
      applet->priv->delay = g_value_get_uint(value);
      panel_applet_gconf_set_int(PANEL_APPLET(applet), CONF_DELAY, applet->priv->delay, NULL);
      LM_LIST_FOREACH(l, applet->priv->hosts)
	lm_host_set_delay(l->data, applet->priv->delay);
      break;

    case PROP_TIMEOUT:
      applet->priv->timeout = g_value_get_uint(value);
      panel_applet_gconf_set_int(PANEL_APPLET(applet), CONF_TIMEOUT, applet->priv->timeout, NULL);
      LM_LIST_FOREACH(l, applet->priv->hosts)
	lm_host_set_timeout(l->data, applet->priv->timeout);
      break;
      
    case PROP_SCALE:
      applet->priv->scale = g_value_get_uint(value);
      panel_applet_gconf_set_int(PANEL_APPLET(applet), CONF_SCALE, applet->priv->scale, NULL);
      LM_LIST_FOREACH(l, applet->priv->hosts)
	lm_applet_update_progress_bar(applet, l->data);
      break;

    case PROP_PREFERENCES_WIDTH:
      applet->priv->preferences_width = g_value_get_int(value);
      panel_applet_gconf_set_int(PANEL_APPLET(applet), CONF_PREFERENCES_WIDTH, applet->priv->preferences_width, NULL);
      break;

    case PROP_PREFERENCES_HEIGHT:
      applet->priv->preferences_height = g_value_get_int(value);
      panel_applet_gconf_set_int(PANEL_APPLET(applet), CONF_PREFERENCES_HEIGHT, applet->priv->preferences_height, NULL);
      break;
      
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
};

void
lm_applet_set_delay (LMApplet *applet, unsigned int delay)
{
  g_return_if_fail(LM_IS_APPLET(applet));

  g_object_set(G_OBJECT(applet), "delay", delay, NULL);
}

void
lm_applet_set_timeout (LMApplet *applet, unsigned int timeout)
{
  g_return_if_fail(LM_IS_APPLET(applet));

  g_object_set(G_OBJECT(applet), "timeout", timeout, NULL);
}

void
lm_applet_set_scale (LMApplet *applet, unsigned int scale)
{
  g_return_if_fail(LM_IS_APPLET(applet));

  g_object_set(G_OBJECT(applet), "scale", scale, NULL);
}

void
lm_applet_set_preferences_width (LMApplet *applet, int width)
{
  g_return_if_fail(LM_IS_APPLET(applet));

  g_object_set(G_OBJECT(applet), "preferences-width", width, NULL);
}

void
lm_applet_set_preferences_height (LMApplet *applet, int height)
{
  g_return_if_fail(LM_IS_APPLET(applet));

  g_object_set(G_OBJECT(applet), "preferences-height", height, NULL);
}

static void
lm_applet_get_property (GObject *object,
			unsigned int prop_id,
			GValue *value,
			GParamSpec *pspec)
{
  LMApplet *applet = LM_APPLET(object);

  switch (prop_id)
    {
    case PROP_HOSTS:
      g_value_set_pointer(value, lm_applet_get_hosts(applet));
      break;

    case PROP_DELAY:
      g_value_set_uint(value, lm_applet_get_delay(applet));
      break;

    case PROP_TIMEOUT:
      g_value_set_uint(value, lm_applet_get_timeout(applet));
      break;

    case PROP_SCALE:
      g_value_set_uint(value, lm_applet_get_scale(applet));
      break;
      
    case PROP_PREFERENCES_WIDTH:
      g_value_set_int(value, lm_applet_get_preferences_width(applet));
      break;

    case PROP_PREFERENCES_HEIGHT:
      g_value_set_int(value, lm_applet_get_preferences_height(applet));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

GSList *
lm_applet_get_hosts (LMApplet *applet)
{
  g_return_val_if_fail(LM_IS_APPLET(applet), NULL);

  return applet->priv->hosts;
}

unsigned int
lm_applet_get_delay (LMApplet *applet)
{
  g_return_val_if_fail(LM_IS_APPLET(applet), 0);

  return applet->priv->delay;
}

unsigned int
lm_applet_get_timeout (LMApplet *applet)
{
  g_return_val_if_fail(LM_IS_APPLET(applet), 0);

  return applet->priv->timeout;
}

unsigned int
lm_applet_get_scale (LMApplet *applet)
{
  g_return_val_if_fail(LM_IS_APPLET(applet), 0);

  return applet->priv->scale;
}

int
lm_applet_get_preferences_width (LMApplet *applet)
{
  g_return_val_if_fail(LM_IS_APPLET(applet), 0);

  return applet->priv->preferences_width;
}

int
lm_applet_get_preferences_height (LMApplet *applet)
{
  g_return_val_if_fail(LM_IS_APPLET(applet), 0);

  return applet->priv->preferences_height;
}

static void
lm_applet_activate (LMApplet *applet)
{
  lm_preferences_display(applet);
}

static void
lm_applet_activate_preferences (LMApplet *applet)
{
  lm_preferences_display(applet);
}

static gboolean
lm_applet_factory_cb (PanelApplet *applet,
		      const char *iid,
		      gpointer data)
{
  if (! strcmp(iid, "OAFIID:GNOME_LinkMonitorApplet"))
    {
      LMApplet *rapplet = LM_APPLET(applet);
      GSList *gconf_hosts;
      GSList *l;

      panel_applet_add_preferences(applet, "/schemas/apps/link-monitor-applet/prefs", NULL);

      rapplet->priv->tooltips = gtk_tooltips_new();
      g_object_ref(rapplet->priv->tooltips);
      gtk_object_sink(GTK_OBJECT(rapplet->priv->tooltips));

      panel_applet_setup_menu_from_file(applet, DATADIR, "GNOME_LinkMonitorApplet.xml", NULL, menu_verbs, applet);

      LM_LIST_FOREACH(l, lm_sockets_get())
        {
	  LMSocket *s = l->data;

	  if (! s->error)
	    {
	      WatchInfo *info;

	      info = g_new(WatchInfo, 1);
	      info->channel = g_io_channel_unix_new(s->s);
	      info->s = s;
	      info->applet = rapplet;

	      info->source_id = g_io_add_watch_full(info->channel, G_PRIORITY_HIGH, G_IO_IN, lm_applet_socket_read_cb, info, NULL);
	      rapplet->priv->watches = g_slist_append(rapplet->priv->watches, info);
	    }
	}

      rapplet->priv->delay = panel_applet_gconf_get_int(PANEL_APPLET(applet), CONF_DELAY, NULL);
      if (rapplet->priv->delay < LM_HOST_MIN_DELAY)
	rapplet->priv->delay = LM_HOST_MIN_DELAY;
      
      rapplet->priv->timeout = panel_applet_gconf_get_int(PANEL_APPLET(applet), CONF_TIMEOUT, NULL);
      if (rapplet->priv->timeout < LM_HOST_MIN_TIMEOUT)
	rapplet->priv->timeout = LM_HOST_MIN_TIMEOUT;

      rapplet->priv->scale = panel_applet_gconf_get_int(PANEL_APPLET(applet), CONF_SCALE, NULL);
      if (rapplet->priv->scale < MIN_SCALE)
	rapplet->priv->scale = MIN_SCALE;

      rapplet->priv->preferences_width = panel_applet_gconf_get_int(PANEL_APPLET(applet), CONF_PREFERENCES_WIDTH, NULL);
      if (! rapplet->priv->preferences_width)
	rapplet->priv->preferences_width = DEFAULT_PREFERENCES_WIDTH;

      rapplet->priv->preferences_height = panel_applet_gconf_get_int(PANEL_APPLET(applet), CONF_PREFERENCES_HEIGHT, NULL);
      if (! rapplet->priv->preferences_height)
	rapplet->priv->preferences_height = DEFAULT_PREFERENCES_HEIGHT;

      gconf_hosts = panel_applet_gconf_get_list(PANEL_APPLET(applet), CONF_HOSTS, GCONF_VALUE_STRING, NULL);
      LM_LIST_FOREACH(l, gconf_hosts)
	lm_applet_add_host_internal(rapplet, l->data);
      lm_slist_free(gconf_hosts);

      lm_applet_configure(rapplet);
      lm_applet_update_tooltip(rapplet);

      g_signal_connect(G_OBJECT(applet), "button-press-event", G_CALLBACK(lm_applet_button_press_event_h), NULL);
      g_signal_connect(G_OBJECT(applet), "change-orient", G_CALLBACK(lm_applet_reconfigure), NULL);
      g_signal_connect(G_OBJECT(applet), "change-size", G_CALLBACK(lm_applet_reconfigure), NULL);

      gtk_widget_show(GTK_WIDGET(applet));

      return TRUE;
    }

  return FALSE;
}

static gboolean
lm_applet_socket_read_cb (GIOChannel *source,
			  GIOCondition condition,
			  gpointer data)
{
  WatchInfo *info = data;
  LMICMPReply reply;

  if (lm_icmp_receive(info->s, &reply))
    {
      GSList *l;

      LM_LIST_FOREACH(l, info->applet->priv->hosts)
        {
	  LMHost *host = l->data;

	  if (lm_host_get_seq(host) == reply.seq)
	    {
	      if (reply.echo_reply)
		lm_host_set_roundtrip_time(host, reply.roundtrip_time);
	      else
		lm_host_set_error(host, reply.description);
	    }
	}
    }

  return TRUE;			/* keep source */
}

static void
lm_applet_reconfigure (LMApplet *applet)
{
  g_return_if_fail(LM_IS_APPLET(applet));

  if (applet->priv->reconfigure_timeout_id)
    /* cancel the previously queued reconfigure */
    g_source_remove(applet->priv->reconfigure_timeout_id);

  /* do not flood the panel with reconfigures, queue it */
  applet->priv->reconfigure_timeout_id = g_timeout_add(10, lm_applet_reconfigure_timeout_cb, applet);
}

static gboolean
lm_applet_reconfigure_timeout_cb (gpointer data)
{
  LMApplet *applet = data;
  GSList *l;

  GDK_THREADS_ENTER();

  lm_applet_configure(applet);
  LM_LIST_FOREACH(l, applet->priv->hosts)
    {
      lm_applet_configure_progress_bar(applet, l->data);
      lm_applet_update_progress_bar(applet, l->data);
    }

  GDK_THREADS_LEAVE();

  return FALSE;			/* remove source */
}

static gboolean
lm_applet_button_press_event_h (GtkWidget *widget,
				GdkEventButton *event,
				gpointer user_data)
{
  if (event->button == 1 && event->type == GDK_2BUTTON_PRESS)
    {
      g_signal_emit(widget, applet_signals[ACTIVATE], 0);
      return TRUE;		/* do not propagate event */
    }

  return FALSE;			/* propagate event */
}

static void
lm_applet_preferences_cb (BonoboUIComponent *component,
			  gpointer user_data,
			  const char *cname)
{
  LMApplet *applet = user_data;

  g_signal_emit(applet, applet_signals[ACTIVATE_PREFERENCES], 0);
}

static void
lm_applet_help_cb (BonoboUIComponent *component,
		   gpointer user_data,
		   const char *cname)
{
  lm_display_help(NULL);
}

static void
lm_applet_about_cb (BonoboUIComponent *component,
		    gpointer user_data,
		    const char *cname)
{
  LMApplet *applet = user_data;
  static const char *authors[] = { "Jean-Yves Lefort <jylefort@brutele.be>", NULL };
  static const char *documenters[] = { "Jean-Yves Lefort <jylefort@brutele.be>", NULL };
  GdkPixbuf *logo;
  GdkPixbuf *icon;

  if (applet->priv->about)
    {
      gtk_window_present(GTK_WINDOW(applet->priv->about));
      return;
    }

  logo = lm_pixbuf_new("about-logo.png");
  applet->priv->about = gnome_about_new(_("Link Monitor"),
					VERSION,
					"Copyright \302\251 2004 Jean-Yves Lefort",
					_("The Link Monitor displays the round-trip time to one or more hosts."),
					authors,
					documenters,
					_("Jean-Yves Lefort <jylefort@brutele.be>"),
					logo);
  g_object_unref(logo);

  icon = lm_pixbuf_new("link-monitor-applet.png");
  if (icon)
    {
      gtk_window_set_icon(GTK_WINDOW(applet->priv->about), icon);
      g_object_unref(icon);
    }
  
  g_object_add_weak_pointer(G_OBJECT(applet->priv->about), (gpointer *) &applet->priv->about);
  gtk_widget_show(applet->priv->about);
}

static void
lm_applet_configure (LMApplet *applet)
{
  GSList *l;
  GtkWidget *child;

  g_return_if_fail(LM_IS_APPLET(applet));

  /* ref the progress bars to keep them alive when we'll remove the box */
  LM_LIST_FOREACH(l, applet->priv->hosts)
    {
      LMHost *host = l->data;
      GtkWidget *progress_bar;

      progress_bar = g_object_get_data(G_OBJECT(host), KEY_PROGRESS_BAR);
      g_return_if_fail(GTK_IS_PROGRESS_BAR(progress_bar));

      g_object_ref(progress_bar);
      gtk_object_sink(GTK_OBJECT(progress_bar));
    }
  
  child = gtk_bin_get_child(GTK_BIN(applet));
  if (child)
    gtk_container_remove(GTK_CONTAINER(applet), child);

  if (applet->priv->hosts)
    {
      PanelAppletOrient orient;

      orient = panel_applet_get_orient(PANEL_APPLET(applet));

      switch (orient)
	{
	case PANEL_APPLET_ORIENT_UP:
	case PANEL_APPLET_ORIENT_DOWN:
	  child = gtk_hbox_new(FALSE, BAR_SPACING);
	  break;
      
	case PANEL_APPLET_ORIENT_LEFT:
	case PANEL_APPLET_ORIENT_RIGHT:
	  child = gtk_vbox_new(FALSE, BAR_SPACING);
	  break;
      
	default:
	  g_return_if_reached();
	}

      LM_LIST_FOREACH(l, applet->priv->hosts)
        {
	  LMHost *host = l->data;
	  GtkWidget *progress_bar;

	  progress_bar = g_object_get_data(G_OBJECT(host), KEY_PROGRESS_BAR);
	  g_return_if_fail(GTK_IS_PROGRESS_BAR(progress_bar));

	  gtk_box_pack_start(GTK_BOX(child), progress_bar, FALSE, FALSE, 0);
	  gtk_widget_show(progress_bar);

	  g_object_unref(progress_bar);
	}
    }
  else
    {
      GdkPixbuf *pixbuf;

      pixbuf = lm_pixbuf_new("link-monitor-applet.png");
      if (pixbuf)
	{
	  int size;
	  GdkPixbuf *scaled;

	  size = panel_applet_get_size(PANEL_APPLET(applet)) - (ICON_PADDING * 2);
	  scaled = gdk_pixbuf_scale_simple(pixbuf, size, size, GDK_INTERP_BILINEAR);
	  g_object_unref(pixbuf);
	  
	  child = gtk_image_new_from_pixbuf(scaled);
	  g_object_unref(scaled);
	}
      else			/* use label as fallback */
	{
	  child = gtk_label_new(_("<span size=\"small\">Link Monitor</span>"));
	  gtk_label_set_use_markup(GTK_LABEL(child), TRUE);
	}
    }

  gtk_container_add(GTK_CONTAINER(applet), child);
  gtk_widget_show(child);
}

int
lm_applet_factory_main (void)
{
  return panel_applet_factory_main("OAFIID:GNOME_LinkMonitorApplet_Factory",
				   LM_TYPE_APPLET,
				   lm_applet_factory_cb,
				   NULL);
}

static void
lm_applet_sync_gconf_hosts (LMApplet *applet)
{
  GSList *gconf_hosts = NULL;
  GSList *l;

  g_return_if_fail(LM_IS_APPLET(applet));

  LM_LIST_FOREACH(l, applet->priv->hosts)
    gconf_hosts = g_slist_append(gconf_hosts, g_strdup(lm_host_get_name(l->data)));

  panel_applet_gconf_set_list(PANEL_APPLET(applet),
			      CONF_HOSTS,
			      GCONF_VALUE_STRING,
			      gconf_hosts,
			      NULL);
  lm_slist_free(gconf_hosts);
}

static unsigned int
lm_applet_get_unique_seq (LMApplet *applet)
{
  unsigned int seq = 0;
  GSList *l;

 loop:
  LM_LIST_FOREACH(l, applet->priv->hosts)
    if (lm_host_get_seq(l->data) == seq)
      {
	seq++;
	goto loop;
      }

  return seq;
}

static LMHost *
lm_applet_add_host_internal (LMApplet *applet, const char *name)
{
  LMHost *host;
  GtkWidget *progress_bar;
  GtkWidget *child;

  g_return_val_if_fail(LM_IS_APPLET(applet), NULL);
  g_return_val_if_fail(name != NULL, NULL);

  host = lm_host_new(name, lm_applet_get_unique_seq(applet), applet->priv->delay, applet->priv->timeout);

  progress_bar = gtk_progress_bar_new();
  g_object_set_data(G_OBJECT(host), KEY_PROGRESS_BAR, progress_bar);

  applet->priv->hosts = g_slist_append(applet->priv->hosts, host);
  g_signal_connect(G_OBJECT(host), "notify", G_CALLBACK(lm_applet_host_notify_h), applet);

  lm_applet_configure_progress_bar(applet, host);
  lm_applet_update_progress_bar(applet, host);
  lm_applet_update_tooltip(applet);

  child = gtk_bin_get_child(GTK_BIN(applet));
  if (child && GTK_IS_BOX(child))
    {
      gtk_box_pack_start(GTK_BOX(child), progress_bar, FALSE, FALSE, 0);
      gtk_widget_show(progress_bar);
    }
  else
    lm_applet_configure(applet);

  return host;
}

static LMHost *
lm_applet_replace_host_internal (LMApplet *applet,
				 LMHost *old,
				 const char *new)
{
  GSList *elem;
  unsigned int seq;
  GtkWidget *progress_bar;
  LMHost *new_host;

  g_return_val_if_fail(LM_IS_APPLET(applet), NULL);
  g_return_val_if_fail(LM_IS_HOST(old), NULL);
  g_return_val_if_fail(new != NULL, NULL);

  elem = g_slist_find(applet->priv->hosts, old);
  g_return_val_if_fail(elem != NULL, NULL);

  seq = lm_applet_get_unique_seq(applet);

  progress_bar = g_object_get_data(G_OBJECT(old), KEY_PROGRESS_BAR);
  g_return_val_if_fail(GTK_IS_PROGRESS_BAR(progress_bar), NULL);

  g_object_unref(old);

  new_host = lm_host_new(new, seq, applet->priv->delay, applet->priv->timeout);
  g_object_set_data(G_OBJECT(new_host), KEY_PROGRESS_BAR, progress_bar);

  elem->data = new_host;
  g_signal_connect(G_OBJECT(new_host), "notify", G_CALLBACK(lm_applet_host_notify_h), applet);

  lm_applet_update_progress_bar(applet, new_host);
  lm_applet_update_tooltip(applet);

  return new_host;
}

static void
lm_applet_swap_hosts_internal (LMApplet *applet, LMHost *host1, LMHost *host2)
{
  GSList *elem1 = NULL;
  GSList *elem2 = NULL;
  LMHost *tmp;
  GSList *l;
      
  g_return_if_fail(LM_IS_APPLET(applet));
  g_return_if_fail(LM_IS_HOST(host1));
  g_return_if_fail(LM_IS_HOST(host2));

  elem1 = g_slist_find(applet->priv->hosts, host1);
  elem2 = g_slist_find(applet->priv->hosts, host2);

  g_return_if_fail(elem1 != NULL && elem2 != NULL && elem1 != elem2);

  tmp = elem1->data;
  elem1->data = elem2->data;
  elem2->data = tmp;

  lm_applet_update_tooltip(applet);
  lm_applet_configure(applet);

  LM_LIST_FOREACH(l, applet->priv->hosts)
    lm_applet_update_progress_bar(applet, l->data);
}

static void
lm_applet_remove_host_internal (LMApplet *applet, LMHost *host)
{
  GSList *elem;
  GtkWidget *progress_bar;

  g_return_if_fail(LM_IS_APPLET(applet));
  g_return_if_fail(LM_IS_HOST(host));
  
  elem = g_slist_find(applet->priv->hosts, host);
  g_return_if_fail(elem != NULL);

  progress_bar = g_object_get_data(G_OBJECT(host), KEY_PROGRESS_BAR);
  g_return_if_fail(GTK_IS_PROGRESS_BAR(progress_bar));

  applet->priv->hosts = g_slist_delete_link(applet->priv->hosts, elem);
  g_object_unref(host);
  gtk_widget_destroy(progress_bar);

  lm_applet_update_tooltip(applet);
  if (! applet->priv->hosts)
    lm_applet_configure(applet);
}

static void
lm_applet_host_notify_h (GObject *object,
			 GParamSpec *pspec,
			 gpointer user_data)
{
  LMHost *host = LM_HOST(object);
  const char *name = g_param_spec_get_name(pspec);
  LMApplet *applet = user_data;

  if (! strcmp(name, "alive") || ! strcmp(name, "roundtrip-time"))
    lm_applet_update_progress_bar(applet, host);
  else if (! strcmp(name, "status"))
    lm_applet_update_tooltip(applet);
}

static void
lm_applet_configure_progress_bar (LMApplet *applet, LMHost *host)
{
  GtkWidget *progress_bar;
  PanelAppletOrient orient;
  int size;
  
  g_return_if_fail(LM_IS_APPLET(applet));
  g_return_if_fail(LM_IS_HOST(host));

  progress_bar = g_object_get_data(G_OBJECT(host), KEY_PROGRESS_BAR);
  g_return_if_fail(GTK_IS_PROGRESS_BAR(progress_bar));

  orient = panel_applet_get_orient(PANEL_APPLET(applet));
  size = panel_applet_get_size(PANEL_APPLET(applet)) - (BAR_PADDING * 2);

  switch (orient)
    {
    case PANEL_APPLET_ORIENT_UP:
    case PANEL_APPLET_ORIENT_DOWN:
      gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(progress_bar), GTK_PROGRESS_BOTTOM_TO_TOP);
      gtk_widget_set_size_request(progress_bar, BAR_THICKNESS, size);
      break;
	  
    case PANEL_APPLET_ORIENT_LEFT:
    case PANEL_APPLET_ORIENT_RIGHT:
      gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(progress_bar), GTK_PROGRESS_LEFT_TO_RIGHT);
      gtk_widget_set_size_request(progress_bar, size, BAR_THICKNESS);
      break;
      
    default:
      g_return_if_reached();
    }
}

static void
lm_applet_update_progress_bar (LMApplet *applet, LMHost *host)
{
  GtkWidget *progress_bar;

  g_return_if_fail(LM_IS_APPLET(applet));
  g_return_if_fail(LM_IS_HOST(host));

  progress_bar = g_object_get_data(G_OBJECT(host), KEY_PROGRESS_BAR);
  g_return_if_fail(GTK_IS_PROGRESS_BAR(progress_bar));

  if (lm_host_get_alive(host))
    {
      double fraction;

      fraction = lm_host_get_roundtrip_time(host) / applet->priv->scale;
      gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress_bar), CLAMP(fraction, 0.0, 1.0));
      gtk_widget_modify_bg(progress_bar, GTK_STATE_NORMAL, NULL);
    }
  else
    {
      const GdkColor color = { 0, 0xFFFF, 0x0000, 0x0000 }; /* red */
      
      gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress_bar), 0);
      gtk_widget_modify_bg(progress_bar, GTK_STATE_NORMAL, &color);
    }
}

static void
lm_applet_update_tooltip (LMApplet *applet)
{
  GString *tip;
  GSList *l;

  g_return_if_fail(LM_IS_APPLET(applet));

  tip = g_string_new(NULL);
  LM_LIST_FOREACH(l, applet->priv->hosts)
    {
      LMHost *host = l->data;
      const char *name;
      const char *ip;
      const char *status;

      if (*tip->str)
	g_string_append_c(tip, '\n');

      name = lm_host_get_name(host);
      ip = lm_host_get_ip(host);
      status = lm_host_get_status(host);

      if (ip && strcmp(ip, name))
	g_string_append_printf(tip, "\342\200\242 %s (%s): %s", name, ip, status);
      else
	g_string_append_printf(tip, "\342\200\242 %s: %s", name, status);
    }

  if (! *tip->str)
    g_string_assign(tip, _("No host is being monitored."));
  
  gtk_tooltips_set_tip(applet->priv->tooltips, GTK_WIDGET(applet), tip->str, NULL);
  g_string_free(tip, TRUE);
}

LMHost *
lm_applet_add_host (LMApplet *applet, const char *name)
{
  LMHost *host;

  g_return_val_if_fail(LM_IS_APPLET(applet), NULL);
  g_return_val_if_fail(name != NULL, NULL);

  host = lm_applet_add_host_internal(applet, name);
  lm_applet_sync_gconf_hosts(applet);

  return host;
}

LMHost *
lm_applet_replace_host (LMApplet *applet, LMHost *old, const char *new)
{
  LMHost *new_host;

  g_return_val_if_fail(LM_IS_APPLET(applet), NULL);
  g_return_val_if_fail(LM_IS_HOST(old), NULL);
  g_return_val_if_fail(new != NULL, NULL);

  new_host = lm_applet_replace_host_internal(applet, old, new);
  lm_applet_sync_gconf_hosts(applet);

  return new_host;
}

void
lm_applet_swap_hosts (LMApplet *applet, LMHost *host1, LMHost *host2)
{
  g_return_if_fail(LM_IS_APPLET(applet));
  g_return_if_fail(LM_IS_HOST(host1));
  g_return_if_fail(LM_IS_HOST(host2));

  lm_applet_swap_hosts_internal(applet, host1, host2);
  lm_applet_sync_gconf_hosts(applet);
}

void
lm_applet_remove_host (LMApplet *applet, LMHost *host)
{
  g_return_if_fail(LM_IS_APPLET(applet));
  g_return_if_fail(LM_IS_HOST(host));

  lm_applet_remove_host_internal(applet, host);
  lm_applet_sync_gconf_hosts(applet);
}
