/*
 * $Id: lm-preferences.c,v 1.9 2004/07/22 17:26:33 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <gtk/gtk.h>
#include "lm-applet.h"
#include "lm-util.h"

/*** cpp *********************************************************************/

/* glib-i18n.h does not provide a stub for ngettext */
#ifndef ENABLE_NLS
#ifndef ngettext		/* but future versions might define it */
#define ngettext(Msgid1, Msgid2, N) \
  ((N) == 1 ? (const char *) (Msgid1) : (const char *) (Msgid2))
#endif /* ngettext */
#endif /* ENABLE_NLS */

/*** types *******************************************************************/

enum {
  COLUMN_HOST,
  COLUMN_NAME,
  N_COLUMNS
};
    
typedef struct
{
  GtkWidget	*dialog;
  GtkWidget	*list;
  GtkWidget	*selected_label;
  GtkWidget	*remove;
  GtkWidget	*up;
  GtkWidget	*down;
  GtkWidget	*delay_label;
  GtkWidget	*delay_spin;
  GtkWidget	*timeout_label;
  GtkWidget	*timeout_spin;
  GtkWidget	*scale_label;
  GtkWidget	*scale_spin;

  GtkWidget	*host_menu;
  GtkWidget	*remove_item;
  GtkWidget	*up_item;
  GtkWidget	*down_item;

  LMApplet	*applet;
  GtkListStore	*store;
  int		add_pending_count;
  GtkTreeIter	add_pending_iter;
} PreferencesDialog;

/*** variables ***************************************************************/

static PreferencesDialog preferences = { NULL };

/*** functions ***************************************************************/

static void lm_preferences_add_host (void);
static GSList *lm_preferences_get_selected_rows (void);
static void lm_preferences_move_host (int direction);
static void lm_preferences_remove_host (void);

static void lm_preferences_selection_changed_h (GtkTreeSelection *selection,
						gpointer user_data);
static void lm_preferences_host_editing_canceled_h (GtkCellRenderer *renderer,
						    gpointer user_data);
static void lm_preferences_host_edited_h (GtkCellRendererText *renderer,
					  const char *path_string,
					  const char *new_text,
					  gpointer data);

static void lm_preferences_update_selected_label (void);
static void lm_preferences_update_sensitivity (void);

/*** implementation **********************************************************/

void
lm_preferences_display (LMApplet *applet)
{
  GtkSizeGroup *size_group;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkTreeSelection *selection;
  GSList *l;

  if (preferences.dialog)
    {
      gtk_window_present(GTK_WINDOW(preferences.dialog));
      return;
    }

  preferences.applet = applet;
  preferences.add_pending_count = 0;

  lm_create_interface("preferences",
		       "dialog", &preferences.dialog,
		       "list", &preferences.list,
		       "selected_label", &preferences.selected_label,
		       "remove", &preferences.remove,
		       "up", &preferences.up,
		       "down", &preferences.down,
		       "delay_label", &preferences.delay_label,
		       "delay_spin", &preferences.delay_spin,
		       "timeout_label", &preferences.timeout_label,
		       "timeout_spin", &preferences.timeout_spin,
		       "scale_label", &preferences.scale_label,
		       "scale_spin", &preferences.scale_spin,
		       "host_menu", &preferences.host_menu,
		       "remove_item", &preferences.remove_item,
		       "up_item", &preferences.up_item,
		       "down_item", &preferences.down_item,
		       NULL);

  /* create the GtkSizeGroup */

  size_group = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
  gtk_size_group_add_widget(size_group, preferences.delay_label);
  gtk_size_group_add_widget(size_group, preferences.timeout_label);
  gtk_size_group_add_widget(size_group, preferences.scale_label);
  g_object_unref(size_group);

  /* finish the hosts list */

  preferences.store = gtk_list_store_new(N_COLUMNS, LM_TYPE_HOST, G_TYPE_STRING);
  gtk_tree_view_set_model(GTK_TREE_VIEW(preferences.list), GTK_TREE_MODEL(preferences.store));

  renderer = gtk_cell_renderer_text_new();
  g_object_set(G_OBJECT(renderer), "editable", TRUE, NULL);
  g_signal_connect(G_OBJECT(renderer), "editing-canceled", G_CALLBACK(lm_preferences_host_editing_canceled_h), NULL);
  g_signal_connect(G_OBJECT(renderer), "edited", G_CALLBACK(lm_preferences_host_edited_h), NULL);

  column = gtk_tree_view_column_new_with_attributes(_("Host"),
						    renderer,
						    "text", COLUMN_NAME,
						    NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(preferences.list), column);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.list));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
  g_signal_connect(G_OBJECT(selection), "changed", G_CALLBACK(lm_preferences_selection_changed_h), NULL);

  /* and fill it */

  LM_LIST_FOREACH(l, lm_applet_get_hosts(preferences.applet))
    {
      LMHost *host = l->data;
      GtkTreeIter iter;

      gtk_list_store_append(preferences.store, &iter);
      gtk_list_store_set(preferences.store, &iter,
			 COLUMN_HOST, host,
			 COLUMN_NAME, lm_host_get_name(host),
			 -1);
    }

  /* set the options */

  gtk_window_set_default_size(GTK_WINDOW(preferences.dialog),
			      lm_applet_get_preferences_width(preferences.applet),
			      lm_applet_get_preferences_height(preferences.applet));

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(preferences.delay_spin), lm_applet_get_delay(preferences.applet));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(preferences.timeout_spin), lm_applet_get_timeout(preferences.applet));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(preferences.scale_spin), lm_applet_get_scale(preferences.applet));

  lm_preferences_update_selected_label();
  lm_preferences_update_sensitivity();

  gtk_widget_show(preferences.dialog);
}

static void
lm_preferences_add_host (void)
{
  GtkTreePath *path;
  GtkTreeViewColumn *column;

  gtk_list_store_append(preferences.store, &preferences.add_pending_iter);
  preferences.add_pending_count++;

  path = gtk_tree_model_get_path(GTK_TREE_MODEL(preferences.store), &preferences.add_pending_iter);
  column = gtk_tree_view_get_column(GTK_TREE_VIEW(preferences.list), 0);

  gtk_widget_grab_focus(preferences.list);
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(preferences.list), path, column, FALSE, 0, 0);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(preferences.list), path, column, TRUE);
  gtk_tree_path_free(path);
}

static GSList *
lm_preferences_get_selected_rows (void)
{
  GtkTreeSelection *selection;
  GList *paths;
  GList *l;
  GSList *references = NULL;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.list));
  paths = gtk_tree_selection_get_selected_rows(selection, NULL);

  LM_LIST_FOREACH(l, paths)
    {
      GtkTreePath *path = l->data;

      references = g_slist_append(references, gtk_tree_row_reference_new(GTK_TREE_MODEL(preferences.store), path));
      gtk_tree_path_free(path);
    }
  g_list_free(paths);

  return references;
}

static void
lm_preferences_move_host (int direction)
{
  GSList *references;
  GSList *l;

  references = lm_preferences_get_selected_rows();
  if (direction == +1)
    references = g_slist_reverse(references);

  LM_LIST_FOREACH(l, references)
    {
      GtkTreeRowReference *reference = l->data;
      GtkTreePath *path;
      GtkTreeIter iter;
      GtkTreeIter other_iter;
      gboolean status;

      path = gtk_tree_row_reference_get_path(reference);

      status = gtk_tree_model_get_iter(GTK_TREE_MODEL(preferences.store), &iter, path);
      g_return_if_fail(status == TRUE);

      if (direction == -1)
	gtk_tree_path_prev(path);
      else
	gtk_tree_path_next(path);

      if (gtk_tree_model_get_iter(GTK_TREE_MODEL(preferences.store), &other_iter, path))
	{
	  LMHost *host;
	  LMHost *other_host;

	  gtk_tree_model_get(GTK_TREE_MODEL(preferences.store), &iter, COLUMN_HOST, &host, -1);
	  gtk_tree_model_get(GTK_TREE_MODEL(preferences.store), &other_iter, COLUMN_HOST, &other_host, -1);

	  gtk_list_store_swap(preferences.store, &iter, &other_iter);
	  lm_applet_swap_hosts(preferences.applet, host, other_host);

	  g_object_unref(host);
	  g_object_unref(other_host);
	}
      
      gtk_tree_path_free(path);
      gtk_tree_row_reference_free(reference);
    }
  g_slist_free(references);

  lm_preferences_update_sensitivity();	/* for up and down */
}

static void
lm_preferences_remove_host (void)
{
  GSList *references;
  GSList *l;

  references = lm_preferences_get_selected_rows();
  LM_LIST_FOREACH(l, references)
    {
      GtkTreeRowReference *reference = l->data;
      GtkTreePath *path;
      GtkTreeIter iter;
      gboolean status;
      LMHost *host;

      path = gtk_tree_row_reference_get_path(reference);
      status = gtk_tree_model_get_iter(GTK_TREE_MODEL(preferences.store), &iter, path);
      g_return_if_fail(status == TRUE);
      gtk_tree_path_free(path);

      gtk_tree_model_get(GTK_TREE_MODEL(preferences.store), &iter, COLUMN_HOST, &host, -1);

      lm_applet_remove_host(preferences.applet, host);
      gtk_list_store_remove(preferences.store, &iter);

      g_object_unref(host);
      gtk_tree_row_reference_free(reference);
    }
  g_slist_free(references);
}

static void
lm_preferences_selection_changed_h (GtkTreeSelection *selection,
				    gpointer user_data)
{
  lm_preferences_update_selected_label();
  lm_preferences_update_sensitivity();
}

static void
lm_preferences_host_editing_canceled_h (GtkCellRenderer *renderer,
					gpointer user_data)
{
  if (preferences.add_pending_count)
    {
      preferences.add_pending_count--;
      gtk_list_store_remove(preferences.store, &preferences.add_pending_iter);
    }
}

static void
lm_preferences_host_edited_h (GtkCellRendererText *renderer,
			      const char *path_string,
			      const char *new_text,
			      gpointer data)
{
  GtkTreePath *path;
  GtkTreeIter iter;
  LMHost *old_host;
  LMHost *host;

  path = gtk_tree_path_new_from_string(path_string);
  gtk_tree_model_get_iter(GTK_TREE_MODEL(preferences.store), &iter, path);
  gtk_tree_path_free(path);

  gtk_tree_model_get(GTK_TREE_MODEL(preferences.store), &iter, COLUMN_HOST, &old_host, -1);

  if (old_host)
    {
      host = lm_applet_replace_host(preferences.applet, old_host, new_text);
      g_object_unref(old_host);
    }
  else
    host = lm_applet_add_host(preferences.applet, new_text);

  gtk_list_store_set(preferences.store, &iter,
		     COLUMN_HOST, host,
		     COLUMN_NAME, new_text,
		     -1);

  if (preferences.add_pending_count)
    preferences.add_pending_count--;
}

static void
lm_preferences_update_selected_label (void)
{
  GtkTreeSelection *selection;
  int n_rows;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.list));
  n_rows = gtk_tree_selection_count_selected_rows(selection);

  if (n_rows == 0)
    gtk_label_set_text(GTK_LABEL(preferences.selected_label), _("No host selected."));
  else
    {
      char *str;

      str = g_strdup_printf(ngettext("%i host selected.",
				     "%i hosts selected.",
				     n_rows), n_rows);
      gtk_label_set_text(GTK_LABEL(preferences.selected_label), str);
      g_free(str);
    }
}

static void
lm_preferences_update_sensitivity (void)
{
  GtkTreeSelection *selection;
  GList *paths;
  gboolean has_selection = FALSE;
  gboolean has_prev = FALSE;
  gboolean has_next = FALSE;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.list));
  paths = gtk_tree_selection_get_selected_rows(selection, NULL);

  if (paths)
    {
      GList *l;
      GtkTreeIter iter;
      gboolean status;

      has_selection = TRUE;

      /* has prev? */

      l = paths;
      if (gtk_tree_path_prev(l->data))
	{
	  gtk_tree_path_next(l->data); /* restore it */
	  has_prev = TRUE;
	}

      /* has next? */

      l = g_list_last(paths);
      status = gtk_tree_model_get_iter(GTK_TREE_MODEL(preferences.store), &iter, l->data);
      g_return_if_fail(status == TRUE);
      if (gtk_tree_model_iter_next(GTK_TREE_MODEL(preferences.store), &iter))
	has_next = TRUE;

      /* cleanup */

      LM_LIST_FOREACH(l, paths)
	gtk_tree_path_free(l->data);
      g_list_free(paths);
    }

  gtk_widget_set_sensitive(preferences.remove, has_selection);
  gtk_widget_set_sensitive(preferences.remove_item, has_selection);

  gtk_widget_set_sensitive(preferences.up, has_prev);
  gtk_widget_set_sensitive(preferences.up_item, has_prev);

  gtk_widget_set_sensitive(preferences.down, has_next);
  gtk_widget_set_sensitive(preferences.down_item, has_next);
}

/* libglade callbacks */

gboolean
lm_preferences_list_popup_menu_h (GtkWidget *widget, gpointer user_data)
{
  gtk_menu_popup(GTK_MENU(preferences.host_menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());

  return TRUE;			/* a menu was activated */
}

gboolean
lm_preferences_list_button_press_event_h (GtkWidget *widget,
					  GdkEventButton *event,
					  gpointer user_data)
{
  if (event->button == 3)
    gtk_menu_popup(GTK_MENU(preferences.host_menu), NULL, NULL, NULL, NULL, event->button, event->time);

  return FALSE;			/* propagate event */
}

void
lm_preferences_add_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_preferences_add_host();
}

void
lm_preferences_remove_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_preferences_remove_host();
}

void
lm_preferences_up_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_preferences_move_host(-1);
}

void
lm_preferences_down_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_preferences_move_host(+1);
}

void
lm_preferences_add_item_activate_h (GtkMenuItem *menuitem, gpointer user_data)
{
  lm_preferences_add_host();
}

void
lm_preferences_remove_item_activate_h (GtkMenuItem *menuitem, gpointer user_data)
{
  lm_preferences_remove_host();
}

void
lm_preferences_up_item_activate_h (GtkMenuItem *menuitem, gpointer user_data)
{
  lm_preferences_move_host(-1);
}

void
lm_preferences_down_item_activate_h (GtkMenuItem *menuitem, gpointer user_data)
{
  lm_preferences_move_host(+1);
}

void
lm_preferences_delay_spin_value_changed_h (GtkSpinButton *spinbutton,
					   gpointer user_data)
{
  lm_applet_set_delay(preferences.applet, gtk_spin_button_get_value_as_int(spinbutton));
}

void
lm_preferences_timeout_spin_value_changed_h (GtkSpinButton *spinbutton,
					     gpointer user_data)
{
  lm_applet_set_timeout(preferences.applet, gtk_spin_button_get_value_as_int(spinbutton));
}

void
lm_preferences_scale_spin_value_changed_h (GtkSpinButton *spinbutton,
					   gpointer user_data)
{
  lm_applet_set_scale(preferences.applet, gtk_spin_button_get_value_as_int(spinbutton));
}

gboolean
lm_preferences_configure_event_h (GtkWidget *widget,
				  GdkEventConfigure *event,
				  gpointer user_data)
{
  lm_applet_set_preferences_width(preferences.applet, event->width);
  lm_applet_set_preferences_height(preferences.applet, event->height);

  return FALSE;			/* propagate event */
}

void
lm_preferences_response_h (GtkDialog *dialog,
			   int response,
			   gpointer user_data)
{
  switch (response)
    {
    case GTK_RESPONSE_HELP:
      lm_display_help("preferences");
      break;

    case GTK_RESPONSE_CLOSE:
      gtk_widget_destroy(preferences.dialog);
      break;
    }
}

void
lm_preferences_destroy_h (GtkObject *object, gpointer user_data)
{
  gtk_widget_destroy(preferences.host_menu);
  g_object_unref(preferences.store);
  preferences.dialog = NULL;
}

void
lm_preferences_entry_activate_h (gpointer user_data, GtkEntry *entry)
{
  GtkWidget *next = user_data;

  gtk_widget_grab_focus(next);
}
