/*
 * $Id: lm-sockets.c,v 1.3 2004/07/15 23:22:02 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <errno.h>
#include <unistd.h>
#include <stdio.h>		/* required by stdlib.h on Darwin */
#include <stdlib.h>		/* required by sys/socket.h on Darwin */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <glib.h>
#include <glib/gi18n-lib.h>
#include "lm-sockets.h"
#include "lm-util.h"

/*** variables ***************************************************************/

static GSList *sockets = NULL;

/*** functions ***************************************************************/

static void lm_sockets_add (int domain, int protocol);

/*** implementation **********************************************************/

/*
 * Initialize sockets (step 1, requires root privileges).
 */
void
lm_sockets_init1 (void)
{
  lm_sockets_add(AF_INET, IPPROTO_ICMP);
#ifdef WITH_IPV6
  lm_sockets_add(AF_INET6, IPPROTO_ICMPV6);
#endif
}

/*
 * Initialize sockets (step 2, does not require root privileges).
 */
void
lm_sockets_init2 (void)
{
  GSList *l;

  LM_LIST_FOREACH(l, sockets)
    {
      LMSocket *s = l->data;

      if (! s->error)
	{
	  int flags;

	  flags = fcntl(s->s, F_GETFL, 0);
	  if (fcntl(s->s, F_SETFL, flags | O_NONBLOCK) < 0)
	    s->error = g_strdup_printf(_("unable to set O_NONBLOCK flag: %s"), g_strerror(errno));
	}
    }
}

static void
lm_sockets_add (int domain, int protocol)
{
  LMSocket *s;

  s = g_new0(LMSocket, 1);

  s->domain = domain;
  s->s = socket(domain, SOCK_RAW, protocol);
  if (s->s < 0)
    s->error = g_strdup(g_strerror(errno));

  sockets = g_slist_append(sockets, s);
}

GSList *
lm_sockets_get (void)
{
  return sockets;
}

const LMSocket *
lm_sockets_get_by_domain (int domain)
{
  GSList *l;
  
  LM_LIST_FOREACH(l, sockets)
    {
      LMSocket *s = l->data;

      if (s->domain == domain)
	return s;
    }
  
  return NULL;
}
