/*
 * $Id: lm-util.c,v 1.15 2004/10/07 18:29:14 jylefort Exp $
 *
 * Copyright (c) 1989, 1993 The Regents of the University of California
 * Copyright (c) 2004 Jean-Yves Lefort
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <stdio.h>		/* required by stdlib.h on Darwin */
#include <stdlib.h>
#include <stdarg.h>
#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif /* TM_IN_SYS_TIME */
#include <errno.h>
#include <gnome.h>
#include <glade/glade.h>
#include <eel/eel.h>
#include <eel/eel-alert-dialog.h>
#include "lm-util.h"

/*** functions ***************************************************************/

static GtkWidget *lm_menu_item_new (const char *stock_id, const char *mnemonic);
static void lm_error_dialog_real (GtkWindow *parent,
				  gboolean blocking,
				  const char *primary,
				  const char *secondary);

/*** implementation **********************************************************/

/**
 * lm_g_object_slist_free:
 * @list: a #GSList of #GObject instances
 *
 * Equivalent of eel_g_object_list_free() for a singly-linked list.
 **/
void
lm_g_object_slist_free (GSList *list)
{
  eel_g_slist_free_deep_custom(list, (GFunc) g_object_unref, NULL);
}

void
lm_create_interface (const char *name, ...)
{
  char *filename;
  char *pathname;
  GladeXML *xml;
  va_list args;
  const char *widget_name;

  g_return_if_fail(name != NULL);

  filename = g_strconcat(name, ".glade", NULL);
  pathname = g_build_filename(UIDIR, filename, NULL);
  g_free(filename);

  xml = glade_xml_new(pathname, NULL, NULL);
  g_free(pathname);

  g_return_if_fail(xml != NULL);
  glade_xml_signal_autoconnect(xml);

  va_start(args, name);
  while ((widget_name = va_arg(args, const char *)))
    {
      GtkWidget **widget;

      widget = va_arg(args, GtkWidget **);
      g_return_if_fail(widget != NULL);

      *widget = glade_xml_get_widget(xml, widget_name);
      if (! *widget)
	g_critical(_("widget %s not found in interface %s"), widget_name, name);
    }
  va_end(args);
  
  g_object_unref(xml);
}

GdkPixbuf *
lm_pixbuf_new (const char *filename)
{
  char *pathname;
  GdkPixbuf *pixbuf;
  GError *err = NULL;

  g_return_val_if_fail(filename != NULL, NULL);

  pathname = g_build_filename(UIDIR, filename, NULL);
  pixbuf = gdk_pixbuf_new_from_file(pathname, &err);
  g_free(pathname);

  if (! pixbuf)
    {
      g_warning(_("error loading image: %s"), err->message);
      g_error_free(err);
    }

  return pixbuf;
}

void
lm_thread_create (GThreadFunc func, gpointer data)
{
  GError *err = NULL;

  g_return_if_fail(func != NULL);
  
  if (! g_thread_create(func, data, FALSE, &err))
    {
      lm_fatal_error_dialog(NULL, _("Unable to create a thread: %s."), err->message);
      g_error_free(err);
    }
}

time_t
lm_time (void)
{
  time_t t;

  t = time(NULL);
  if (t < 0)
    {
      g_warning(_("unable to get current time: %s"), g_strerror(errno));
      t = 0;
    }

  return t;
}

/*
 * Subtract 2 timeval structs: out = out - in. Out is assumed to be >=
 * in (based on tvsub() from FreeBSD's ping.c).
 */
void
lm_tvsub (GTimeVal *out, const GTimeVal *in)
{
  g_return_if_fail(out != NULL);
  g_return_if_fail(in != NULL);

  if ((out->tv_usec -= in->tv_usec) < 0)
    {
      --out->tv_sec;
      out->tv_usec += 1000000;
    }
  out->tv_sec -= in->tv_sec;
}

void
lm_display_help (const char *link_id)
{
  GError *err = NULL;

  if (! gnome_help_display("link-monitor-applet.xml", link_id, &err))
    {
      lm_error_dialog(NULL, _("Unable to display help"), "%s", err->message);
      g_error_free(err);
    }
}

/**
 * lm_menu_shell_append:
 * @shell: the #GtkMenuShell to append to
 * @stock_id: the stock ID of the item, or NULL
 * @mnemonic: the mnemonic of the item, or NULL
 *
 * Creates a new menu item, shows it and appends it to @shell.
 *
 * If both @stock_id and @mnemonic are provided, a #GtkImageMenuItem
 * will be created using the text of @mnemonic and the icon of
 * @stock_id.
 *
 * If only @stock_id is provided, a #GtkImageMenuitem will be created
 * using the text and icon of @stock_id.
 *
 * If only @mnemonic is provided, a #GtkMenuItem will be created using
 * the text of @mnemonic.
 *
 * If @stock_id and @mnemonic are both NULL, a #GtkSeparatorMenuItem
 * will be created.
 *
 * Return value: the new menu item
 **/
GtkWidget *
lm_menu_shell_append (GtkMenuShell *shell,
		      const char *stock_id,
		      const char *mnemonic)
{
  GtkWidget *item;

  g_return_val_if_fail(GTK_IS_MENU_SHELL(shell), NULL);

  item = lm_menu_item_new(stock_id, mnemonic);
  gtk_menu_shell_append(shell, item);
  gtk_widget_show(item);
  
  return item;
}

static GtkWidget *
lm_menu_item_new (const char *stock_id, const char *mnemonic)
{
  GtkWidget *item;

  if (stock_id && mnemonic)
    {
      GtkWidget *image;

      item = gtk_image_menu_item_new_with_mnemonic(mnemonic);
      
      image = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_MENU);
      gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), image);
      gtk_widget_show(image);
    }
  else if (stock_id)
    item = gtk_image_menu_item_new_from_stock(stock_id, NULL);
  else if (mnemonic)
    item = gtk_menu_item_new_with_mnemonic(mnemonic);
  else
    item = gtk_separator_menu_item_new();
  
  return item;
}

static void
lm_error_dialog_real (GtkWindow *parent,
		      gboolean blocking,
		      const char *primary,
		      const char *secondary)
{
  GtkWidget *dialog;

  dialog = eel_alert_dialog_new(parent,
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				primary,
				secondary,
				NULL);

  if (blocking)
    {
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    }
  else
    {
      g_signal_connect(dialog, "response", G_CALLBACK(gtk_widget_destroy), NULL);
      gtk_widget_show(dialog);
    }
}

void
lm_error_dialog (GtkWindow *parent,
		 const char *primary,
		 const char *format,
		 ...)
{
  char *secondary;

  if (format)
    {
      va_list args;
      
      va_start(args, format);
      secondary = g_strdup_vprintf(format, args);
      va_end(args);
    }
  else
    secondary = NULL;

  lm_error_dialog_real(parent, FALSE, primary, secondary);
  g_free(secondary);
}

void
lm_fatal_error_dialog (GtkWindow *parent, const char *format, ...)
{
  va_list args;
  char *secondary;

  g_return_if_fail(format != NULL);

  va_start(args, format);
  secondary = g_strdup_vprintf(format, args);
  va_end(args);

  lm_error_dialog_real(parent, TRUE, _("A fatal error has occurred in Link Monitor"), secondary);
  g_free(secondary);

  exit(1);
}
