/********************************************************************/
/*                                                                  */
/*            L   I  QQ  U U I DD    W   W  A  RR    555            */
/*            L   I Q  Q U U I D D   W   W A A R R   5              */
/*            L   I Q  Q U U I D D   W W W AAA RR    55             */
/*            L   I Q Q  U U I D D   WW WW A A R R     5            */
/*            LLL I  Q Q  U  I DD    W   W A A R R   55             */
/*                                                                  */
/*                             b                                    */
/*                             bb  y y                              */
/*                             b b yyy                              */
/*                             bb    y                              */
/*                                 yy                               */
/*                                                                  */
/*                     U U       FFF  O   O  TTT                    */
/*                     U U       F   O O O O  T                     */
/*                     U U TIRET FF  O O O O  T                     */
/*                     U U       F   O O O O  T                     */
/*                      U        F    O   O   T                     */
/*                                                                  */
/********************************************************************/

/********************************************************************/
/* this software is protected by the GPL, see copying.txt           */
/********************************************************************/

/********************************************************************/
/* name          : srvcont.c                                        */
/* content       : global controler of the network channels         */
/* date de modif : 3 mai 98                                         */
/********************************************************************/

/*==================================================================*/
/* includes                                                         */
/*==================================================================*/

#include <stdlib.h>
#include <sys/types.h>

#include "srvcont.h"
#include "srvchan.h"
#include "sockgen.h"
#include "log.h"
#include "server.h"
#include "netkey.h"
#include "srvtime.h"

/*==================================================================*/
/* constants                                                        */
/*==================================================================*/

#define LW_SRVCONT_INCREASE_LAG_LIMIT 10
#define LW_SRVCONT_DECREASE_LAG_LIMIT 20

#define LW_SRVCONT_DISPLAY_STATE_DELAY 60

/*==================================================================*/
/* static functions                                                 */
/*==================================================================*/

static void reset(LW_SRVCONT *cont);
static int distribute_teams(LW_SRVCONT *cont);
static void find_chan_and_team_by_server_id(LW_SRVCONT *cont,
					    int server_id,
					    int *chan,
					    int *team);
static int send_lag_key_presses(LW_SRVCONT *cont,int lag);

/*------------------------------------------------------------------*/
/*
 * Resets the LW_SRVCONT struct
 */
static void reset(LW_SRVCONT *cont)
{
  cont->sock=-1;
  cont->nb_teams=0;
  cont->nb_chans=0;
  cont->map=NULL;
  memset(&(cont->config),0,sizeof(LW_NETCONF));
}

/*------------------------------------------------------------------*/
/*
 * Affects consistent values to all the teams
 */
static int distribute_teams(LW_SRVCONT *cont)
{
  int result=1;
  int i,j,k;
  int i_orig,j_orig;

  /*
   * The first step is to attribute a unique number to each
   * local team for each channel.
   */
  k=0;
  for (i=0;i<cont->nb_chans;++i)
    {
      for (j=0;j<NB_TEAMS;++j)
	{
	  if (cont->chan[i].team[j].active)
	    {
	      cont->chan[i].team[j].server_id=k++;
	    }
	}
    }

  /*
   * Now k should be equal to the number of available teams,
   * or it means there's a serious error...
   */
  if (k!=cont->nb_teams)
    {
      log_println_str("Unable to attribute teams!");
      result=0;
    }

  /*
   * The second step is to define which teams in each channel
   * will be the "network" teams.
   */
  for (i=0;i<cont->nb_chans;++i)
    {
      for (k=0;k<cont->nb_teams;++k)
	{
	  if (lw_srvchan_find_team_by_server_id(&(cont->chan[i]),k)<0)
	    {
	      j=lw_srvchan_find_first_free_team(&(cont->chan[i]));
	      if (j>=0)
		{
		  cont->chan[i].team[j].active=1;
		  cont->chan[i].team[j].network=1;
		  cont->chan[i].team[j].server_id=k;
		}
	      else
		{
		  result=0;
		  log_println_str("Unable to find a free team!");
		}
	    }
	}
    }

  /*
   * Last step, we copy the team names into all the "nickname" fields.
   * This will save useless CPU waste later...
   */
  for (i=0;i<cont->nb_chans;++i)
    {
      for (j=0;j<NB_TEAMS;++j)
	{
	  if (cont->chan[i].team[j].active)
	    {
	      find_chan_and_team_by_server_id(cont,
					      cont->chan[i].team[j].server_id,
					      &i_orig,
					      &j_orig);
	      if (i_orig>=0 && j_orig>=0)
		{
		  strcpy(cont->chan[i].team[j].nickname,
			 cont->chan[i_orig].nickname);
		}
	      else
		{
		  result=0;
		  log_println_str("Unable to find a free team!");
		}
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Gets the chan and team index for a given server id
 */
static void find_chan_and_team_by_server_id(LW_SRVCONT *cont,
					    int server_id,
					    int *chan,
					    int *team)
{
  int i;
  int j;

  *chan=*team=-1;
  for (i=0;i<cont->nb_chans;++i)
    {
      for (j=0;j<NB_TEAMS;++j)
	{
	  if (cont->chan[i].team[j].active && 
	      (cont->chan[i].team[j].server_id==server_id) &&
	      !(cont->chan[i].team[j].network))
	    {
	      (*chan)=i;
	      (*team)=j;
	    }
	}
    }
}

/*==================================================================*/
/* global functions                                                 */
/*==================================================================*/

/*------------------------------------------------------------------*/
/* 
 * Waits for a given number of teams to connect
 */
int lw_srvcont_wait_teams(LW_SRVCONT *cont, int nb_teams, int port)
{
  int free_teams;
  int result=0;

  reset(cont);

  if (lw_sock_listen(&(cont->sock),
		     port))
    {
      log_print_str("Listening on port ");
      log_print_int(port);
      log_println_str("...");
      log_flush();

      /*
       * Now we just wait until there are enough teams connected
       */
      while (cont->nb_teams<nb_teams && cont->nb_chans<NB_TEAMS)
	{
	  free_teams=nb_teams-cont->nb_teams;
	  if (lw_srvchan_wait_teams(&(cont->chan[cont->nb_chans]), 
				    &free_teams,
				    cont->sock,
				    &(cont->map),
				    &(cont->config)))
	    {
	      cont->nb_teams=nb_teams-free_teams;
	      ++(cont->nb_chans);
	    }
	}

      /*
       * For now the server is a "blocking" server which is never statisfied
       * until the good number of players has connected itself, so if we
       * get here it means the operation has been successfull
       */
      result=1;
    }
  else
    {
      log_print_str("Unable to bind socket on port ");
      log_print_int(port);
      log_println("!");
    }

  return result;
}

/*------------------------------------------------------------------*/
/* 
 * Tells the teams about what the other teams are
 */
int lw_srvcont_tell_who(LW_SRVCONT *cont)
{
  int i;
  int result=0;

  /*
   * Now we close the main socket which was accepting new connections.
   * If we did not do this the server would keep on accepting
   * connections, which is baaaad.
   */
  lw_sock_close(cont->sock);

  if (distribute_teams(cont))
    {
      result=1;
      for (i=0;i<cont->nb_chans;++i)
	{
	  if (!lw_srvchan_tell_who(&(cont->chan[i]),
				   cont->map,
				   &(cont->config)))
	    {
	      result=0;
	    }
	}
    }

  return result;
}


/*------------------------------------------------------------------*/
/* 
 * Sends a final OK message to all the clients
 */
int lw_srvcont_final_ok(LW_SRVCONT *cont)
{
  int i;
  int result=1;

  for (i=0;i<cont->nb_chans;++i)
    {
      if (!lw_srvchan_final_ok(&(cont->chan[i])))
	{
	  result=0;
	}
    }

  if (result)
    {
      log_print_str("Game start");
      log_println();
    }
  else
    {
      log_print_str("Unable to start game");
      log_println();
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends a serie of blank key presses to create an artificial lag.
 * This lag is usefull, since this way, a few game cycles will 
 * have to be run before a real key press comes back to a given
 * player. This leaves times for other players to send their own
 * key presses and this way the game can run smoothly.
 */
static int send_lag_key_presses(LW_SRVCONT *cont,int lag)
{
  int result=1;
  int i,j;
  LW_NETKEY netkey;

  lw_netkey_reset(&netkey);
  for (j=0;j<lag && result;++j)
    {
      for (i=0;i<cont->nb_chans && result;++i)
	{
	  if (!lw_srvchan_send_keys(&(cont->chan[i]),&netkey))
	    {
	      result=0;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/* 
 * Sends a final OK message to all the clients
 */
int lw_srvcont_replicate_keys(LW_SRVCONT *cont,int lag)
{
  int result=1;
  LW_NETKEY netkey;
  int i,j;
  int increase_lag_counter=0;
  int decrease_lag_counter=0;
  int all_clients_request_increase_lag;
  int all_clients_request_decrease_lag;
  int a_client_requests_increase_lag;
  int a_client_requests_decrease_lag;
  int rounds=0;
  int last_rounds=0;
  int average_lag=0;
  time_t start,end;
  double elapsed;

  result=send_lag_key_presses(cont,lag);

  if (!result)
    {
      log_print_str("Error sending lag keys, disconnecting everyone");
      log_println();
    }
  else
    {
      start=lw_srvtime_seconds();

      while (cont->nb_chans>0)
	{
	  lw_netkey_reset(&netkey);

	  all_clients_request_increase_lag=1;
	  all_clients_request_decrease_lag=1;
	  a_client_requests_increase_lag=0;
	  a_client_requests_decrease_lag=0;

	  /*
	   * First we get the informations from all the players
	   */
	  for (i=0;i<cont->nb_chans;++i)
	    {
	      if (lw_srvchan_recv_keys(&(cont->chan[i]),&netkey))
		{
		  /*
		   * if at least a client does not request a lag
		   * increase, then we don't do it
		   */
		  if (netkey.cmd==LW_NETKEY_CMD_INCREASE_LAG)
		    {
		      a_client_requests_increase_lag=1;
		    }
		  else
		    {
		      all_clients_request_increase_lag=0;
		    }

		  /*
		   * if at least a client does not request a lag
		   * decrease, then we don't do it
		   */
		  if (netkey.cmd==LW_NETKEY_CMD_DECREASE_LAG)
		    {
		      a_client_requests_decrease_lag=1;
		    }
		  else
		    {
		      all_clients_request_decrease_lag=0;
		    }
		}
	      else
		{
		  /*
		   * There's been an error receiving data from this team
		   * so we decide to "close" it.
		   */
		  log_print_str("Disconnecting \"");
		  log_print_str(cont->chan[i].nickname);
		  log_print_str("\"");
		  log_println();
		  for (j=i+1;j<cont->nb_chans;++j)
		    {
		      cont->chan[j-1]=cont->chan[j];
		    }
		  cont->nb_chans--;
		}
	    }

	  /*
	   * we reset the cmd and arg fields
	   */
	  netkey.cmd=LW_NETKEY_CMD_NONE;
	  netkey.arg=0;

	  /*
	   * we use a counter system, so that all client must answer
	   * "yes we want a lag increase" several times in a row
	   * before we actually update the lag
	   */
	  if (all_clients_request_increase_lag)
	    {
	      increase_lag_counter++;
	    }
	  else
	    {
	      increase_lag_counter=0;
	    }

	  /*
	   * we use a counter system, so that all client must answer
	   * "yes we want a lag decrease" several times in a row
	   * before we actually update the lag
	   */
	  if (a_client_requests_decrease_lag) 
	    {
	      decrease_lag_counter++;
	    }
	  else
	    {
	      decrease_lag_counter=0;
	    }

	  /*
	   * If we need to increase the lag, we send blank key presses
	   */
	  if (increase_lag_counter>=LW_SRVCONT_INCREASE_LAG_LIMIT)
	    {
	      increase_lag_counter=0;
	      lag++;
	      send_lag_key_presses(cont,1);
	    }

	  /*
	   * If we need to decrease the lag, we simply do not send
	   * the key presses...
	   */
	  if (decrease_lag_counter>=LW_SRVCONT_DECREASE_LAG_LIMIT)
	    {
	      decrease_lag_counter=0;
	      lag--;
	    }
	  else
	    {
	      /*
	       * Now we send the information to all the players
	       */
	      for (i=0;i<cont->nb_chans;++i)
		{
		  if (!lw_srvchan_send_keys(&(cont->chan[i]),&netkey))
		    {
		      log_print_str("Disconnecting ");
		      log_print_str(cont->chan[i].nickname);
		      log_println();
		      /*
		       * There's been an error receiving data from this team
		       * so we decide to "close" it.
		       */
		      for (j=i+1;j<cont->nb_chans;++j)
			{
			  cont->chan[j-1]=cont->chan[j];
			}
		      cont->nb_chans--;
		    }
		}
	    }

	  rounds++;
	  last_rounds++;
	  average_lag+=lag;

	  end=lw_srvtime_seconds();
	  elapsed = end-start;

	  if (((int) elapsed)>LW_SRVCONT_DISPLAY_STATE_DELAY)
	    {
	      if (rounds<=0)
		{
		  rounds=1;
		}
	      if (last_rounds<=0)
		{
		  last_rounds=1;
		}

	      log_print_int(rounds);
	      log_print_str(" rounds, ");
	      log_print_int(last_rounds/((int) elapsed));
	      log_print_str(" rounds/sec, average lag is ");
	      log_print_int(average_lag/last_rounds);
	      log_println();

	      start=end;
	      last_rounds=0;
	      average_lag=0;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/* 
 * Closes everything and frees the resources
 */
void lw_srvcont_close(LW_SRVCONT *cont)
{
  int i;

  lw_sock_close(cont->sock);

  log_print_str("Game over");
  log_println();

  for (i=0;i<cont->nb_chans;++i)
    {
      lw_srvchan_close(&(cont->chan[i]));
    }

  if (cont->map!=NULL)
    {
      free (cont->map);
    }

  reset(cont);
}

