/********************************************************************/
/*                                                                  */
/*            L   I  QQ  U U I DD    W   W  A  RR    555            */
/*            L   I Q  Q U U I D D   W   W A A R R   5              */
/*            L   I Q  Q U U I D D   W W W AAA RR    55             */
/*            L   I Q Q  U U I D D   WW WW A A R R     5            */
/*            LLL I  Q Q  U  I DD    W   W A A R R   55             */
/*                                                                  */
/*                             b                                    */
/*                             bb  y y                              */
/*                             b b yyy                              */
/*                             bb    y                              */
/*                                 yy                               */
/*                                                                  */
/*                     U U       FFF  O   O  TTT                    */
/*                     U U       F   O O O O  T                     */
/*                     U U TIRET FF  O O O O  T                     */
/*                     U U       F   O O O O  T                     */
/*                      U        F    O   O   T                     */
/*                                                                  */
/********************************************************************/

/*****************************************************************************/
/* Liquid War is a unique multiplayer wargame                                */
/* Copyright (C) 1998-2002 Christian Mauduit                                 */
/*                                                                           */
/* This program is free software; you can redistribute it and/or modify      */
/* it under the terms of the GNU General Public License as published by      */
/* the Free Software Foundation; either version 2 of the License, or         */
/* (at your option) any later version.                                       */
/*                                                                           */
/* This program is distributed in the hope that it will be useful,           */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             */
/* GNU General Public License for more details.                              */
/*                                                                           */
/* You should have received a copy of the GNU General Public License         */
/* along with this program; if not, write to the Free Software               */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
/*                                                                           */
/* Liquid War homepage : http://www.ufoot.org/liquidwar                      */
/* Contact author      : ufoot@ufoot.org                                     */
/*****************************************************************************/

/********************************************************************/
/* name          : protocol.c                                       */
/* content       : messages sent by the client when connecting      */
/* last update   : April 16th 2001                                  */
/********************************************************************/

/*==================================================================*/
/* includes                                                         */
/*==================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "sock2gen.h"
#include "netmess.h"
#include "netmap.h"
#include "netconf.h"
#include "protocol.h"

/*==================================================================*/
/* static functions                                                 */
/*==================================================================*/

static int recv_ok(int sock);

/*==================================================================*/
/* fonctions                                                        */
/*==================================================================*/

/*------------------------------------------------------------------*/
static int recv_ok(int sock)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];
  int ret;
  LW_NETMESS *netmess;
  
  while ((ret=lw_sock_recv_str(sock,message))==0);
    
  if (ret>0)
    {
      netmess=lw_netmess_read(message);
      if (netmess!=NULL)
	{
	  if (netmess->code==LW_NETMESS_CODE_OK && netmess->argc==0)
	    {
	      result=1;
	    }
	  lw_netmess_free(netmess);
	}
    }
    

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends a dummy "PING" message to the server, to check that the
 * connection is working OK.
 */
int lw_protocol_do_ping (int sock)
{
  int result=0;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_PING))
    {
      if (recv_ok(sock))
	{
	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends the name of the running client to the server. This is to
 * prevent a liquidwar client to connect itself on something else
 * than a real liquidwar server.
 */
int lw_protocol_tell_program (int sock, char *program)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];

  sprintf(message,"%s %s",LW_NETMESS_TEXT_PROGRAM,program);
  if (lw_sock_send_str(sock,message))
    {
      if (recv_ok(sock))
	{
	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends the version of the client to the server.
 */
int lw_protocol_tell_version (int sock, char *version)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];

  sprintf(message,"%s %s",LW_NETMESS_TEXT_VERSION,version);
  if (lw_sock_send_str(sock,message))
    {
      if (recv_ok(sock))
	{
	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Asks the server how many players can connect to it.
 */
int lw_protocol_ask_free_teams (int sock,int *nb)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];
  int ret;
  LW_NETMESS *netmess;

  *nb=0;
  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_FREE))
    {
      while ((ret=lw_sock_recv_str(sock,message))==0);
      if (ret>0)
	{
	  netmess=lw_netmess_read(message);
	  if (netmess!=NULL)
	    {
	      if (netmess->code==LW_NETMESS_CODE_OK && netmess->argc==1)
		{
		  *nb=atoi(netmess->argv[0]);
		  result=1;
		}
	      lw_netmess_free(netmess);
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Inform the server that this player will be playing.
 */
int lw_protocol_reserve_team (int sock, int num, char *name)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];

  sprintf(message,"%s %d,'%s'",LW_NETMESS_TEXT_TEAM,num,name);
  if (lw_sock_send_str(sock,message))
    {
      if (recv_ok(sock))
	{
	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Ask the server who is playing on a given channel
 */
int lw_protocol_ask_who (int sock, 
			 int num,
			 LW_WHO *who)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];
  int ret;
  LW_NETMESS *netmess;

  who->active=0;
  who->network=0;
  who->server_id=-1;
  who->name[0]='\0';

  sprintf(message,"%s %d",LW_NETMESS_TEXT_WHO,num);
  if (lw_sock_send_str(sock,message))
    {
      while ((ret=lw_sock_recv_str(sock,message))==0);
      if (ret>0)
	{
	  netmess=lw_netmess_read(message);
	  if (netmess!=NULL)
	    {
	      switch (netmess->code)
		{
		case LW_NETMESS_CODE_NOBODY:
		  if (netmess->argc==0)
		    {
		      /*
		       * We leave the server id to the default -1 value
		       */
		      result=1;
		    }
		  break;
		case LW_NETMESS_CODE_NETWORK:
		  who->network=1;
		  /*
		   * no "break" here since the only difference between
		   * the LOCAL and NETWORK messages is that the network
		   * flag is set to 1 in the "NETWORK" case.
		   */
		case LW_NETMESS_CODE_LOCAL:
		  if (netmess->argc>=1 && netmess->argc==2)
		    {
		      who->active=1;
		      who->server_id=atoi(netmess->argv[0]);
		      strncpy(who->name,netmess->argv[1],NAME_SIZE);
		      who->name[NAME_SIZE]='\0';
		      result=1;
		    }
		  break;
		}
	      lw_netmess_free(netmess);
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends the selected map to the server
 */
int lw_protocol_send_map (int sock, void *map)
{
  int result=0;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_SENDMAP))
    {
      if (lw_netmap_send(sock,map))
	{
	  if (recv_ok(sock))
	    {
	      result=1;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Receives the map from the server
 */
int lw_protocol_recv_map (int sock, void **map)
{
  int result=0;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_RECVMAP))
    {
      if (lw_netmap_recv(sock,map))
	{
	  if (recv_ok(sock))
	    {
	      result=1;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends the config to the server
 */
int lw_protocol_send_config (int sock)
{
  int result=0;
  LW_NETCONF config;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_SENDCONFIG))
    {
      config.cursor_increase_speed=CONFIG_CURSOR_INCREASE_SPEED;
      config.fighter_attack=CONFIG_FIGHTER_ATTACK;
      config.fighter_defense=CONFIG_FIGHTER_DEFENSE;
      config.fighter_new_health=CONFIG_FIGHTER_NEW_HEALTH;
      config.number_influence=CONFIG_NUMBER_INFLUENCE;
      config.fighter_number=CONFIG_FIGHTER_NUMBER;
      config.game_time=CONFIG_GAME_TIME;
      config.cpu_advantage=CONFIG_CPU_ADVANTAGE;

      if (lw_netconf_send(sock,&config))
	{
	  if (recv_ok(sock))
	    {
	      result=1;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Receives the config from the server
 */
int lw_protocol_recv_config (int sock)
{
  int result=0;
  LW_NETCONF config;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_RECVCONFIG))
    {
      if (lw_netconf_recv(sock,&config))
	{
	  if (recv_ok(sock))
	    {
	      CONFIG_CURSOR_INCREASE_SPEED=config.cursor_increase_speed;
	      CONFIG_FIGHTER_ATTACK=config.fighter_attack;
	      CONFIG_FIGHTER_DEFENSE=config.fighter_defense;
	      CONFIG_FIGHTER_NEW_HEALTH=config.fighter_new_health;
	      CONFIG_NUMBER_INFLUENCE=config.number_influence;
	      CONFIG_FIGHTER_NUMBER=config.fighter_number;
	      CONFIG_GAME_TIME=config.game_time;
	      CONFIG_CPU_ADVANTAGE=config.cpu_advantage;

	      result=1;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Tell the server that the client is ready to play
 */
int lw_protocol_i_am_ready (int sock)
{
  int result=0;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_READY))
    {
      if (recv_ok(sock))
	{
	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Tells the server one is waiting for the game to start
 */
int lw_protocol_waiting (int sock, int *waited_teams)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];
  int ret;
  LW_NETMESS *netmess;

  (*waited_teams)=0;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_WAITING))
    {
      while ((ret=lw_sock_recv_str(sock,message))==0);
      if (ret>0)
	{
	  netmess=lw_netmess_read(message);
	  if (netmess!=NULL)
	    {
	      if (netmess->code==LW_NETMESS_CODE_OK && netmess->argc==1)
		{
		  *waited_teams=atoi(netmess->argv[0]);
		  result=1;
		}
	      lw_netmess_free(netmess);
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Forces the server to start the game
 */
int lw_protocol_force_start (int sock)
{
  int result=0;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_START))
    {
      if (recv_ok(sock))
	{
	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends a "NEXT" message
 */
int lw_protocol_next (int sock)
{
  int result=0;

  if (lw_sock_send_str(sock,LW_NETMESS_TEXT_NEXT))
    {
      if (recv_ok(sock))
	{
	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends a "TEAMSTARTINFO" message
 */
int lw_protocol_ask_teamstartinfo (int sock, int team,
				   LW_TEAMSTARTINFO *team_start_info)
{
  int result=0;
  char message[LW_SOCK_MESSAGE_SIZE];
  int ret;
  LW_NETMESS *netmess;

  lw_teamstartinfo_reset(team_start_info);

  sprintf(message,"%s %d",LW_NETMESS_TEXT_TEAMSTARTINFO,team);
  if (lw_sock_send_str(sock,message))
    {
      while ((ret=lw_sock_recv_str(sock,message))==0);
      if (ret>0)
	{
	  netmess=lw_netmess_read(message);
	  if (netmess!=NULL)
	    {
	      switch (netmess->code)
		{
		case LW_NETMESS_CODE_OK:
		  if (netmess->argc==3)
		    {
		      team_start_info->active=atoi(netmess->argv[0]);
		      team_start_info->start=atoi(netmess->argv[1]);
		      strncpy(team_start_info->name,netmess->argv[2],NAME_SIZE);
		      team_start_info->name[NAME_SIZE]='\0';

		      result=1;
		    }
		  break;
		}
	      lw_netmess_free(netmess);
	    }
	}
    }

  return result;
}







