/*
* A Random map generator for Liquid War.
*
* Copyright (C) 2003, David Redick, Christian Mauduit
* Released under the GNU General Public License (v2)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "map.h"
#include "func.h"


void do_args(int argc, char **argv);
void print_help();
void print_version();

#ifdef DOS
/*
 * These macros reduce the size of the DOS executable
 */
BEGIN_GFX_DRIVER_LIST
END_GFX_DRIVER_LIST

BEGIN_COLOR_DEPTH_LIST
COLOR_DEPTH_8
END_COLOR_DEPTH_LIST

BEGIN_DIGI_DRIVER_LIST
END_DIGI_DRIVER_LIST

BEGIN_MIDI_DRIVER_LIST
END_MIDI_DRIVER_LIST

BEGIN_JOYSTICK_DRIVER_LIST
END_JOYSTICK_DRIVER_LIST
#endif

#ifdef WIN32
/*
 * Under Win32 we want a console executable so we don't use
 * the "END_OF_MAIN" stuff, and this implies to undef "main".
 */
#undef main
#endif

int main( int argc, char **argv )
{
     PALETTE pal;
     int i;


     /* init allegro and init palette */
     install_allegro(SYSTEM_NONE, &errno, atexit);
     set_color_depth(8);
     set_color_conversion(COLORCONV_REDUCE_TO_256);
     for( i = 0; i < 256; i++ )
     {
          /* divided by 4 because the colour value ranges from 0-63 */
          pal[i].r = pal[i].g = pal[i].b = i/4;
     }


     srand(time(NULL));

     init_map();
     do_args(argc, argv);

     map.map = create_bitmap_ex(8, map.width, map.height);

     clear_map();

     printf("generating map using: %s...\n", func[map.func_id].name);
     (*map.func)();

     print_map();

     save_bitmap(map.filename, map.map, pal);

return EXIT_SUCCESS;
}

#ifndef WIN32
END_OF_MAIN();
#endif

/*****************************************************************************/

void do_args( int argc, char **argv )
{
     int i;

     for( i = 1; i < argc; i++ )
     {
          if( strcmp(argv[i], "-o") == 0 || strcmp(argv[i], "--out") == 0 )
          {
               i++;
               free(map.filename);
               map.filename = malloc( sizeof(char) * strlen(argv[i]) );
               if( map.filename == NULL )
               {
                    fprintf(stderr, "fatal error: can't malloc space for map name.\n");
                    exit(EXIT_FAILURE);
               }
               strcpy(map.filename, argv[i]);
          }
          else if( strcmp(argv[i], "-w") == 0 || strcmp(argv[i], "--width") == 0 )
          {
               map.width = atoi(argv[++i]);
               if( map.width < MIN_MAP_WIDTH )
               {
                    fprintf(stderr, "map width too small using: %d\n", MIN_MAP_WIDTH);
                    map.width = MIN_MAP_WIDTH;
               }
               else if( map.width > MAX_MAP_WIDTH )
               {
                    fprintf(stderr, "map width too large using: %d\n", MAX_MAP_WIDTH);
                    map.width = MAX_MAP_WIDTH;
               }
          }
          else if( strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "--height") == 0 )
          {
               map.height = atoi(argv[++i]);
               if( map.height < MIN_MAP_HEIGHT )
               {
                    fprintf(stderr, "map height too small using: %d\n", MIN_MAP_HEIGHT);
                    map.height = MIN_MAP_HEIGHT;
               }
               else if( map.height > MAX_MAP_HEIGHT )
               {
                    fprintf(stderr, "map height too large using: %d\n", MAX_MAP_HEIGHT);
                    map.height = MAX_MAP_HEIGHT;
               }
          }
          else if( strcmp(argv[i], "-g") == 0 || strcmp(argv[i], "--grid") == 0 )
          {
               int grid_size = atoi(argv[++i]);
               if( grid_size < MIN_MAP_GRID )
               {
                    fprintf(stderr, "map grid too small using: %d\n", MIN_MAP_GRID);
                    grid_size = MIN_MAP_GRID;
               }
               else if( grid_size > MAX_MAP_GRID )
               {
                    fprintf(stderr, "map grid too large using: %d\n", MAX_MAP_GRID);
                    grid_size = MAX_MAP_GRID;
               }

               map.num_row = map_grid_size[grid_size][0];
               map.num_col = map_grid_size[grid_size][1];
          }
          else if( strcmp(argv[i], "-f") == 0 || strcmp(argv[i], "--function") == 0 )
          {
               int f;
               i++;
               for( f = 0; f < MAX_FUNC; f++ )
               {
                    if( strcmp(argv[i], func[f].name) == 0 )
                         break;
               }
               if( f >= MAX_FUNC )
               {
                    fprintf(stderr, "fatal error: can't find function: %s\n", argv[i]);
                    exit(EXIT_FAILURE);
               }
               else
               {
                    map.func_id = f;
                    map.func = func[f].func;
               }
          }
          else if( strcmp(argv[i], "-l") == 0 || strcmp(argv[i], "--list") == 0 )
          {
               int f;
               printf("Random map generating functions:\n");
               for( f = 0; f < MAX_FUNC; f++ )
                    printf("%2d)  %s\t%s\n", f, func[f].name, func[f].desc);

               exit(EXIT_SUCCESS);
          }
          else if( strcmp(argv[i], "--help") == 0 )
          {
               print_version();
               print_help();
               exit(EXIT_SUCCESS);
          }
          else if( strcmp(argv[i], "--version") == 0 )
          {
               print_version();
               exit(EXIT_SUCCESS);
          }
          else
          {
               fprintf(stderr, "error: unknown arg: %s\n", argv[i]);
               fprintf(stderr, "ignoring...\n");
          }
     }

return;
}

/*****************************************************************************/

void print_help()
{
     printf("-o    --out       <filename>        Save bitmap to <filename>.    [lwmap.bmp]\n");
     printf("-w    --width     <%d-%d>         Bitmap width.                 [%d]\n",
          MIN_MAP_WIDTH, MAX_MAP_WIDTH, DEFAULT_MAP_WIDTH);
     printf("-h    --height    <%d-%d>         Bitmap height.                [%d]\n",
          MIN_MAP_HEIGHT, MAX_MAP_HEIGHT, DEFAULT_MAP_HEIGHT);
     printf("-g    --grid      <%d-%d>             Map Grid Size                 [random]\n",
          MIN_MAP_GRID, MAX_MAP_GRID);
     printf("\n");
     printf("-f    --function  <function_name>   Which function to use.        [random]\n");
     printf("-l    --list                        List all functions.\n");
     printf("\n");
     printf("      --help                        Print this help.\n");
     printf("      --version                     Print Version.\n");

return;
}

/*****************************************************************************/

void print_version()
{
     printf(
     "Liquid War Random Map Generator Version 0.0.0\n"
     "Copyright (C) 2003, David Redick, Chris Guirl, Christian Mauduit.\n"
     "Released under the GNU General Public License (v2).\n"
     );

return;
}

/*****************************************************************************/
