/*
  Liquid War is a multiplayer wargame.
  Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War homepage : http://www.ufoot.org/liquidwar
  Contact author      : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <GL/glu.h>
#include <SDL/SDL_image.h>
#include <expat.h>
#include "../liquidwar6common.h"
#include "../liquidwar6gfx.h"
#include "internal.h"

#define IMAGE_DIR_1 "gfx/image/"
#define IMAGE_DIR_2 "data/gfx/image/"
#define IMAGE_DIR_3 "../data/gfx/image/"
#define IMAGE_DIR_4 LW6_DATA_DIR "gfx/image/"
#define IMAGE_DIR_DEFAULT LW6_DATA_DIR "gfx/image/"
#define IMAGE_TEST "background.png"

#define FONT_DIR_1 "gfx/font/"
#define FONT_DIR_2 "data/gfx/font/"
#define FONT_DIR_3 "../data/gfx/font/"
#define FONT_DIR_4 LW6_DATA_DIR "gfx/font/"
#define FONT_DIR_DEFAULT LW6_DATA_DIR "gfx/font/"
#define FONT_TEST "FreeSansBold.ttf"

/*
 * Dummy wrapper to handle cast & errors.
 */
static SDL_Surface *load_image(char *dirname, char *filename) {
  SDL_Surface *ret=NULL;
  int len;
  char *buf=NULL;

  len=strlen(dirname)+strlen(filename);
  buf=(char *) LW6SYS_MALLOC(len+1);  
  if (buf) {
    memset(buf,0,len+1);
    snprintf(buf, len+1, "%s%s", dirname, filename);
    ret=IMG_Load(buf);
    LW6SYS_FREE(buf);
  }

  if (ret==NULL) {
    /*
     * Carefull not to use buf in error message.
     */
    lw6sys_log_error("gfx", _("unable to load image \"%s\""), buf);
  }

  return ret;
}

/*
 * Loads images from disk.
 */
int _lw6gfx_load_images(_LW6GFX_CONTEXT *context) {
  int ret=0;
  char *image_dir=NULL;

  if (lw6sys_file_exists(IMAGE_DIR_1 IMAGE_TEST)) {
    image_dir=IMAGE_DIR_1;
  } else if (lw6sys_file_exists(IMAGE_DIR_2 IMAGE_TEST)) {
    image_dir=IMAGE_DIR_2;
  } else if (lw6sys_file_exists(IMAGE_DIR_3 IMAGE_TEST)) {
    image_dir=IMAGE_DIR_3;
  } else if (lw6sys_file_exists(IMAGE_DIR_4 IMAGE_TEST)) {
    image_dir=IMAGE_DIR_4;
  }
  if (!image_dir) {
    image_dir=IMAGE_DIR_DEFAULT;
  }

  lw6sys_log_info("gfx", _("loading images from \"%s\""), image_dir);

  ret=((context->image_data.background=load_image(image_dir,"background.png"))!=NULL) &&
    ((context->image_data.background_bubble=load_image(image_dir,"background-bubble.png"))!=NULL);

  if (!ret) {
    /*
     * If we can't load images, we simply exit the game...
     */
    exit(1);
  }

  return ret;
}

int lw6gfx_load_images(void *context) {
  return _lw6gfx_load_images((_LW6GFX_CONTEXT *) context);
}

/*
 * Dummy wrapper to handle cast & errors.
 */
static void unload_image(SDL_Surface *image) {
  if (image!=NULL) {
    SDL_FreeSurface(image);
  } else {
    lw6sys_log_warning("gfx",_("trying to free NULL image"));
  }
}

/*
 * Free memory.
 */
void _lw6gfx_unload_images(_LW6GFX_CONTEXT *context) {
  unload_image(context->image_data.background);
  unload_image(context->image_data.background_bubble);

  memset(&context->image_data,0,sizeof(_LW6GFX_IMAGE_DATA));   
}

void lw6gfx_unload_images(void *context) {
  _lw6gfx_unload_images((_LW6GFX_CONTEXT *) context);
}

/*
 * Dummy wrapper to handle cast & errors.
 */
static GLuint load_texture(_LW6GFX_CONTEXT *context, SDL_Surface *surface) {
  GLuint ret=0;

  ret=_lw6gfx_surface2texture(context, surface);

  if (!ret) {
    lw6sys_log_error("gfx", _("unable to load texture"));
  }

  return ret;
}

/*
 * Dummy wrapper to handle cast & errors.
 */
static void unload_texture(_LW6GFX_CONTEXT *context, int texture) {
  if (texture!=0) {
    _lw6gfx_delete_texture(context,texture);
  } else {
    /*
     * Not sure wether 0 is actually an invalid texture id,
     * in doubt we issue a warning, if it ever shows up,
     * will be time to decide wether to disable this check or not.
     */
    lw6sys_log_warning("gfx",_("trying to delete NULL texture"));
  }
}

/*
 * Loads textures from disk.
 */
int _lw6gfx_load_textures(_LW6GFX_CONTEXT *context) {
  int ret=0;

  ret=((context->texture_data.background=load_texture(context, context->image_data.background))!=0) &&
    ((context->texture_data.background_bubble=load_texture(context, context->image_data.background_bubble))!=0);

  if (!ret) {
    /*
     * If we can't load textures, we simply exit the game...
     */
    exit(1);
  }

  return ret;
}

int lw6gfx_load_textures(void *context) {
  return _lw6gfx_load_textures((_LW6GFX_CONTEXT *) context);
}

/*
 * Free memory.
 */
void _lw6gfx_unload_textures(_LW6GFX_CONTEXT *context) {
  unload_texture(context,context->texture_data.background);
  unload_texture(context,context->texture_data.background_bubble);
  
  _lw6gfx_delete_scheduled_textures(context);
  
  memset(&context->texture_data,0,sizeof(_LW6GFX_TEXTURE_DATA));   
}

void lw6gfx_unload_textures(void *context) {
_lw6gfx_unload_textures((_LW6GFX_CONTEXT *) context);
}

/*
 * Dummy wrapper to handle cast & errors.
 */
static TTF_Font *load_font(char *dirname, char *filename, int size) {
  TTF_Font *ret=NULL;
  int len;
  char *buf=NULL;

  len=strlen(dirname)+strlen(filename);
  buf=(char *) LW6SYS_MALLOC(len+1);  
  if (buf) {
    memset(buf,0,len+1);
    snprintf(buf, len+1, "%s%s", dirname, filename);
    ret=TTF_OpenFont(buf,size);
    LW6SYS_FREE(buf);
  }

  if (ret==NULL) {
    lw6sys_log_error("gfx", _("unable to load font \"%s\""), filename);
  }

  return ret;
}

/*
 * Dummy wrapper to handle cast & errors.
 */
static void unload_font(TTF_Font *font) {
  if (font!=NULL) {
    TTF_CloseFont(font);
  } else {
    lw6sys_log_warning("gfx",_("trying to close NULL font"));
  }
}

/*
 * Loads fonts
 */
int _lw6gfx_load_fonts(_LW6GFX_CONTEXT *context) {
  int ret=0;
  char *font_dir=NULL;

  if (lw6sys_file_exists(FONT_DIR_1 FONT_TEST)) {
    font_dir=FONT_DIR_1;
  } else if (lw6sys_file_exists(FONT_DIR_2 FONT_TEST)) {
    font_dir=FONT_DIR_2;
  } else if (lw6sys_file_exists(FONT_DIR_3 FONT_TEST)) {
    font_dir=FONT_DIR_3;
  } else if (lw6sys_file_exists(FONT_DIR_4 FONT_TEST)) {
    font_dir=FONT_DIR_4;
  }
  if (!font_dir) {
    font_dir=FONT_DIR_DEFAULT;
  }

  lw6sys_log_info("gfx", _("loading fonts from \"%s\""), font_dir);

  ret=((context->font_data.menu=load_font(font_dir,"FreeSansBold.ttf",_LW6GFX_FONT_MENU_SIZE))!=NULL) &&
    ((context->font_data.system=load_font(font_dir,"FreeSansBold.ttf",_LW6GFX_FONT_SYSTEM_SIZE))!=NULL);

  if (!ret) {
    /*
     * If we can't load fonts, we simply exit the game...
     */
    exit(1);
  }

  return ret;  
}

int lw6gfx_load_fonts(void *context) {
  return _lw6gfx_load_fonts((_LW6GFX_CONTEXT *) context);
}

/*
 * Unload fonts, free memory
 */
void _lw6gfx_unload_fonts(_LW6GFX_CONTEXT *context) {
  unload_font(context->font_data.menu);
  unload_font(context->font_data.system);

  memset(&context->font_data,0,sizeof(_LW6GFX_FONT_DATA));   
}

void lw6gfx_unload_fonts(void *context) {
  _lw6gfx_unload_fonts((_LW6GFX_CONTEXT *) context);
}

