/*
Liquid War is a multiplayer wargame.
Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

Liquid War homepage : http://www.ufoot.org/liquidwar/v6
Contact author      : ufoot@ufoot.org
*/

#include <stdio.h>

#include "../liquidwar6common.h"
#include "../liquidwar6gfx.h"
#include "internal.h"

/*
 * Pushes a key event.
 */
static int push_keypress(_LW6GFX_KEYBOARD_STATE *state, int sym, int unicode, char *label) {
  (state->keypress_queue)++;
  if ((state->keypress_queue)<_LW6GFX_KEYPRESS_BUFFER_SIZE) {
    state->keypress_current=((state->keypress_current)+1)%_LW6GFX_KEYPRESS_BUFFER_SIZE;
    state->keypress_buffer[state->keypress_current].sym=sym;
    state->keypress_buffer[state->keypress_current].unicode=unicode;
    state->keypress_buffer[state->keypress_current].label=label;
  }
}

/*
 * Internal poll function.
 */
static int poll(_LW6GFX_CONTEXT *context) {
  int ret;
  _LW6GFX_KEYBOARD_STATE *state=&context->keyboard_state;

  SDL_Event event;

  ret=SDL_PollEvent(&event) ? 1 : 0;

  if (ret) {
    switch (event.type) {
    case SDL_KEYDOWN:
      push_keypress(state,
		    event.key.keysym.sym,
		    event.key.keysym.unicode,
		    SDL_GetKeyName(event.key.keysym.sym));
      break;
    case SDL_QUIT:
      state->quit=1;
      break;
    case SDL_VIDEORESIZE:
      _lw6gfx_resize_video_mode(context,event.resize.w,event.resize.h);
      break;
    }
  }

  return ret;
}

/*
 * Polls a key press.
 */
int _lw6gfx_poll_keypress(_LW6GFX_CONTEXT *context, LW6GFX_KEYPRESS *keypress) {
  int ret=0;

  poll(context);

  if (context->keyboard_state.keypress_queue>0) {
    ret=1;
    (context->keyboard_state.keypress_queue)--;
    if (keypress) {
      int pos;
      
      pos=(_LW6GFX_KEYPRESS_BUFFER_SIZE
	   +context->keyboard_state.keypress_current
	   -context->keyboard_state.keypress_queue)
	%_LW6GFX_KEYPRESS_BUFFER_SIZE;
      *keypress=(context->keyboard_state.keypress_buffer)[pos];
    }
  }

  return ret;
}

int lw6gfx_poll_keypress(void *context, LW6GFX_KEYPRESS *keypress) {
  return _lw6gfx_poll_keypress((_LW6GFX_CONTEXT *) context, keypress);
}

/*
 * Following functions are used to export to Guile the internal
 * SDL esc/up/down/left/right/enter keycodes which are the basics
 * to handle menu stuff. Other keys are basically ascii keys 
 * and/or special control keys but which do not have any special
 * meaning since they're used for controlling cursors, independently
 * of their expected behavior. Point is esc/up/down/left/right/enter
 * are necessary for menu stuff. One option would be to export all
 * SLDK_ symbols to Guile, but it's 1) useless and 2) it would tie
 * scheme code to SDL, which is not needed nor required nor wished.
 */

int lw6gfx_keysym_esc(void *context) {
  return SDLK_ESCAPE;
}

int lw6gfx_keysym_up(void *context) {
  return SDLK_UP;
}

int lw6gfx_keysym_down(void *context) {
  return SDLK_DOWN;
}

int lw6gfx_keysym_left(void *context) {
  return SDLK_LEFT;
}

int lw6gfx_keysym_right(void *context) {
  return SDLK_RIGHT;
}

int lw6gfx_keysym_enter(void *context) {
  return SDLK_RETURN;
}

int lw6gfx_keysym_del(void *context) {
  return SDLK_DELETE;
}

int lw6gfx_keysym_backspace(void *context) {
  return SDLK_BACKSPACE;
}

int lw6gfx_keysym_help(void *context) {
  return SDLK_F1;
}

int lw6gfx_keysym_quit(void *context) {
  return SDLK_F10;
}

/*
 * Polls a quit event
 */
int _lw6gfx_poll_quit(_LW6GFX_CONTEXT *context) {
  poll(context);

  return context->keyboard_state.quit;
}

int lw6gfx_poll_quit(void *context) {
  return _lw6gfx_poll_quit((_LW6GFX_CONTEXT *) context);
}

