/*
Liquid War is a multiplayer wargame.
Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

Liquid War homepage : http://www.ufoot.org/liquidwar/v6
Contact author      : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6GFX_INTERNAL_H
#define LIQUIDWAR6GFX_INTERNAL_H

#include <GL/gl.h>
#include <SDL/SDL.h>
#include <SDL/SDL_ttf.h>

#define _LW6GFX_FONT_MENU_SIZE 36
#define _LW6GFX_FONT_SYSTEM_SIZE 12

#define _LW6GFX_MENU_TEXTURE_W 512
#define _LW6GFX_MENU_TEXTURE_H 128

#define _LW6GFX_BACKGROUND_MAX_BUBBLES 64

typedef struct {
  int width;
  int height;
  int bpp;
  int fullscreen;
} _LW6GFX_SCREEN;

typedef enum 
  {
    _LW6GFX_RENDER_NONE     = 0,
    _LW6GFX_RENDER_2D       = 1,
    _LW6GFX_RENDER_2D_BLEND = 2,
    _LW6GFX_RENDER_3D_MAP   = 3,
    _LW6GFX_RENDER_3D_MENU  = 4
  }
  _LW6GFX_RENDER_MODE;

typedef struct {
  int active;
  int t0;
  float size;
  float x;
} _LW6GFX_BACKGROUND_BUBBLE;

typedef struct {
  _LW6GFX_BACKGROUND_BUBBLE bubbles[_LW6GFX_BACKGROUND_MAX_BUBBLES];
} _LW6GFX_BACKGROUND_STATE;

/*
 * Internal event buffer. One might wonder why we recode
 * some stuff which could be handled by SDL directly.
 * Answer: we need to do some basic processing before
 * sending the info to guile/scheme. For instance we
 * want key strokes to arrive with all the keycode/scancode/label
 * to be sent. We might also wish to send virtual key events
 * such as "mouse up" or "joy up".
 */
#define _LW6GFX_KEYPRESS_BUFFER_SIZE 256

typedef struct {
  LW6GFX_KEYPRESS keypress_buffer[_LW6GFX_KEYPRESS_BUFFER_SIZE];  
  int keypress_current;
  int keypress_queue;
  int quit;
}
_LW6GFX_KEYBOARD_STATE;

typedef struct {
  SDL_Surface *background;
  SDL_Surface *background_bubble;
} 
_LW6GFX_IMAGE_DATA;

typedef struct {
  GLuint texture;
  /*
   * Note that an opaque pointer here is ugly, however pointer
   * on "self-structs" are always ugly in C, I've got 3 minutes
   * maximum to fix that problem, it's solved in 3 minutes by
   * a simple void * and 2 brutal casts, final dot.
   */
  void *next;
}
_LW6GFX_TEXTURE_CHAIN;

typedef struct {
  GLuint background;
  GLuint background_bubble;
  _LW6GFX_TEXTURE_CHAIN *to_delete; 
} 
_LW6GFX_TEXTURE_DATA;

typedef struct {
  TTF_Font *system;
  TTF_Font *menu;
}
_LW6GFX_FONT_DATA;

typedef struct {
  SDL_Color rgb;
  float alpha;
} 
_LW6GFX_COLOR;

typedef struct {
  float persp_fovy;
  float persp_znear;
  float persp_zfar;
  int menu_max_displayed_items;
  _LW6GFX_COLOR menu_fg;
  _LW6GFX_COLOR menu_bg;
  _LW6GFX_COLOR menu_fg_selected;
  _LW6GFX_COLOR menu_bg_selected;
  float menucyl_radius1;
  float menucyl_between1;
  float menucyl_height;
  int menucyl_slices1;
  int menucyl_slices_min;
  int menucyl_stacks;
  float menucyl_oscil_range1;
  int menucyl_oscil_period;
  _LW6GFX_COLOR system_font_fg;
  _LW6GFX_COLOR system_font_shade;
  int system_font_dh;
  int system_font_dw;
  float system_font_hcoef;
  int background_nb_bubbles;
  float background_yspeed;
  float background_bubble_yspeed;
  float background_bubble_size_min;
  float background_bubble_size_max;
}
_LW6GFX_CONST_DATA;

typedef struct {
  _LW6GFX_SCREEN screen;
  _LW6GFX_RENDER_MODE render_mode;
  _LW6GFX_IMAGE_DATA image_data;
  _LW6GFX_TEXTURE_DATA texture_data;
  _LW6GFX_FONT_DATA font_data;
  _LW6GFX_BACKGROUND_STATE background_state;
  _LW6GFX_KEYBOARD_STATE keyboard_state;
  _LW6GFX_CONST_DATA const_data;
} 
_LW6GFX_CONTEXT;

/*
 * In background.c
 */
extern void _lw6gfx_display_background(_LW6GFX_CONTEXT *context);

/*
 * In const.c
 */
extern int _lw6gfx_load_consts(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_unload_consts(_LW6GFX_CONTEXT *context);

/*
 * In data.c
 */
extern int _lw6gfx_load_images(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_unload_images(_LW6GFX_CONTEXT *context);
extern int _lw6gfx_load_textures(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_unload_textures(_LW6GFX_CONTEXT *context);
extern int _lw6gfx_load_fonts(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_unload_fonts(_LW6GFX_CONTEXT *context);

/*
 * In event.c
 */
extern int _lw6gfx_poll_keypress(_LW6GFX_CONTEXT *context, LW6GFX_KEYPRESS *keypress);
extern int _lw6gfx_poll_quit(_LW6GFX_CONTEXT *context);

/*
 * In font.c
 */
extern int _lw6gfx_get_system_text_width(_LW6GFX_CONTEXT *context, char *text);
extern void _lw6gfx_draw_system_text(_LW6GFX_CONTEXT *context, char *text, int x, int y);
extern void _lw6gfx_draw_system_top_right(_LW6GFX_CONTEXT *context, char **text);
extern void _lw6gfx_draw_system_bottom_left(_LW6GFX_CONTEXT *context, char **text);

/*
 * In info.c
 */
extern void _lw6gfx_display_sysinfo(_LW6GFX_CONTEXT *context, int fps, int rps, int bytes_in, int bytes_out);
extern void _lw6gfx_display_log(_LW6GFX_CONTEXT *context, char **log);

/*
 * In light.c
 */
extern void _lw6gfx_main_light(_LW6GFX_CONTEXT *context);

/*
 * In menu.c
 */
extern void _lw6gfx_display_menu(_LW6GFX_CONTEXT *context, LW6GFX_MENU *menu);

/*
 * In render.c
 */
extern void _lw6gfx_prepare_buffer(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_swap_buffers(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_set_render_mode_2d(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_set_render_mode_2d_blend(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_set_render_mode_3d_map(_LW6GFX_CONTEXT *context);
extern void _lw6gfx_set_render_mode_3d_menu(_LW6GFX_CONTEXT *context);

/*
 * In setup.c
 */
extern _LW6GFX_CONTEXT *_lw6gfx_init();
extern int _lw6gfx_set_video_mode(_LW6GFX_CONTEXT *context, int width, int height, int fullscreen);
extern int _lw6gfx_resize_video_mode(_LW6GFX_CONTEXT *context,int width, int height);
extern void _lw6gfx_quit(_LW6GFX_CONTEXT *context);

/*
 * In surface.c
 */
extern SDL_Surface *_lw6gfx_create_surface(int w, int h);
extern void _lw6gfx_delete_surface(SDL_Surface *surface);
extern void _lw6gfx_clear_surface(SDL_Surface *surface, SDL_Color color);

/*
 * In texture.c
 */
extern float _lw6gfx_texture_scale(int size);
extern GLuint _lw6gfx_surface2texture_xywh(_LW6GFX_CONTEXT *context, SDL_Surface *surface, int x, int y, int w, int h);
extern GLuint _lw6gfx_surface2texture_wh(_LW6GFX_CONTEXT *context, SDL_Surface *surface, int w, int h);
extern GLuint _lw6gfx_surface2texture(_LW6GFX_CONTEXT *context, SDL_Surface *surface);
extern void _lw6gfx_delete_texture(_LW6GFX_CONTEXT *context, GLuint texture);
extern void _lw6gfx_schedule_delete_texture(_LW6GFX_CONTEXT *context, GLuint texture);
extern void _lw6gfx_delete_scheduled_textures(_LW6GFX_CONTEXT *context);

/*
 * In timer.c
 */
extern void _lw6gfx_delay(_LW6GFX_CONTEXT *context, int ms);
extern int _lw6gfx_get_ticks(_LW6GFX_CONTEXT *context);

#endif
