/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <GL/glu.h>
#include <SDL/SDL_image.h>
#include <expat.h>

#include "config.h"
#include "../../gfx.h"
#include "gl-utils.h"
#include "gl-utils-internal.h"

#define FONT_DIR "gfx/gl/utils/font/"
#define FONT_FILE "FreeSansBold.ttf"

/*
 * Dummy wrapper to handle cast & errors.
 */
SDL_Surface *
mod_gl_utils_load_image (MOD_GL_UTILS_CONTEXT * utils_context, char *dirname,
			 char *filename)
{
  SDL_Surface *ret = NULL;
  char *buf = NULL;

  buf = lw6sys_str_concat (dirname, filename);
  if (buf)
    {
      char *image_file;

      image_file = lw6sys_find_data_file (buf);

      if (image_file)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl",
		      _("loading image \"%s\""), image_file);
	  ret = IMG_Load (image_file);
	  LW6SYS_FREE (image_file);
	}
      LW6SYS_FREE (buf);
    }

  if (ret != NULL)
    {
      utils_context->surface_counter.new_counter++;
    }
  else
    {
      /*
       * Carefull not to use buf in error message.
       */
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		  _("unable to load image \"%s\""), filename);
    }

  return ret;
}

/*
 * Dummy wrapper to handle cast & errors.
 */
void
mod_gl_utils_unload_image (MOD_GL_UTILS_CONTEXT * utils_context,
			   SDL_Surface * image)
{
  mod_gl_utils_delete_surface (utils_context, image);
}

/*
 * Dummy wrapper to handle cast & errors.
 */
GLuint
mod_gl_utils_load_texture (MOD_GL_UTILS_CONTEXT * utils_context,
			   SDL_Surface * surface)
{
  GLuint ret = 0;

  ret = mod_gl_utils_surface2texture (utils_context, surface);

  if (!ret)
    {
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		  _("unable to load texture"));
    }

  return ret;
}

/*
 * Dummy wrapper to handle cast & errors.
 */
void
mod_gl_utils_unload_texture (MOD_GL_UTILS_CONTEXT * utils_context,
			     int texture)
{
  if (texture != 0)
    {
      mod_gl_utils_delete_texture (utils_context, texture);
    }
  else
    {
      /*
       * Not sure wether 0 is actually an invalid texture id,
       * in doubt we issue a warning, if it ever shows up,
       * will be time to decide wether to disable this check or not.
       */
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _("trying to delete NULL texture"));
    }
}

/*
 * Dummy wrapper to handle cast & errors.
 */
static TTF_Font *
load_font (char *dirname, char *filename, int size)
{
  TTF_Font *ret = NULL;
  char *buf = NULL;

  buf = lw6sys_str_concat (dirname, filename);
  if (buf)
    {
      char *font_file;

      font_file = lw6sys_find_data_file (buf);

      if (font_file)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("loading font \"%s\""),
		      font_file);
	  ret = TTF_OpenFont (font_file, size);
	  LW6SYS_FREE (font_file);
	}
      LW6SYS_FREE (buf);
    }

  if (ret == NULL)
    {
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		  _("unable to load font \"%s\""), filename);
    }

  return ret;
}

/*
 * Dummy wrapper to handle cast & errors.
 */
static void
unload_font (TTF_Font * font)
{
  if (font != NULL)
    {
      TTF_CloseFont (font);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _("trying to close NULL font"));
    }
}

/*
 * Loads fonts
 */
int
mod_gl_utils_load_fonts (MOD_GL_UTILS_CONTEXT * utils_context)
{
  int ret = 0;

  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("loading fonts"));

  ret =
    ((utils_context->font_data.menu =
      load_font (FONT_DIR, FONT_FILE,
		 utils_context->const_data.menu_font_size)) != NULL)
    &&
    ((utils_context->font_data.system =
      load_font (FONT_DIR, FONT_FILE,
		 utils_context->const_data.system_font_size)) != NULL);

  if (!ret)
    {
      /*
       * If we can't load fonts, we simply exit the game...
       */
      exit (1);
    }

  return ret;
}

/*
 * Unload fonts, free memory
 */
void
mod_gl_utils_unload_fonts (MOD_GL_UTILS_CONTEXT * utils_context)
{
  unload_font (utils_context->font_data.menu);
  unload_font (utils_context->font_data.system);

  memset (&utils_context->font_data, 0, sizeof (MOD_GL_UTILS_FONT_DATA));
}

/*
 * Putting all the load/unload functions together
 */
int
mod_gl_utils_load_data (MOD_GL_UTILS_CONTEXT * utils_context)
{
  return
    ((utils_context->texture_data.to_delete = lw6sys_list_new (NULL)) != NULL)
    && _mod_gl_utils_load_consts (utils_context)
    && mod_gl_utils_load_fonts (utils_context);
}

int
mod_gl_load_data (void *context)
{
  return mod_gl_utils_load_data ((MOD_GL_UTILS_CONTEXT *) context);
}

void
mod_gl_utils_unload_data (MOD_GL_UTILS_CONTEXT * utils_context)
{
  mod_gl_utils_clear_menucache (utils_context);

  mod_gl_utils_clear_texturecache (utils_context);

  mod_gl_utils_delete_scheduled_textures (utils_context);
  LW6SYS_FREE (utils_context->texture_data.to_delete);

  memset (&utils_context->texture_data, 0,
	  sizeof (MOD_GL_UTILS_TEXTURE_DATA));

  _mod_gl_utils_unload_consts (utils_context);
  mod_gl_utils_unload_fonts (utils_context);
}

void
mod_gl_unload_data (void *context)
{
  mod_gl_utils_unload_data ((MOD_GL_UTILS_CONTEXT *) context);
}
