/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <GL/glu.h>
#include <SDL/SDL_image.h>

#include "config.h"
#include "../../gfx.h"
#include "gl-utils.h"
#include "gl-utils-internal.h"

/*
 * Initialize display.
 */
int
mod_gl_utils_set_video_mode (MOD_GL_UTILS_CONTEXT * utils_context, int width,
			     int height, int fullscreen,
			     void (*resize_callback) (int width, int height,
						      int fullscreen))
{
  /* Information about the current video settings. */
  const SDL_VideoInfo *info = NULL;
  /* Color depth in bits of our window. */
  int bpp = 0;
  /* Flags we will pass into SDL_SetVideoMode. */
  int flags = 0;

  utils_context->resize_callback = resize_callback;

  /* Let's get some video information. */
  info = SDL_GetVideoInfo ();

  if (!info)
    {
      /* This should probably never happen. */
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		  _("SDL GetVideoInfo failed: \"%s\""), SDL_GetError ());
      fflush (stderr);
      return 0;
    }

  bpp = info->vfmt->BitsPerPixel;

  /*
   * Now, we want to setup our requested
   * window attributes for our OpenGL window.
   * We want *at least* 5 bits of red, green
   * and blue. We also want at least a 16-bit
   * depth buffer.
   *
   * The last thing we do is request a double
   * buffered window. '1' turns on double
   * buffering, '0' turns it off.
   *
   * Note that we do not use SDL_DOUBLEBUF in
   * the flags to SDL_SetVideoMode. That does
   * not affect the GL attribute state, only
   * the standard 2D blitting setup.
   */
  SDL_GL_SetAttribute (SDL_GL_RED_SIZE, 5);
  SDL_GL_SetAttribute (SDL_GL_GREEN_SIZE, 5);
  SDL_GL_SetAttribute (SDL_GL_BLUE_SIZE, 5);
  SDL_GL_SetAttribute (SDL_GL_DEPTH_SIZE, 16);
  SDL_GL_SetAttribute (SDL_GL_DOUBLEBUFFER, 1);

  flags = SDL_OPENGL | (fullscreen ? SDL_FULLSCREEN : SDL_RESIZABLE);

  /*
   * Set the video mode
   */
  if (SDL_SetVideoMode (width, height, bpp, flags) == 0)
    {
      /* 
       * This could happen for a variety of reasons,
       * including DISPLAY not being set, the specified
       * resolution not being available, etc.
       */
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		  _("SDL SetVideoMode %dx%d failed: \"%s\""), width,
		  height, SDL_GetError ());
      fflush (stderr);
      return 0;
    }
  else
    {
      glViewport (0, 0, width, height);	// required by some drivers
      lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("SDL SetVideoMode %dx%d"),
		  width, height);
      utils_context->screen.width = width;
      utils_context->screen.height = height;
      utils_context->screen.bpp = bpp;
      utils_context->screen.fullscreen = fullscreen;

      glGetIntegerv (GL_MAX_TEXTURE_SIZE,
		     &(utils_context->caps.max_texture_size));
      if (!utils_context->caps.max_texture_size)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		      _("unable to get MAX_TEXTURE_SIZE"));
	  utils_context->caps.max_texture_size = 512;
	}
    }

  SDL_WM_SetCaption (PACKAGE_STRING, PACKAGE_TARNAME);

  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("%d BPP"),
	      SDL_GetVideoSurface ()->format->BitsPerPixel);
  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("OpenGL vendor \"%s\""),
	      glGetString (GL_VENDOR));
  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("OpenGL renderer \"%s\""),
	      glGetString (GL_RENDERER));
  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("OpenGL version \"%s\""),
	      glGetString (GL_VERSION));

  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl",
	      _("OpenGL MAX_TEXTURE_SIZE = %d"),
	      utils_context->caps.max_texture_size);

  if (utils_context->caps.max_texture_size <
      MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _
		  ("OpenGL driver only handles %dx%d textures when %dx%d textures are needed"),
		  utils_context->caps.max_texture_size,
		  utils_context->caps.max_texture_size,
		  MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE,
		  MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE);
    }

  return 1;
}


/*
 * Called whenever window resize is asked for.
 */
int
mod_gl_utils_resize_video_mode (MOD_GL_UTILS_CONTEXT * utils_context,
				int width, int height)
{
  int ret = 0;

  if (width != utils_context->screen.width
      || height != utils_context->screen.height)
    {
      int flags = 0;

      flags =
	SDL_OPENGL | (utils_context->screen.
		      fullscreen ? SDL_FULLSCREEN : SDL_RESIZABLE);

      if (SDL_SetVideoMode (width, height, utils_context->screen.bpp, flags)
	  == 0)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		      _("unable to resize screen %dx%d"), width, height);
	}
      else
	{
	  glViewport (0, 0, width, height);
	  utils_context->screen.width = width;
	  utils_context->screen.height = height;

	  if (utils_context->resize_callback)
	    {
	      utils_context->resize_callback (width,
					      height,
					      utils_context->screen.
					      fullscreen);
	    }

	  lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl",
		      _("resizing screen %dx%d"), width, height);

	  ret = 1;
	}
    }
  else
    {
      /*
       * Nothing to do...
       */
      ret = 1;
    }

  return ret;
}
