/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <GL/glu.h>

#include "config.h"
#include "../../../gfx.h"
#include "gl-flat.h"
#include "gl-flat-internal.h"

static void
set_map_options (MOD_GL_UTILS_CONTEXT * utils_context,
		 _MOD_GL_VIEW_FLAT_CONTEXT * flat_context)
{
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
}

static void
display_texture_ex (MOD_GL_UTILS_CONTEXT * utils_context,
		    _MOD_GL_VIEW_FLAT_CONTEXT * flat_context,
		    GLuint texture, float x1, float y1, float x2, float y2,
		    float scale_w, float scale_h)
{
  glBindTexture (GL_TEXTURE_2D, texture);
  set_map_options (utils_context, flat_context);

  glMatrixMode (GL_TEXTURE);
  glPushMatrix ();
  glLoadIdentity ();
  glBegin (GL_QUADS);
  glTexCoord2d (0.0f, 0.0f);
  glVertex3f (x1, y1, 0.0f);	// top left
  glTexCoord2d (scale_w, 0.0f);
  glVertex3f (x2, y1, 0.0f);	// top right
  glTexCoord2d (scale_w, scale_h);
  glVertex3f (x2, y2, 0.0f);	// bottom right
  glTexCoord2d (0.0f, scale_h);
  glVertex3f (x1, y2, 0.0f);	// bottom left
  glEnd ();

  glMatrixMode (GL_TEXTURE);
  glPopMatrix ();
}

static void
display_texture (MOD_GL_UTILS_CONTEXT * utils_context,
		 _MOD_GL_VIEW_FLAT_CONTEXT * flat_context,
		 int texture_w, int texture_h, int shape_w, int shape_h,
		 GLuint texture, LW6MAP_PARAM * map_param)
{
  float x1, y1, x2, y2, scale_w, scale_h;

  scale_w = mod_gl_utils_texture_scale (texture_w);
  scale_h = mod_gl_utils_texture_scale (texture_h);

  mod_gl_utils_viewport_calc_map_area (utils_context, &x1, &y1, &x2, &y2,
				       shape_w, shape_h, map_param);

  display_texture_ex (utils_context,
		      flat_context,
		      texture, x1, y1, x2, y2, scale_w, scale_h);
}

static void
display_texture_array (MOD_GL_UTILS_CONTEXT * utils_context,
		       _MOD_GL_VIEW_FLAT_CONTEXT * flat_context,
		       int texture_array_w,
		       int texture_array_h,
		       int shape_w,
		       int shape_h,
		       MOD_GL_UTILS_TEXTURE_ARRAY * texture_array,
		       LW6MAP_PARAM * param)
{
  float x1, y1, x2, y2, w, h, coef_x, coef_y;
  float scale_w, scale_h;
  int n_x, n_y;
  float *array_x = NULL;
  float *array_y = NULL;
  int view_x, view_y, view_w, view_h;
  float min_x, min_y, max_x, max_y;

  mod_gl_utils_viewport_calc_map_area (utils_context, &x1, &y1, &x2, &y2,
				       shape_w, shape_h, param);
  w = x2 - x1;
  h = y2 - y1;
  coef_x = w / texture_array_w;
  coef_y = h / texture_array_h;

  mod_gl_utils_viewport_calc_view_area (utils_context, &view_x, &view_y,
					&view_w, &view_h);
  min_x = view_x;
  min_y = view_y;
  max_x = view_x + view_w;
  max_y = view_y + view_h;

  array_x = LW6SYS_MALLOC ((texture_array->layout.n_w + 1) * sizeof (float));
  array_y = LW6SYS_MALLOC ((texture_array->layout.n_h + 1) * sizeof (float));

  if (array_x && array_y)
    {
      if (texture_array->layout.n_w > 0)
	{
	  for (n_x = 0; n_x < texture_array->layout.n_w; ++n_x)
	    {
	      array_x[n_x] = x1 + texture_array->layout.x0[n_x] * coef_x;
	    }
	  array_x[texture_array->layout.n_w] =
	    array_x[texture_array->layout.n_w - 1] +
	    texture_array->layout.w[texture_array->layout.n_w - 1] * coef_x;
	}

      if (texture_array->layout.n_h > 0)
	{
	  for (n_y = 0; n_y < texture_array->layout.n_h; ++n_y)
	    {
	      array_y[n_y] = y1 + texture_array->layout.y0[n_y] * coef_y;
	    }
	  array_y[texture_array->layout.n_h] =
	    array_y[texture_array->layout.n_h - 1] +
	    texture_array->layout.h[texture_array->layout.n_h - 1] * coef_y;
	}

      for (n_y = 0; n_y < texture_array->layout.n_h; ++n_y)
	{
	  for (n_x = 0; n_x < texture_array->layout.n_w; ++n_x)
	    {
	      if (array_x[n_x] > max_x ||
		  array_y[n_y] > max_y ||
		  array_x[n_x + 1] < min_x || array_y[n_y + 1] < min_y)
		{
		  // skip texture, it's outside our screen...
		}
	      else
		{
		  scale_w =
		    mod_gl_utils_texture_scale (texture_array->layout.w[n_x]);
		  scale_h =
		    mod_gl_utils_texture_scale (texture_array->layout.h[n_y]);
		  display_texture_ex (utils_context,
				      flat_context,
				      mod_gl_utils_get_texture_array_texture
				      (texture_array, n_x, n_y),
				      array_x[n_x],
				      array_y[n_y],
				      array_x[n_x + 1],
				      array_y[n_y + 1], scale_w, scale_h);
		}
	    }
	}
      LW6SYS_FREE (array_x);
      LW6SYS_FREE (array_y);
    }
}

static void
display_map_preview (MOD_GL_UTILS_CONTEXT * utils_context,
		     _MOD_GL_VIEW_FLAT_CONTEXT * flat_context,
		     LW6MAP_MAP * map)
{
  display_texture_array (utils_context, flat_context,
			 flat_context->game_context.map.map->texture.shape.w,
			 flat_context->game_context.map.map->
			 texture.shape.h,
			 flat_context->game_context.map.map->
			 texture.shape.w,
			 flat_context->game_context.map.map->
			 texture.shape.h,
			 &(flat_context->game_context.map.map_texture_array),
			 &(map->param));
}

void
_mod_gl_view_flat_display_preview (MOD_GL_UTILS_CONTEXT * utils_context,
				   _MOD_GL_VIEW_FLAT_CONTEXT *
				   flat_context, LW6MAP_MAP * map)
{
  if (utils_context && flat_context && map && map->texture.data)
    {
      _mod_gl_view_flat_game_context_update_map (utils_context,
						 flat_context,
						 &(flat_context->
						   game_context.map), map);

      mod_gl_utils_set_render_mode_2d_blend (utils_context);

      glColor3f (1.0, 1.0, 1.0);
      glEnable (GL_TEXTURE_2D);	// for texture

      display_map_preview (utils_context, flat_context, map);
    }
}

void
mod_gl_view_flat_display_preview (MOD_GL_UTILS_CONTEXT * utils_context,
				  void *flat_context, LW6MAP_MAP * map)
{
  _mod_gl_view_flat_display_preview (utils_context,
				     (_MOD_GL_VIEW_FLAT_CONTEXT *)
				     flat_context, map);
}

static void
display_map_zones (MOD_GL_UTILS_CONTEXT * utils_context,
		   _MOD_GL_VIEW_FLAT_CONTEXT * flat_context,
		   LW6MAP_MAP * map, LW6KER_MAP_STRUCT * map_struct)
{
  GLuint texture = 0;
  SDL_Surface *surface = NULL;

  surface = mod_gl_utils_create_zones_surface (utils_context, map_struct);
  if (surface)
    {
      texture = mod_gl_utils_surface2texture (utils_context, surface);

      if (texture)
	{
	  display_texture (utils_context, flat_context, map_struct->shape.w,
			   map_struct->shape.h, map->texture.shape.w,
			   map->texture.shape.h, texture, &(map->param));

	  mod_gl_utils_schedule_delete_texture (utils_context, texture);
	}

      mod_gl_utils_delete_surface (utils_context, surface);
    }
}

void
_mod_gl_view_flat_display_zones (MOD_GL_UTILS_CONTEXT * utils_context,
				 _MOD_GL_VIEW_FLAT_CONTEXT *
				 flat_context, LW6MAP_MAP * map,
				 LW6KER_MAP_STRUCT * map_struct)
{
  if (utils_context && flat_context && map && map_struct)
    {
      //_mod_gl_set_render_mode_2d (context);
      mod_gl_utils_set_render_mode_2d_blend (utils_context);

      glColor3f (1.0, 1.0, 1.0);
      glEnable (GL_TEXTURE_2D);	// for texture

      display_map_zones (utils_context, flat_context, map, map_struct);
    }
}

void
mod_gl_view_flat_display_zones (MOD_GL_UTILS_CONTEXT * utils_context,
				void *flat_context, LW6MAP_MAP * map,
				LW6KER_MAP_STRUCT * map_struct)
{
  _mod_gl_view_flat_display_zones (utils_context,
				   (_MOD_GL_VIEW_FLAT_CONTEXT *)
				   flat_context, map, map_struct);
}

static void
display_map_gradient (MOD_GL_UTILS_CONTEXT * utils_context,
		      _MOD_GL_VIEW_FLAT_CONTEXT * flat_context,
		      LW6MAP_MAP * map,
		      LW6KER_MAP_STATE * map_state, int team_id)
{
  GLuint texture = 0;
  SDL_Surface *surface = NULL;

  surface =
    mod_gl_utils_create_gradient_surface (utils_context, map_state, team_id);
  if (surface)
    {
      texture = mod_gl_utils_surface2texture (utils_context, surface);

      if (texture)
	{
	  display_texture (utils_context, flat_context, map_state->shape.w,
			   map_state->shape.h, map->texture.shape.w,
			   map->texture.shape.h, texture, &(map->param));

	  mod_gl_utils_schedule_delete_texture (utils_context, texture);
	}

      mod_gl_utils_delete_surface (utils_context, surface);
    }
}

void
_mod_gl_view_flat_display_gradient (MOD_GL_UTILS_CONTEXT * utils_context,
				    _MOD_GL_VIEW_FLAT_CONTEXT *
				    flat_context,
				    LW6MAP_MAP * map,
				    LW6KER_MAP_STATE * map_state, int team_id)
{
  if (utils_context && flat_context && map_state)
    {
      mod_gl_utils_set_render_mode_2d_blend (utils_context);

      glColor3f (1.0, 1.0, 1.0);
      glEnable (GL_TEXTURE_2D);	// for texture

      display_map_gradient (utils_context, flat_context, map, map_state,
			    team_id);
    }
}

void
mod_gl_view_flat_display_gradient (MOD_GL_UTILS_CONTEXT * utils_context,
				   void *flat_context,
				   LW6MAP_MAP * map,
				   LW6KER_MAP_STATE * map_state, int team_id)
{
  _mod_gl_view_flat_display_gradient (utils_context,
				      (_MOD_GL_VIEW_FLAT_CONTEXT *)
				      flat_context, map, map_state, team_id);
}

/*
 * Game_state is not passed as args but retrived
 * from flat_context struct, not to give the illusion that
 * any game_state could be displayed. One *needs* to call
 * game_context_init first.
 */
static void
display_armies (MOD_GL_UTILS_CONTEXT * utils_context,
		_MOD_GL_VIEW_FLAT_CONTEXT *
		flat_context, LW6MAP_PARAM * map_param,
		LW6GFX_GAME_LOOK * game_look)
{
  int i;

  for (i = 0; i < flat_context->game_context.armies.game_state->map.nb_layers;
       ++i)
    {
      mod_gl_utils_update_game_texture_array (utils_context,
					      &(flat_context->game_context.
						armies.armies_surface_array),
					      &(flat_context->game_context.
						armies.
						armies_texture_arrays[i]),
					      flat_context->game_context.
					      armies.game_state, map_param,
					      game_look, i);
      display_texture_array (utils_context, flat_context,
			     flat_context->game_context.armies.game_state->
			     map.shape.w,
			     flat_context->game_context.armies.game_state->
			     map.shape.h,
			     flat_context->game_context.map.map->texture.
			     shape.w,
			     flat_context->game_context.map.map->texture.
			     shape.h,
			     &(flat_context->game_context.armies.
			       armies_texture_arrays[i]), map_param);
    }
}

/*
 * Game_state is not passed as args but retrived
 * from flat_context struct, not to give the illusion that
 * any game_state could be displayed. One *needs* to call
 * game_context_init first.
 */
static void
display_game (MOD_GL_UTILS_CONTEXT * utils_context,
	      _MOD_GL_VIEW_FLAT_CONTEXT *
	      flat_context, LW6MAP_MAP * map, LW6GFX_GAME_LOOK * game_look)
{
  display_map_preview (utils_context, flat_context, map);
  display_armies (utils_context, flat_context, &(map->param), game_look);
}

void
_mod_gl_view_flat_display_game (MOD_GL_UTILS_CONTEXT * utils_context,
				_MOD_GL_VIEW_FLAT_CONTEXT *
				flat_context,
				LW6MAP_MAP * map,
				LW6KER_GAME_STATE * game_state,
				LW6GFX_GAME_LOOK * game_look)
{
  if (utils_context && flat_context && map && game_state)
    {
      _mod_gl_view_flat_game_context_update (utils_context,
					     flat_context,
					     &flat_context->
					     game_context, map, game_state,
					     game_look);

      mod_gl_utils_set_render_mode_2d_blend (utils_context);

      glColor3f (1.0, 1.0, 1.0);
      glEnable (GL_TEXTURE_2D);	// for texture

      display_game (utils_context, flat_context, map, game_look);
    }
}

void
mod_gl_view_flat_display_game (MOD_GL_UTILS_CONTEXT * utils_context,
			       void *flat_context,
			       LW6MAP_MAP * map,
			       LW6KER_GAME_STATE * game_state,
			       LW6GFX_GAME_LOOK * game_look)
{
  _mod_gl_view_flat_display_game (utils_context,
				  (_MOD_GL_VIEW_FLAT_CONTEXT *)
				  flat_context, map, game_state, game_look);
}
