/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "config.h"
#include "ker.h"
#include "ker-internal.h"

/*
 * The code in this file is heavily inspired from Liquid War 5,
 * it's an adaptation of "src/mesh.c".
 */

/*
 * Internal structure used just when creating the mesh.
 * The idea is to have w*h zones, of size 1x1. Of course
 * these will be grouped afterwards, but when creating the
 * zones, it's usefull to have this in memory.
 * This was called "MESHER" in Liquid War 5.
 */
typedef struct DRAFT_ZONE_STRUCT
{
  LW6SYS_XY pos;
  LW6SYS_INT32 used;
  LW6SYS_INT32 size;
  LW6SYS_INT32 link[LW6KER_NB_DIRS];
  LW6SYS_INT32 corres;
}
DRAFT_ZONE;

typedef struct DRAFT_ZONES_STRUCT
{
  LW6SYS_WH shape;
  LW6SYS_INT32 max_zone_size;
  DRAFT_ZONE *zones;
}
DRAFT_ZONES;

void
_lw6ker_map_struct_update_checksum (LW6KER_MAP_STRUCT *
				    map_struct, LW6SYS_UINT32 * checksum)
{
  int i;

  lw6sys_checksum_update_wh (checksum, &(map_struct->shape));
  lw6sys_checksum_update_int32 (checksum, map_struct->nb_zones);
  lw6sys_checksum_update_int32 (checksum, map_struct->nb_slots);
  lw6sys_checksum_update_int32 (checksum, map_struct->nb_usable_slots);
  lw6sys_checksum_update_int32 (checksum, map_struct->room_for_armies);
  lw6sys_checksum_update_int32 (checksum, map_struct->max_depth);
  lw6sys_checksum_update_int32 (checksum, map_struct->max_zone_size);
  for (i = 0; i < map_struct->nb_zones; ++i)
    {
      _lw6ker_zone_struct_update_checksum (&(map_struct->zones[i]), checksum);
    }
  for (i = 0; i < map_struct->nb_slots; ++i)
    {
      _lw6ker_slot_struct_update_checksum (&(map_struct->slots[i]), checksum);
    }
}

static LW6SYS_INT32
relative_index (LW6MAP_MAP * map, LW6SYS_INT32 x, LW6SYS_INT32 y)
{
  LW6SYS_INT32 w, h;

  w = map->depth.shape.w;
  h = map->depth.shape.h;

  if (x >= w)
    {
      switch (map->param.options.x_polarity)
	{
	case 1:
	  x = 0;
	  break;
	case -1:
	  x = 0;
	  y = h - 1 - y;
	  break;
	default:
	  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		      _("x too great (%d/%d) for a non wrapped map"), x, w);
	  break;
	}
    }

  if (y >= h)
    {
      switch (map->param.options.y_polarity)
	{
	case 1:
	  y = 0;
	  break;
	case -1:
	  x = w - 1 - x;
	  y = 0;
	  break;
	default:
	  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		      _("y too great (%d/%d) for a non wrapped map"), y, h);
	  break;
	}
    }

  if (x < 0)
    {
      switch (map->param.options.x_polarity)
	{
	case 1:
	  x = w - 1;
	  break;
	case -1:
	  x = w - 1;
	  y = h - 1 - y;
	  break;
	default:
	  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		      _("x too small (%d) for a non wrapped map"), x);
	  break;
	}
    }

  if (y < 0)
    {
      switch (map->param.options.y_polarity)
	{
	case 1:
	  y = h - 1;
	  break;
	case -1:
	  x = w - 1 - x;
	  y = h - 1;
	  break;
	default:
	  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		      _("y too small (%d) for a non wrapped map"), y);
	  break;
	}
    }

  return y * w + x;
}

/*
 * Creates a new DRAFT_ZONES object. The idea is to create a structure which
 * is of the same nature as the final MAP_STRUCT "zones", but unoptimized.
 * We create a structure (a "MESH" in Liquid War 5)) where each square
 * has pointers on its neighbours. As there are 12 directions, these is
 * very redundant. This is just a temporary struct which will be
 * optimized afterwards.
 */
static DRAFT_ZONES *
draft_zones_new (LW6MAP_MAP * map)
{
  DRAFT_ZONES *ret;

  ret = (DRAFT_ZONES *) LW6SYS_CALLOC (sizeof (DRAFT_ZONES));
  if (ret)
    {
      LW6SYS_INT32 w, h, size;

      ret->shape = map->depth.shape;
      ret->max_zone_size = 1;

      w = ret->shape.w;
      h = ret->shape.h;
      size = w * h;

      if (size >= map->param.options.min_map_surface)
	{
	  ret->zones =
	    (DRAFT_ZONE *) LW6SYS_CALLOC (size * sizeof (DRAFT_ZONE));

	  if (ret->zones)
	    {
	      LW6SYS_INT32 i, j, x, y, x1, x2, y1, y2;

	      for (i = 0; i < size; ++i)
		{
		  ret->zones[i].size = 1;
		  for (j = 0; j < LW6KER_NB_DIRS; ++j)
		    {
		      ret->zones[i].link[j] = -1;
		    }
		}

	      for (y = 0; y < h; ++y)
		{
		  for (x = 0; x < w; ++x)
		    {
		      i = y * w + x;

		      ret->zones[i].pos.x = x;
		      ret->zones[i].pos.y = y;
		      ret->zones[i].used =
			lw6map_depth_get (&(map->depth), x, y) ? 1 : 0;
		    }
		}

	      x1 = 0;
	      x2 = w;
	      y1 = 0;
	      y2 = h;

	      if (map->param.options.x_polarity != 1
		  && map->param.options.x_polarity != -1)
		{
		  x1++;
		  x2--;
		}

	      if (map->param.options.y_polarity != 1
		  && map->param.options.y_polarity != -1)
		{
		  y1++;
		  y2--;
		}

	      for (y = y1; y < y2; ++y)
		{
		  for (x = x1; x < x2; ++x)
		    {
		      i = relative_index (map, x, y);

		      if (ret->zones[i].used)
			{
			  j = relative_index (map, x, y - 1);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_NNW] =
				ret->zones[i].link[LW6KER_DIR_NNE] = j;
			    }
			  j = relative_index (map, x + 1, y - 1);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_NE] = j;
			    }
			  j = relative_index (map, x + 1, y);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_ENE] =
				ret->zones[i].link[LW6KER_DIR_ESE] = j;
			    }
			  j = relative_index (map, x + 1, y + 1);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_SE] = j;
			    }
			  j = relative_index (map, x, y + 1);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_SSE] =
				ret->zones[i].link[LW6KER_DIR_SSW] = j;
			    }
			  j = relative_index (map, x - 1, y + 1);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_SW] = j;
			    }
			  j = relative_index (map, x - 1, y);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_WSW] =
				ret->zones[i].link[LW6KER_DIR_WNW] = j;
			    }
			  j = relative_index (map, x - 1, y - 1);
			  if (ret->zones[j].used)
			    {
			      ret->zones[i].link[LW6KER_DIR_NW] = j;
			    }
			}
		    }
		}
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
			  _("unable to allocate %d draft zones"), size);
	    }
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		      _("map is too small, size=%d, MIN_MAP_SIZE=%d"),
		      size, map->param.options.min_map_surface);
	}
    }

  return ret;
}

static void
draft_zones_free (DRAFT_ZONES * draft_zones)
{
  if (draft_zones)
    {
      if (draft_zones->zones)
	{
	  LW6SYS_FREE (draft_zones->zones);
	}
      LW6SYS_FREE (draft_zones);
    }
}

/*
 * Here we scan the whole DRAFT_ZONES data, and whenever we find
 * 4 zones which form a square and respect "some conditions",
 * we group them into a bigger zone. The only thing to be carefull
 * about is to handle correctly all the pointers (links) on
 * other zones, especially those that link back to the squares
 * we suppress.
 */
static LW6SYS_INT32
draft_zones_group (DRAFT_ZONES * draft_zones, LW6SYS_INT32 step)
{
  LW6SYS_INT32 found = 0;
  LW6SYS_INT32 x, y, w, h, size, i, j, k;
  LW6SYS_INT32 i_ne, i_se, i_sw, i_nw;
  DRAFT_ZONE *zone_ne, *zone_se, *zone_sw, *zone_nw;
  LW6SYS_INT32 *test;

  w = draft_zones->shape.w;
  h = draft_zones->shape.h;
  size = h * w;

  for (y = 0; y < h - step; y += step * 2)
    {
      for (x = 0; x < w - step; x += step * 2)
	{
	  i = y * w + x;
	  i_ne = i + step;
	  i_se = i + (w + 1) * step;
	  i_sw = i + w * step;
	  i_nw = i;
	  zone_ne = &draft_zones->zones[i_ne];
	  zone_se = &draft_zones->zones[i_se];
	  zone_sw = &draft_zones->zones[i_sw];
	  zone_nw = &draft_zones->zones[i_nw];

	  /*
	   * This test contains all the criteria which make
	   * us decide wether we group the squares are not.
	   * It's a wise choice to be rather restrictive.
	   * For instance the fact tha NE SW NW NE links
	   * actually point to something might seem useless.
	   * It's not, the main reason is that it's a rather
	   * good choice that whenever there's an obstacle,
	   * the zones around it are forced to small.
	   */
	  if (zone_ne->used && zone_ne->size == step
	      && zone_se->used && zone_se->size == step
	      && zone_sw->used && zone_sw->size == step
	      && zone_nw->used && zone_nw->size == step
	      && zone_ne->link[LW6KER_DIR_NNW] ==
	      zone_ne->link[LW6KER_DIR_NNE]
	      && zone_ne->link[LW6KER_DIR_ENE] ==
	      zone_ne->link[LW6KER_DIR_ESE]
	      && zone_se->link[LW6KER_DIR_ENE] ==
	      zone_se->link[LW6KER_DIR_ESE]
	      && zone_se->link[LW6KER_DIR_SSE] ==
	      zone_se->link[LW6KER_DIR_SSW]
	      && zone_sw->link[LW6KER_DIR_SSE] ==
	      zone_sw->link[LW6KER_DIR_SSW]
	      && zone_sw->link[LW6KER_DIR_WSW] ==
	      zone_sw->link[LW6KER_DIR_WNW]
	      && zone_nw->link[LW6KER_DIR_WSW] ==
	      zone_nw->link[LW6KER_DIR_WNW]
	      && zone_nw->link[LW6KER_DIR_NNW] ==
	      zone_nw->link[LW6KER_DIR_NNE]
	      && zone_ne->link[LW6KER_DIR_NE] >= 0
	      && zone_se->link[LW6KER_DIR_SE] >= 0
	      && zone_sw->link[LW6KER_DIR_SW] >= 0
	      && zone_nw->link[LW6KER_DIR_NW] >= 0)
	    {
	      /*
	       * Delete unused zones
	       */
	      zone_ne->used = 0;
	      zone_se->used = 0;
	      zone_sw->used = 0;

	      /*
	       * Double the zone size of the remaining zone
	       */
	      zone_nw->size = step * 2;

	      /*
	       * Change the links of the remaining zone
	       */
	      zone_nw->link[LW6KER_DIR_NNE] = zone_ne->link[LW6KER_DIR_NNE];
	      zone_nw->link[LW6KER_DIR_NE] = zone_ne->link[LW6KER_DIR_NE];
	      zone_nw->link[LW6KER_DIR_ENE] = zone_ne->link[LW6KER_DIR_ENE];
	      zone_nw->link[LW6KER_DIR_ESE] = zone_se->link[LW6KER_DIR_ESE];
	      zone_nw->link[LW6KER_DIR_SE] = zone_se->link[LW6KER_DIR_SE];
	      zone_nw->link[LW6KER_DIR_SSE] = zone_se->link[LW6KER_DIR_SSE];
	      zone_nw->link[LW6KER_DIR_SSW] = zone_sw->link[LW6KER_DIR_SSW];
	      zone_nw->link[LW6KER_DIR_SW] = zone_sw->link[LW6KER_DIR_SW];
	      zone_nw->link[LW6KER_DIR_WSW] = zone_sw->link[LW6KER_DIR_WSW];

	      /*
	       * Change all the links of the adjacent zones,
	       * which were pointing on the deleted zones,
	       * and must now point on the remaining one.
	       */
	      for (j = 0; j < LW6KER_NB_DIRS; ++j)
		{
		  for (k = 0; k < LW6KER_NB_DIRS; ++k)
		    {
		      if (zone_nw->link[j] >= 0)
			{
			  test =
			    &(draft_zones->zones[zone_nw->link[j]].link[k]);
			  if (*test == i_ne || *test == i_se || *test == i_sw)
			    {
			      *test = i_nw;
			    }
			}
		    }
		}

	      draft_zones->max_zone_size = zone_nw->size;
	      ++found;
	    }
	}
    }

  return found;
}

/*
 * Transform the temporary DRAFT_ZONES into the proper
 * final MAP_STRUCT. The major change is to actually
 * delete and remove from memory all the unused ZONES.
 * We "compact" the zones so that instead of using
 * tons of RAM, we use just what we need. While this
 * complicates a little bit the algorithms (we can't
 * assume any more than y * w + x = i) reducing memory
 * usage is always a good thing, the more we reduce it,
 * the more we get a chance to have things cached.
 */
static LW6SYS_BOOL
draft_zones_to_map_struct (LW6KER_MAP_STRUCT * map_struct,
			   DRAFT_ZONES * draft_zones)
{
  LW6SYS_BOOL ret = 0;
  LW6SYS_INT32 w, h, size, nb_zones;
  LW6SYS_INT32 i, j, k, temp;

  w = draft_zones->shape.w;
  h = draft_zones->shape.h;
  size = w * h;
  nb_zones = 0;

  for (i = 0; i < size; ++i)
    {
      if (draft_zones->zones[i].used)
	{
	  ++nb_zones;
	}
    }

  map_struct->shape.w = w;
  map_struct->shape.h = h;
  map_struct->nb_slots = w * h;
  map_struct->nb_zones = nb_zones;
  map_struct->max_zone_size = draft_zones->max_zone_size;

  map_struct->zones =
    (LW6KER_ZONE_STRUCT *) LW6SYS_MALLOC (nb_zones *
					  sizeof (LW6KER_ZONE_STRUCT));

  if (map_struct->zones)
    {
      j = 0;
      for (i = 0; i < size; ++i)
	{
	  if (draft_zones->zones[i].used)
	    {
	      map_struct->zones[j].pos = draft_zones->zones[i].pos;
	      map_struct->zones[j].size = draft_zones->zones[i].size;
	      for (k = 0; k < LW6KER_NB_DIRS; ++k)
		{
		  map_struct->zones[j].link[k] =
		    draft_zones->zones[i].link[k];
		}
	      draft_zones->zones[i].corres = j;

	      j++;
	    }
	}

      for (j = 0; j < nb_zones; ++j)
	{
	  for (k = 0; k < LW6KER_NB_DIRS; ++k)
	    {
	      if ((temp = (map_struct->zones[j].link[k])) >= 0)
		{
		  map_struct->zones[j].link[k] =
		    draft_zones->zones[temp].corres;
		}
	    }
	}
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		  _("unable to create zones for map struct"));
    }

  return ret;
}

/*
 * Create the slots, the main interest in doing this is to
 * have a quick way to get the right zone for a given (x,y).
 */
static LW6SYS_BOOL
init_slots (LW6KER_MAP_STRUCT * map_struct, LW6MAP_MAP * map)
{
  LW6SYS_BOOL ret = 0;

  map_struct->slots =
    (LW6KER_SLOT_STRUCT *) LW6SYS_CALLOC (map_struct->shape.w *
					  map_struct->shape.h *
					  sizeof (LW6KER_SLOT_STRUCT));

  if (map_struct)
    {
      LW6SYS_INT32 i, x, y;
      LW6SYS_INT32 depth;

      LW6KER_ZONE_STRUCT *zone;

      /*
       * Set up the id lookup table
       */
      for (y = 0; y < map_struct->shape.h; ++y)
	{
	  for (x = 0; x < map_struct->shape.w; ++x)
	    {
	      lw6ker_map_struct_set_zone_id (map_struct, x, y, -1);
	    }
	}

      for (i = 0; i < map_struct->nb_zones; ++i)
	{
	  zone = &map_struct->zones[i];
	  for (y = 0; y < map_struct->zones[i].size; ++y)
	    {
	      for (x = 0; x < map_struct->zones[i].size; ++x)
		{
		  lw6ker_map_struct_set_zone_id (map_struct, zone->pos.x + x,
						 zone->pos.y + y, i);
		  map_struct->nb_usable_slots++;
		}
	    }
	}

      /*
       * Copy depth information from the map.
       */
      map_struct->max_depth = 0;
      for (y = 0; y < map_struct->shape.h; ++y)
	{
	  for (x = 0; x < map_struct->shape.w; ++x)
	    {
	      depth = lw6map_depth_get (&(map->depth), x, y);
	      lw6ker_map_struct_set_depth (map_struct, x, y, depth);
	      map_struct->room_for_armies += depth;
	      if (map_struct->max_depth < depth)
		{
		  map_struct->max_depth = depth;
		}
	    }
	}

      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		  _("unable to create slots for map struct"));
    }

  return ret;
}

/*
 * Returns a "useless" ratio which is "how much we compressed
 * the map by using all this ZONE wizardry. Just to keep track
 * of wether it's usefull to implement this. Debugging tool.
 */
float
_lw6ker_map_struct_get_compression (LW6KER_MAP_STRUCT * map_struct)
{
  float ret;

  ret =
    ((float) map_struct->nb_zones) /
    ((float) (map_struct->shape.w * map_struct->shape.h));

  return ret;
}

/*
 * This is one key function: it calls the transformation of a LW6MAP_MAP,
 * that is a plain object which reflects only the content of PNG & XML
 * files, into an in-memory "ready for algorithmic stuff" structure.
 */
LW6SYS_BOOL
_lw6ker_map_struct_init (LW6KER_MAP_STRUCT * map_struct, LW6MAP_MAP * map)
{
  LW6SYS_BOOL ret = 0;
  DRAFT_ZONES *draft_zones;

  memset (map_struct, 0, sizeof (LW6KER_MAP_STRUCT));

  draft_zones = draft_zones_new (map);
  if (draft_zones)
    {
      LW6SYS_INT32 i = 1;

      ret = 1;

      while (i <= map->param.options.max_zone_size / 2
	     && draft_zones_group (draft_zones, i))
	{
	  i *= 2;
	}

      ret = ret && draft_zones_to_map_struct (map_struct, draft_zones);

      draft_zones_free (draft_zones);

      ret = ret && init_slots (map_struct, map);

      if (ret)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, "ker",
		      _
		      ("map struct created %dx%d, %d zones, %1.1f%% compression"),
		      map_struct->shape.w, map_struct->shape.h,
		      map_struct->nb_zones,
		      _lw6ker_map_struct_get_compression (map_struct) *
		      100.0f);
	}
    }

  return ret;
}

void
_lw6ker_map_struct_clear (LW6KER_MAP_STRUCT * map_struct)
{
  if (map_struct)
    {
      if (map_struct->zones)
	{
	  LW6SYS_FREE (map_struct->zones);
	}
      if (map_struct->slots)
	{
	  LW6SYS_FREE (map_struct->slots);
	}
      memset (map_struct, 0, sizeof (LW6KER_MAP_STRUCT));
    }
}

void
lw6ker_map_struct_find_free_slot_near (LW6KER_MAP_STRUCT * map_struct,
				       LW6SYS_XY * there, LW6SYS_XY here)
{
  LW6SYS_BOOL found = 0;

  there->x = there->y = 0;

  if (lw6ker_map_struct_get_depth (map_struct, here.x, here.y) > 0)
    {
      *there = here;
      found = 1;
    }
  else
    {
      LW6SYS_INT32 angle, radius;
      LW6SYS_INT32 max_radius, max_angle;
      LW6SYS_INT32 x, y;

      max_radius = map_struct->shape.w + map_struct->shape.h;	// +, not *
      for (radius = 1; radius < max_radius && !found; ++radius)
	{
	  max_angle = radius * M_PI * 2;
	  for (angle = 0; angle < max_angle && !found; ++angle)
	    {
	      x =
		here.x +
		(lw6ker_cos ((angle * LW6KER_TRIGO_2PI) / max_angle) *
		 radius) / LW6KER_TRIGO_RADIUS;
	      y =
		here.y -
		(lw6ker_sin ((angle * LW6KER_TRIGO_2PI) / max_angle) *
		 radius) / LW6KER_TRIGO_RADIUS;
	      if (x >= 0 && x < map_struct->shape.w && y >= 0
		  && y < map_struct->shape.h)
		{
		  if (lw6ker_map_struct_get_depth (map_struct, x, y) > 0)
		    {
		      found = 1;
		      there->x = x;
		      there->y = y;
		    }
		}
	    }
	}
    }

  if (!found)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		  _("unable to find a free zone near %dx%d"), here.x, here.y);
    }
}
