/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>

#include <libpng/png.h>

#include "config.h"
#include "map.h"
#include "map-internal.h"

/*
 * This is used to stamp maps as they are created.
 * Note that there's no race condition due to the fact that this
 * is global here, in fact even when 2 processes would share
 * this sequence id, it would not matter for they would then
 * try and identify the objects in their on per-process lists,
 * structures, Guile object, whatever they use.
 */
static int seq_id = 0;

/*
 * Creates a new empty map.
 */
LW6MAP_MAP *
lw6map_new ()
{
  LW6MAP_MAP *map = NULL;

  map = (LW6MAP_MAP *) LW6SYS_CALLOC (sizeof (LW6MAP_MAP));

  if (map)
    {
      map->id = ++seq_id;
    }

  return map;
}

/*
 * Frees a map and all its members.
 */
void
lw6map_free (LW6MAP_MAP * map)
{
  if (map)
    {
      lw6map_texture_clear (&map->texture);
      lw6map_elevation_clear (&map->elevation);
      lw6map_depth_clear (&map->depth);
      lw6map_param_clear (&map->param);
      lw6map_metadata_clear (&map->metadata);

      LW6SYS_FREE (map);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "map", _("trying to free NULL map"));
    }
}

/*
 * Frees a map and all its members.
 */
int
lw6map_memory_footprint (LW6MAP_MAP * map)
{
  int memory_footprint = 0;

  if (map)
    {
      memory_footprint += sizeof (LW6MAP_MAP);
      memory_footprint += strlen (map->metadata.title);
      memory_footprint += strlen (map->metadata.readme);
      memory_footprint +=
	map->depth.shape.w * map->depth.shape.h * sizeof (unsigned char);
      memory_footprint +=
	map->elevation.shape.w * map->elevation.shape.h *
	sizeof (signed char);
      memory_footprint +=
	map->texture.shape.w * map->texture.shape.h * sizeof (LW6SYS_COLOR_8);

    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "map",
		  _("calling memory_footprint on NULL map"));
    }

  return memory_footprint;
}

/*
 * Generate a unique string id/repr. Pointer must be freed.
 */
char *
lw6map_repr (LW6MAP_MAP * map)
{
  char *ret = NULL;

  if (map)
    {
      ret =
	lw6sys_new_sprintf ("%d \"%s\" (%dx%d)", map->id, map->metadata.title,
			    map->depth.shape.w, map->depth.shape.h);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "map",
		  _("can't generate string id for NULL map"));
    }

  return ret;
}
