/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6MAP_H
#define LIQUIDWAR6MAP_H

#include "../lw6-common.h"

#include "../sys/sys.h"
#include "../opt/opt.h"

#define LW6MAP_MAX_DEPTH 4	// can't be too high for perf issues

#define LW6MAP_STYLE_DEFAULT_KEEP_RATIO 1
#define LW6MAP_STYLE_DEFAULT_ZOOM 1.0f
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_STYLE "water"
#define LW6MAP_STYLE_DEFAULT_HUD_STYLE "floating"
#define LW6MAP_STYLE_DEFAULT_MENU_STYLE "cylinder"
#define LW6MAP_STYLE_DEFAULT_VIEW_STYLE "flat"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_RED "#f00"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_GREEN "#0f0"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_BLUE "#00f"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_YELLOW "#ff0"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_CYAN "#0ff"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_MAGENTA "#f0f"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_ORANGE "#f80"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_SKY "#8bf"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_PURPLE "#b8f"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_PINK "#f8b"

#define LW6MAP_FILE_MAP_PNG "map.png"
#define LW6MAP_FILE_TEXTURE_PNG "texture.png"
#define LW6MAP_FILE_TEXTURE_JPEG "texture.jpeg"
#define LW6MAP_FILE_TEXTURE_ALPHA_JPEG "texture-alpha.jpeg"

#define LW6MAP_NB_TEAM_COLORS LW6OPT_MAX_NB_TEAMS
typedef enum LW6MAP_TEAM_COLOR_ENUM
{
  LW6MAP_TEAM_COLOR_RED = 0,
  LW6MAP_TEAM_COLOR_GREEN = 1,
  LW6MAP_TEAM_COLOR_BLUE = 2,
  LW6MAP_TEAM_COLOR_YELLOW = 3,
  LW6MAP_TEAM_COLOR_CYAN = 4,
  LW6MAP_TEAM_COLOR_MAGENTA = 5,
  LW6MAP_TEAM_COLOR_ORANGE = 6,
  LW6MAP_TEAM_COLOR_SKY = 7,
  LW6MAP_TEAM_COLOR_PURPLE = 8,
  LW6MAP_TEAM_COLOR_PINK = 9
}
LW6MAP_TEAM_COLOR;

typedef struct LW6MAP_METADATA_STRUCT
{
  char *title;
  char *readme;
}
LW6MAP_METADATA;

typedef struct LW6MAP_DEPTH_STRUCT
{
  LW6SYS_WH shape;
  /*
   * Depth is used for game logic. 8 is the deepest. Would correspond
   * to elevation == -128 in fact. 8 fighters can fit on a slot/point
   * when depth is 8. The default depth (as of Liquid War 5) is 1, and
   * only one fighter can fit. 0 mean it's a wall. This 
   */
  unsigned char *data;
  int max;
}
LW6MAP_DEPTH;

typedef struct LW6MAP_ELEVATION_STRUCT
{
  LW6SYS_WH shape;
  /*
   * Elevation is an analogic parameter, for fancy display, it is
   * not linked to game logic, only used for visual enhancements.
   * It's the responsabibility of lw6map to ensure that this visual
   * information is consistent with the depth.
   */
  signed char *data;
  int min;
  int max;
}
LW6MAP_ELEVATION;

typedef struct LW6MAP_TEXTURE_STRUCT
{
  LW6SYS_WH shape;
  LW6SYS_COLOR_8 *data;
}
LW6MAP_TEXTURE;

typedef struct LW6MAP_COLOR_COUPLE_STRUCT
{
  LW6SYS_COLOR_F fg;
  LW6SYS_COLOR_F bg;
}
LW6MAP_COLOR_COUPLE;

typedef struct LW6MAP_STYLE_STRUCT
{
  LW6MAP_COLOR_COUPLE color;	// used only when no texture
  int keep_ratio;		// boolean
  float zoom;
  char *background_style;
  char *hud_style;
  char *menu_style;
  char *view_style;
  LW6SYS_COLOR_F team_color_red;
  LW6SYS_COLOR_F team_color_green;
  LW6SYS_COLOR_F team_color_blue;
  LW6SYS_COLOR_F team_color_yellow;
  LW6SYS_COLOR_F team_color_cyan;
  LW6SYS_COLOR_F team_color_magenta;
  LW6SYS_COLOR_F team_color_orange;
  LW6SYS_COLOR_F team_color_sky;
  LW6SYS_COLOR_F team_color_purple;
  LW6SYS_COLOR_F team_color_pink;
}
LW6MAP_STYLE;

typedef struct LW6MAP_PARAM_STRUCT
{
  LW6OPT_STATIC options;
  LW6MAP_STYLE style;
}
LW6MAP_PARAM;

typedef struct LW6MAP_MAP_STRUCT
{
  int id;
  LW6MAP_METADATA metadata;
  LW6MAP_DEPTH depth;
  LW6MAP_ELEVATION elevation;
  LW6MAP_TEXTURE texture;
  LW6MAP_PARAM param;
}
LW6MAP_MAP;

typedef struct LW6MAP_ENTRY_STRUCT
{
  char *title;
  char *path;
  int subdir;
}
LW6MAP_ENTRY;

typedef struct LW6MAP_RESAMPLER_STRUCT
{
  LW6SYS_WH target_shape;
  LW6SYS_WH source_shape;
  float scale_x;
  float scale_y;
}
LW6MAP_RESAMPLER;

/*
 * Simple setters/getters. Not so usefull since the purpose of 
 * having plain pointers on a raw buffer is exactly to bypass 
 * these kind of cumbersome functions 8-). In fact having the 
 * setter is a good way to 1) test and 2) find out quickly how 
 * data is stored.
 */
static inline void
lw6map_depth_set (LW6MAP_DEPTH * depth, int x, int y, unsigned char value)
{
  depth->data[y * depth->shape.w + x] = value;
}
static inline unsigned char
lw6map_depth_get (LW6MAP_DEPTH * depth, int x, int y)
{
  return depth->data[y * depth->shape.w + x];
}
static inline void
lw6map_elevation_set (LW6MAP_ELEVATION * elevation, int x, int y,
		      signed char value)
{
  elevation->data[y * elevation->shape.w + x] = value;
}
static inline signed char
lw6map_elevation_get (LW6MAP_ELEVATION * elevation, int x, int y)
{
  return elevation->data[y * elevation->shape.w + x];
}
static inline void
lw6map_texture_set (LW6MAP_TEXTURE * texture, int x, int y,
		    LW6SYS_COLOR_8 value)
{
  texture->data[y * texture->shape.w + x] = value;
}
static inline LW6SYS_COLOR_8
lw6map_texture_get (LW6MAP_TEXTURE * texture, int x, int y)
{
  return texture->data[y * texture->shape.w + x];
}

/*
 * In depth.c
 */
extern int lw6map_depth_read (LW6MAP_DEPTH * depth, char *dirname,
			      LW6OPT_STATIC * static_options);
extern void lw6map_depth_clear (LW6MAP_DEPTH * depth);
extern void lw6map_depth_fill (LW6MAP_DEPTH * depth, unsigned char value);
extern void lw6map_depth_block_up_down (LW6MAP_DEPTH * depth);
extern void lw6map_depth_block_left_right (LW6MAP_DEPTH * depth);

/*
 * In dir.c
 */
extern char *lw6map_get_system_dir ();
extern void lw6map_free_entry (LW6MAP_ENTRY * entry);
extern LW6SYS_LIST *lw6map_get_maps (char *dir);

/*
 * In elevation.c
 */
extern int lw6map_elevation_read (LW6MAP_ELEVATION * elevation, char *dirname,
				  LW6MAP_DEPTH * depth);
extern void lw6map_elevation_clear (LW6MAP_ELEVATION * elevation);

/*
 * In map.c
 */
extern LW6MAP_MAP *lw6map_new ();
extern void lw6map_free (LW6MAP_MAP * map);
extern int lw6map_memory_footprint (LW6MAP_MAP * map);
extern char *lw6map_repr (LW6MAP_MAP * map);

/*
 * In metadata.c
 */
extern int lw6map_metadata_read (LW6MAP_METADATA * metadata, char *dirname);
extern void lw6map_metadata_clear (LW6MAP_METADATA * metadata);

/*
 * In options.c
 */
extern int lw6map_options_read (LW6OPT_STATIC * param, char *dirname);
extern void lw6map_options_clear (LW6OPT_STATIC * param);

/*
 * In param.c
 */
extern void lw6map_param_defaults (LW6MAP_PARAM * param);
extern int lw6map_param_read (LW6MAP_PARAM * param, char *dirname);
extern void lw6map_param_clear (LW6MAP_PARAM * param);
extern void lw6map_param_copy (LW6MAP_PARAM * dst, LW6MAP_PARAM * src);
extern void lw6map_param_set_int (LW6MAP_PARAM * param, char *key, int value);
extern void lw6map_param_set_bool (LW6MAP_PARAM * param, char *key,
				   int value);
extern void lw6map_param_set_float (LW6MAP_PARAM * param, char *key,
				    float value);
extern void lw6map_param_set_string (LW6MAP_PARAM * param, char *key,
				     char *value);

/*
 * In read.c
 */
extern LW6MAP_MAP *lw6map_read (char *dirname, LW6MAP_PARAM * param);

/*
 * In resampler.c
 */
extern void lw6map_resampler_init (LW6MAP_RESAMPLER * resampler,
				   LW6OPT_STATIC * static_options,
				   LW6SYS_WH * source_shape);
extern void lw6map_resampler_source2target (LW6MAP_RESAMPLER * resampler,
					    int *target_x, int *target_y,
					    int source_x, int source_y);
extern void lw6map_resampler_target2source (LW6MAP_RESAMPLER * resampler,
					    int *source_x, int *source_y,
					    int target_x, int target_y);

/*
 * In style.c
 */
extern void lw6map_style_defaults (LW6MAP_STYLE * style);
extern int lw6map_style_read (LW6MAP_STYLE * style, char *dirname);
extern void lw6map_style_clear (LW6MAP_STYLE * style);
extern void lw6map_style_copy (LW6MAP_STYLE * dst, LW6MAP_STYLE * src);
extern void lw6map_style_get_team_color (LW6MAP_STYLE * style,
					 LW6SYS_COLOR_F * color_f, int id);

/*
 * In texture.c
 */
extern int lw6map_texture_read (LW6MAP_TEXTURE * texture, char *dirname,
				LW6MAP_DEPTH * depth,
				LW6MAP_ELEVATION * elevation);
extern void lw6map_texture_clear (LW6MAP_TEXTURE * texture);

#endif
