/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "liquidwar6.h"

static SCM
mark_map (SCM map)
{
  return SCM_BOOL_F;
}

static size_t
free_map (SCM map)
{
  char *repr = NULL;
  LW6MAP_MAP *c_map = lw6_scm_to_map (map);

  size_t memory_footprint = lw6map_memory_footprint (c_map);

  repr = lw6map_repr (c_map);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.map_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated map is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING, "",
		  _("freeing map smob which isn't in global assoc"));
      lw6map_free (c_map);
    }

  /*
   * Now we always return 0, when Guile doc suggests we should
   * return the amount of memory freed. In fact, doing so
   * caused "Aborted" messages, and program collapses... I suspect
   * this is due to the fact that for this memory checking to work,
   * we should use scm_must_malloc() instead of our home-made macros.
   * Point is our home-made macros are just very fine in the LW6
   * context. Moreover, patching liquidwar6map so that it uses scheme
   * stuff does not really makes sense. Those low-level modules need
   * not be aware that they'll be used in a scheme context.
   */
  memory_footprint = 0;

  return memory_footprint;
}

static int
print_map (SCM map, SCM port, scm_print_state * pstate)
{
  LW6MAP_MAP *c_map = lw6_scm_to_map (map);
  char *repr = NULL;

  repr = lw6map_repr (c_map);

  scm_puts ("#<lw6map-map ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

LW6MAP_MAP *
lw6_scm_to_map (SCM map)
{
  LW6MAP_MAP *c_map;

  c_map = (LW6MAP_MAP *) SCM_SMOB_DATA (map);

  return c_map;
}

SCM
lw6_make_scm_map (LW6MAP_MAP * c_map)
{
  char *repr = NULL;

  /*
   * Calling the garbage collector, the idea is that when we
   * create a map object, we're doing something rare, and
   * we're not playing, so there's no harm in garbage collecting.
   * Additionnally, this will free memory for new map processing.
   * Ideally we would garbage collect just after affecting the
   * new map, or just before allocating memory for it. Putting
   * a garbage collection here does not harm anyway.
   */
  scm_gc ();

  repr = lw6map_repr (c_map);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_set (&lw6_global.map_smobs, repr, (void *) c_map);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.map, c_map);
}

static SCM
mark_game_struct (SCM game_struct)
{
  return SCM_BOOL_F;
}

static size_t
free_game_struct (SCM game_struct)
{
  char *repr = NULL;
  LW6KER_GAME_STRUCT *c_game_struct = lw6_scm_to_game_struct (game_struct);

  size_t memory_footprint =
    lw6ker_game_struct_memory_footprint (c_game_struct);

  repr = lw6ker_game_struct_repr (c_game_struct);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.game_struct_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated game_struct is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING, "",
		  _("freeing game_struct smob which isn't in global assoc"));
      lw6ker_game_struct_free (c_game_struct);
    }

  /*
   * Now we always return 0, when Guile doc suggests we should
   * return the amount of memory freed. In fact, doing so
   * caused "Aborted" messages, and program collapses... I suspect
   * this is due to the fact that for this memory checking to work,
   * we should use scm_must_malloc() instead of our home-made macros.
   * Point is our home-made macros are just very fine in the LW6
   * context. Moreover, patching liquidwar6map so that it uses scheme
   * stuff does not really makes sense. Those low-level modules need
   * not be aware that they'll be used in a scheme context.
   */
  memory_footprint = 0;

  return memory_footprint;
}

static int
print_game_struct (SCM game_struct, SCM port, scm_print_state * pstate)
{
  LW6KER_GAME_STRUCT *c_game_struct = lw6_scm_to_game_struct (game_struct);
  char *repr = NULL;

  repr = lw6ker_game_struct_repr (c_game_struct);

  scm_puts ("#<lw6ker-game-struct ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

LW6KER_GAME_STRUCT *
lw6_scm_to_game_struct (SCM game_struct)
{
  LW6KER_GAME_STRUCT *c_game_struct;

  c_game_struct = (LW6KER_GAME_STRUCT *) SCM_SMOB_DATA (game_struct);

  return c_game_struct;
}

SCM
lw6_make_scm_game_struct (LW6KER_GAME_STRUCT * c_game_struct)
{
  char *repr = NULL;

  /*
   * Calling the garbage collector, the idea is that when we
   * create a map object, we're doing something rare, and
   * we're not playing, so there's no harm in garbage collecting.
   * Additionnally, this will free memory for new map processing.
   * Ideally we would garbage collect just after affecting the
   * new map, or just before allocating memory for it. Putting
   * a garbage collection here does not harm anyway.
   */
  scm_gc ();

  repr = lw6ker_game_struct_repr (c_game_struct);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_set (&lw6_global.game_struct_smobs, repr,
			(void *) c_game_struct);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.game_struct, c_game_struct);
}

static SCM
mark_game_state (SCM game_state)
{
  return SCM_BOOL_F;
}

static size_t
free_game_state (SCM game_state)
{
  char *repr = NULL;
  LW6KER_GAME_STATE *c_game_state = lw6_scm_to_game_state (game_state);

  size_t memory_footprint = lw6ker_game_state_memory_footprint (c_game_state);

  repr = lw6ker_game_state_repr (c_game_state);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.game_state_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated game_state is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING, "",
		  _("freeing game_state smob which isn't in global assoc"));
      lw6ker_game_state_free (c_game_state);
    }

  /*
   * Now we always return 0, when Guile doc suggests we should
   * return the amount of memory freed. In fact, doing so
   * caused "Aborted" messages, and program collapses... I suspect
   * this is due to the fact that for this memory checking to work,
   * we should use scm_must_malloc() instead of our home-made macros.
   * Point is our home-made macros are just very fine in the LW6
   * context. Moreover, patching liquidwar6map so that it uses scheme
   * stuff does not really makes sense. Those low-level modules need
   * not be aware that they'll be used in a scheme context.
   */
  memory_footprint = 0;

  return memory_footprint;
}

static int
print_game_state (SCM game_state, SCM port, scm_print_state * pstate)
{
  LW6KER_GAME_STATE *c_game_state = lw6_scm_to_game_state (game_state);
  char *repr = NULL;

  repr = lw6ker_game_state_repr (c_game_state);

  scm_puts ("#<lw6ker-game-state ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

LW6KER_GAME_STATE *
lw6_scm_to_game_state (SCM game_state)
{
  LW6KER_GAME_STATE *c_game_state;

  c_game_state = (LW6KER_GAME_STATE *) SCM_SMOB_DATA (game_state);

  return c_game_state;
}

SCM
lw6_make_scm_game_state (LW6KER_GAME_STATE * c_game_state)
{
  char *repr = NULL;

  /*
   * Calling the garbage collector, the idea is that when we
   * create a map object, we're doing something rare, and
   * we're not playing, so there's no harm in garbage collecting.
   * Additionnally, this will free memory for new map processing.
   * Ideally we would garbage collect just after affecting the
   * new map, or just before allocating memory for it. Putting
   * a garbage collection here does not harm anyway.
   */
  scm_gc ();

  repr = lw6ker_game_state_repr (c_game_state);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_set (&lw6_global.game_state_smobs, repr,
			(void *) c_game_state);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.game_state, c_game_state);
}

static SCM
mark_game_look (SCM game_look)
{
  return SCM_BOOL_F;
}

static size_t
free_game_look (SCM game_look)
{
  char *repr = NULL;
  LW6GFX_GAME_LOOK *c_game_look = lw6_scm_to_game_look (game_look);

  size_t memory_footprint = lw6gfx_game_look_memory_footprint (c_game_look);

  repr = lw6gfx_game_look_repr (c_game_look);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.game_look_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated game_look is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING, "",
		  _("freeing game_look smob which isn't in global assoc"));
      lw6gfx_game_look_free (c_game_look);
    }

  /*
   * Now we always return 0, when Guile doc suggests we should
   * return the amount of memory freed. In fact, doing so
   * caused "Aborted" messages, and program collapses... I suspect
   * this is due to the fact that for this memory checking to work,
   * we should use scm_must_malloc() instead of our home-made macros.
   * Point is our home-made macros are just very fine in the LW6
   * context. Moreover, patching liquidwar6map so that it uses scheme
   * stuff does not really makes sense. Those low-level modules need
   * not be aware that they'll be used in a scheme context.
   */
  memory_footprint = 0;

  return memory_footprint;
}

static int
print_game_look (SCM game_look, SCM port, scm_print_state * pstate)
{
  LW6GFX_GAME_LOOK *c_game_look = lw6_scm_to_game_look (game_look);
  char *repr = NULL;

  repr = lw6gfx_game_look_repr (c_game_look);

  scm_puts ("#<lw6gfx-game-look ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

LW6GFX_GAME_LOOK *
lw6_scm_to_game_look (SCM game_look)
{
  LW6GFX_GAME_LOOK *c_game_look;

  c_game_look = (LW6GFX_GAME_LOOK *) SCM_SMOB_DATA (game_look);

  return c_game_look;
}

SCM
lw6_make_scm_game_look (LW6GFX_GAME_LOOK * c_game_look)
{
  char *repr = NULL;

  /*
   * Calling the garbage collector, the idea is that when we
   * create a map object, we're doing something rare, and
   * we're not playing, so there's no harm in garbage collecting.
   * Additionnally, this will free memory for new map processing.
   * Ideally we would garbage collect just after affecting the
   * new map, or just before allocating memory for it. Putting
   * a garbage collection here does not harm anyway.
   */
  scm_gc ();

  repr = lw6gfx_game_look_repr (c_game_look);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_set (&lw6_global.game_look_smobs, repr,
			(void *) c_game_look);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.game_look, c_game_look);
}

/*
 * Register all the smobs, that is new types.
 */
int
lw6_register_smobs ()
{
  int ret = 1;

  lw6_global.smob_types.map =
    scm_make_smob_type ("lw6map-map", sizeof (LW6MAP_MAP));
  scm_set_smob_mark (lw6_global.smob_types.map, mark_map);
  scm_set_smob_free (lw6_global.smob_types.map, free_map);
  scm_set_smob_print (lw6_global.smob_types.map, print_map);

  lw6_global.smob_types.game_struct =
    scm_make_smob_type ("lw6ker-game-struct", sizeof (LW6KER_GAME_STRUCT));
  scm_set_smob_mark (lw6_global.smob_types.game_struct, mark_game_struct);
  scm_set_smob_free (lw6_global.smob_types.game_struct, free_game_struct);
  scm_set_smob_print (lw6_global.smob_types.game_struct, print_game_struct);

  lw6_global.smob_types.game_state =
    scm_make_smob_type ("lw6ker-game-state", sizeof (LW6KER_GAME_STATE));
  scm_set_smob_mark (lw6_global.smob_types.game_state, mark_game_state);
  scm_set_smob_free (lw6_global.smob_types.game_state, free_game_state);
  scm_set_smob_print (lw6_global.smob_types.game_state, print_game_state);

  lw6_global.smob_types.game_look =
    scm_make_smob_type ("lw6gfx-game-look", sizeof (LW6GFX_GAME_LOOK));
  scm_set_smob_mark (lw6_global.smob_types.game_look, mark_game_look);
  scm_set_smob_free (lw6_global.smob_types.game_look, free_game_look);
  scm_set_smob_print (lw6_global.smob_types.game_look, print_game_look);

  return ret;
}
