/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6SYSTYPES_H
#define LIQUIDWAR6SYSTYPES_H

#include <sys/types.h>

/*
 * intSIZE_t and u_intSIZE_t are defined by glibc
 * 
 * We do not use these types "everywhere there is an integer".
 * In fact we only use them when there's a real need for
 * a precise bit number, sometimes for data alignment, sometimes
 * just to reduce memory footprint on elements which are instanciated
 * 100 000 times.
 */
typedef int64_t LW6SYS_INT64;
typedef int32_t LW6SYS_INT32;
typedef int16_t LW6SYS_INT16;
typedef int8_t LW6SYS_INT8;
typedef u_int64_t LW6SYS_UINT64;
typedef u_int32_t LW6SYS_UINT32;
typedef u_int16_t LW6SYS_UINT16;
typedef u_int8_t LW6SYS_UINT8;

typedef int LW6SYS_BOOL;

typedef struct LW6SYS_XY_STRUCT
{
  /*
   * Forced to 16 bits to optimize data alignment.
   *
   * It's a deliberate choice in Liquid War to handle "limited size"
   * levels. In fact 16b bits still allows 30000x30000 maps, which are
   * at least 1000 times too slow to play now (2006). Should we follow
   * Moore's law we'd have at least 10 years until those are playable,
   * and well, until then, let's wait. The point is that storing this
   * information (x*y) on 4 bytes might be very important in some cases,
   * since it can reduce memory footprint on structs which are stored
   * in numerous quantities, and therefore maximize chances that we
   * use level 1 & 2 caches and other nice things which happen when
   * memory consumption is not too high.
   *
   * Point is: why use INT32 and then limit it to 16 bits instead of
   * using an INT16 or short in the first place? Answer: it's easier
   * to handle INT32 all the time in the rest of the code. Compiler
   * and CPU might even handle that better than short. Then, and only
   * when data will be read/written in the struct will it be truncated.
   * Typical example is: we want to multiplicate y by w (which is a
   * width). Result is beyond INT16/short scope but we want to handle
   * it! Casting everything to INT32/int is a pain. With this int y:16
   * trick, we use y as a "full-featured" INT32/int and well, when it
   * will be read/written we'll loose values over 32767, but we simply
   * do not care.
   */
  LW6SYS_INT32 x:16;
  LW6SYS_INT32 y:16;
}
LW6SYS_XY;

typedef struct LW6SYS_WH_STRUCT
{
  /*
   * 3 differences with its "XY" equivalent:
   * - sometimes w*h reads better than x*y.
   * - xy is signed, wh is unsigned
   * - these are real int32 values, it does not really cost any
   *   memory for it's usually used as a single "shape" attribute
   *   for a whole map. At the same time, it's very often used
   *   as a test value in loops, so it's interesting to have
   *   it in a value that's easy to optimize for the compiler
   *   (exactly one register...)
   */
  LW6SYS_UINT32 w;
  LW6SYS_UINT32 h;
}
LW6SYS_WH;

typedef struct LW6SYS_COLOR_F_STRUCT
{
  float r;			// red   [0 ... 1.0f]
  float g;			// green [0 ... 1.0f]
  float b;			// blue  [0 ... 1.0f]
  float a;			// alpha [0 ... 1.0f]
}
LW6SYS_COLOR_F;

typedef struct LW6SYS_COLOR_8_STRUCT
{
  LW6SYS_UINT8 r;		// red   [0 ... 255]
  LW6SYS_UINT8 g;		// green [0 ... 255]
  LW6SYS_UINT8 b;		// blue  [0 ... 255]
  LW6SYS_UINT8 a;		// alpha [0 ... 255]
}
LW6SYS_COLOR_8;

#endif
