/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>

#include "config.h"
#include "../../gfx.h"
#include "gl-utils.h"
#include "gl-utils-internal.h"

/*
 * Pushes a key event.
 */
static void
key_down (MOD_GL_UTILS_KEYBOARD_STATE * state, int sym, int unicode,
	  char *label)
{
  (state->keypress_queue)++;
  if ((state->keypress_queue) < MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE)
    {
      state->keypress_current =
	((state->keypress_current) + 1) % MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE;
      state->keypress_buffer[state->keypress_current].sym = sym;
      state->keypress_buffer[state->keypress_current].unicode = unicode;
      state->keypress_buffer[state->keypress_current].label = label;
    }

  if (sym >= 0 && sym < MOD_GL_UTILS_KEYPRESS_STATE_SIZE)
    {
      state->keypress_state[sym] = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _("incorrect keysym (%d) for keydown event"), sym);
    }
}

static void
key_up (MOD_GL_UTILS_KEYBOARD_STATE * state, int sym)
{
  if (sym >= 0 && sym < MOD_GL_UTILS_KEYPRESS_STATE_SIZE)
    {
      state->keypress_state[sym] = 0;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _("incorrect keysym (%d) for keydown event"), sym);
    }
}

void
update_mouse_pos (MOD_GL_UTILS_MOUSE_STATE * state, int x, int y)
{
  state->x = x;
  state->y = y;
}

void
mouse_button_down (MOD_GL_UTILS_MOUSE_STATE * state, int sdl_button)
{
  switch (sdl_button)
    {
    case SDL_BUTTON_LEFT:
      state->button_left.pressed = 1;
      state->button_left.queue++;
      break;
    case SDL_BUTTON_RIGHT:
      state->button_right.pressed = 1;
      state->button_right.queue++;
      break;
    }
}

void
mouse_button_up (MOD_GL_UTILS_MOUSE_STATE * state, int sdl_button)
{
  switch (sdl_button)
    {
    case SDL_BUTTON_LEFT:
      state->button_left.pressed = 0;
      break;
    case SDL_BUTTON_RIGHT:
      state->button_right.pressed = 0;
      break;
    }
}

static int
poll_mouse_button (MOD_GL_UTILS_MOUSE_BUTTON_STATE * state)
{
  int ret = 0;

  if (state->queue > 0)
    {
      ret++;
      state->queue--;
    }

  return ret;
}

/*
 * Internal poll function.
 */
static int
poll (MOD_GL_UTILS_CONTEXT * utils_context)
{
  int ret;
  MOD_GL_UTILS_KEYBOARD_STATE *keyboard_state =
    &utils_context->keyboard_state;
  MOD_GL_UTILS_MOUSE_STATE *mouse_state = &utils_context->mouse_state;

  SDL_Event event;

  ret = SDL_PollEvent (&event) ? 1 : 0;

  if (ret)
    {
      switch (event.type)
	{
	case SDL_KEYDOWN:
	  key_down (keyboard_state,
		    event.key.keysym.sym,
		    event.key.keysym.unicode,
		    SDL_GetKeyName (event.key.keysym.sym));
	  break;
	case SDL_KEYUP:
	  key_up (keyboard_state, event.key.keysym.sym);
	  break;
	case SDL_QUIT:
	  utils_context->quit_state.quit = 1;
	  break;
	case SDL_MOUSEMOTION:
	  utils_context->mouse_state.moved = 1;
	  update_mouse_pos (mouse_state, event.motion.x, event.motion.y);
	  break;
	case SDL_MOUSEBUTTONDOWN:
	  update_mouse_pos (mouse_state, event.button.x, event.button.y);
	  mouse_button_down (mouse_state, event.button.button);
	  break;
	case SDL_MOUSEBUTTONUP:
	  update_mouse_pos (mouse_state, event.button.x, event.button.y);
	  mouse_button_up (mouse_state, event.button.button);
	  break;
	case SDL_VIDEORESIZE:
	  mod_gl_utils_resize_video_mode (utils_context, event.resize.w,
					  event.resize.h);
	  break;
	}
    }

  return ret;
}

/*
 * Polls a key press.
 */
int
mod_gl_utils_poll_keypress (MOD_GL_UTILS_CONTEXT * utils_context,
			    LW6GFX_KEYPRESS * keypress)
{
  int ret = 0;

  poll (utils_context);

  if (utils_context->keyboard_state.keypress_queue > 0)
    {
      ret = 1;
      (utils_context->keyboard_state.keypress_queue)--;
      if (keypress)
	{
	  int pos;

	  pos = (MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE
		 + utils_context->keyboard_state.keypress_current
		 - utils_context->keyboard_state.keypress_queue)
	    % MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE;
	  *keypress = (utils_context->keyboard_state.keypress_buffer)[pos];
	}
    }

  return ret;
}

/*
 * Following functions are used to export to Guile the internal
 * SDL esc/up/down/left/right/enter keycodes which are the basics
 * to handle menu stuff. Other keys are basically ascii keys 
 * and/or special control keys but which do not have any special
 * meaning since they're used for controlling cursors, independently
 * of their expected behavior. Point is esc/up/down/left/right/enter
 * are necessary for menu stuff. One option would be to export all
 * SLDK_ symbols to Guile, but it's 1) useless and 2) it would tie
 * scheme code to SDL, which is not needed nor required nor wished.
 */

int
mod_gl_utils_is_key_esc (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return (keysym == SDLK_ESCAPE);
}

int
mod_gl_utils_is_key_up (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return ((keysym == SDLK_UP) || (keysym == SDLK_KP8));
}

int
mod_gl_utils_is_key_down (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return ((keysym == SDLK_DOWN) || (keysym == SDLK_KP2));
}

int
mod_gl_utils_is_key_left (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return ((keysym == SDLK_LEFT) || (keysym == SDLK_KP4));
}

int
mod_gl_utils_is_key_right (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return ((keysym == SDLK_RIGHT) || (keysym == SDLK_KP6));
}

int
mod_gl_utils_is_key_enter (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return ((keysym == SDLK_RETURN) || (keysym == SDLK_KP_ENTER));
}

int
mod_gl_utils_is_key_del (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return (keysym == SDLK_DELETE);
}

int
mod_gl_utils_is_key_backspace (MOD_GL_UTILS_CONTEXT * utils_context,
			       int keysym)
{
  return (keysym == SDLK_BACKSPACE);
}

int
mod_gl_utils_is_key_help (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return (keysym == SDLK_F1);
}

int
mod_gl_utils_is_key_quit (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  return (keysym == SDLK_F10);
}

/*
 * Polls a quit event
 */
int
mod_gl_utils_poll_quit (MOD_GL_UTILS_CONTEXT * utils_context)
{
  poll (utils_context);

  return utils_context->quit_state.quit;
}

/*
 * Sends a quit event
 */
void
mod_gl_utils_send_quit (MOD_GL_UTILS_CONTEXT * utils_context)
{
  utils_context->quit_state.quit = 1;
}

int
mod_gl_utils_get_key_state (MOD_GL_UTILS_CONTEXT * utils_context, int keysym)
{
  int ret = 0;

  if (keysym >= 0 && keysym < MOD_GL_UTILS_KEYPRESS_STATE_SIZE)
    {
      ret = utils_context->keyboard_state.keypress_state[keysym];
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _("can't get key state for incorrect keysym (%d)"), keysym);
    }

  return ret;
}

/*
 * Returns true if mouse is moved. Even if not moved, x & y
 * contains the position of the mouse.
 */
int
mod_gl_utils_poll_mouse_move (MOD_GL_UTILS_CONTEXT * utils_context,
			      int *x, int *y)
{
  int ret = 0;

  poll (utils_context);

  (*x) = utils_context->mouse_state.x;
  (*y) = utils_context->mouse_state.y;

  if (utils_context->mouse_state.moved)
    {
      utils_context->mouse_state.moved = 0;
      ret = 1;
    }

  return ret;
}

int
mod_gl_utils_poll_mouse_button (MOD_GL_UTILS_CONTEXT * utils_context,
				int button)
{
  int ret = 0;

  poll (utils_context);

  switch (button)
    {
    case LW6GFX_MOUSE_BUTTON_LEFT:
      ret = poll_mouse_button (&(utils_context->mouse_state.button_left));
      break;
    case LW6GFX_MOUSE_BUTTON_RIGHT:
      ret = poll_mouse_button (&(utils_context->mouse_state.button_right));
      break;
    default:
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _("unknown button index %d"), button);
      break;
    }

  return ret;
}

void
mod_gl_utils_get_mouse_state (MOD_GL_UTILS_CONTEXT * utils_context, int *x,
			      int *y, int *button_left, int *button_right)
{
  MOD_GL_UTILS_MOUSE_STATE *mouse_state = &utils_context->mouse_state;

  (*x) = mouse_state->x;
  (*y) = mouse_state->y;
  (*button_left) = mouse_state->button_left.pressed;
  (*button_right) = mouse_state->button_right.pressed;
}
