/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>

#include "config.h"
#include "../../gfx.h"
#include "gl-utils.h"
#include "gl-utils-internal.h"

/*
 * Finds a texture in the cache.
 */
static GLuint
find_in_cache (MOD_GL_UTILS_CONTEXT * context, int w, int h,
	       char *str_key, int int_key)
{
  GLuint texture = 0;
  int i;
  MOD_GL_UTILS_TEXTURECACHE_ARRAY *texturecache_array;
  MOD_GL_UTILS_TEXTURECACHE_ITEM *texturecache_item;

  texturecache_array = &(context->texturecache_array);

  for (i = 0; i < MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE; ++i)
    {
      texturecache_item = &(texturecache_array->item_array[i]);

      if (texturecache_item->surface_w == w &&
	  texturecache_item->surface_h == h &&
	  texturecache_item->int_key == int_key &&
	  strcmp (texturecache_item->str_key, str_key) == 0)
	{
	  texture = texturecache_item->texture;
	  texturecache_item->used++;
	  break;
	}
    }

  return texture;
}

/*
 * Stores a texture in the cache. If the texture isn't defined, does nothing.
 */
static void
store_in_cache (MOD_GL_UTILS_CONTEXT * context, int w, int h,
		char *str_key, int int_key, GLuint texture)
{
  int i;
  MOD_GL_UTILS_TEXTURECACHE_ARRAY *texturecache_array;
  MOD_GL_UTILS_TEXTURECACHE_ITEM *texturecache_item;
  int str_key_length = 0;
  char *str_key_copy = NULL;

  texturecache_array = &(context->texturecache_array);

  if (texture)
    {
      for (i = 0; i < MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE; ++i)
	{
	  texturecache_item = &(texturecache_array->item_array[i]);

	  if (texturecache_item->surface_w == 0 &&
	      texturecache_item->surface_h == 0 &&
	      texturecache_item->int_key == 0 &&
	      texturecache_item->str_key == NULL &&
	      texturecache_item->texture == 0)
	    {
	      str_key_length = strlen (str_key);
	      str_key_copy = LW6SYS_MALLOC (str_key_length + 1);
	      if (str_key_copy != NULL)
		{
		  texturecache_item->used = 1;
		  texturecache_item->surface_w = w;
		  texturecache_item->surface_h = h;
		  texturecache_item->int_key = int_key;
		  strncpy (str_key_copy, str_key, str_key_length + 1);
		  str_key_copy[str_key_length] = '\0';
		  texturecache_item->str_key = str_key_copy;
		  texturecache_item->texture = texture;
		  break;
		}
	    }
	}
      if (str_key_copy == NULL)
	{
	  mod_gl_utils_schedule_delete_texture (context, texture);
	}
    }
}

/*
 * Converts a surface to a texture, but first searches it in the cache.
 */
GLuint
mod_gl_utils_surface2texture_xywh_using_cache (MOD_GL_UTILS_CONTEXT * context,
					       SDL_Surface * surface, int x,
					       int y, int w, int h,
					       char *str_key, int int_key)
{
  GLuint texture = 0;

  texture = find_in_cache (context, surface->w, surface->h, str_key, int_key);

  if (!texture)
    {
      texture =
	mod_gl_utils_surface2texture_xywh (context, surface, x, y, w, h);

      store_in_cache (context, surface->w, surface->h, str_key, int_key,
		      texture);
    }

  return texture;
}

GLuint
mod_gl_utils_surface2texture_wh_using_cache (MOD_GL_UTILS_CONTEXT * context,
					     SDL_Surface * surface, int w,
					     int h, char *str_key,
					     int int_key)
{
  GLuint texture = 0;

  texture = find_in_cache (context, surface->w, surface->h, str_key, int_key);

  if (!texture)
    {
      texture = mod_gl_utils_surface2texture_wh (context, surface, w, h);

      store_in_cache (context, surface->w, surface->h, str_key, int_key,
		      texture);
    }

  return texture;
}

GLuint
mod_gl_utils_surface2texture_using_cache (MOD_GL_UTILS_CONTEXT * context,
					  SDL_Surface * surface,
					  char *str_key, int int_key)
{
  GLuint texture = 0;

  texture = find_in_cache (context, surface->w, surface->h, str_key, int_key);

  if (!texture)
    {
      texture = mod_gl_utils_surface2texture (context, surface);

      store_in_cache (context, surface->w, surface->h, str_key, int_key,
		      texture);
    }

  return texture;
}

/*
 * Converts a map to a texture, but first searches it in the cache.
 */
GLuint
mod_gl_utils_map2texture_xywh_using_cache (MOD_GL_UTILS_CONTEXT * context,
					   LW6MAP_MAP * map, int x, int y,
					   int w, int h, char *str_key,
					   int int_key)
{
  GLuint texture = 0;

  texture =
    find_in_cache (context, map->texture.shape.w, map->texture.shape.h,
		   str_key, int_key);

  if (!texture)
    {
      texture = mod_gl_utils_map2texture_xywh (context, map, x, y, w, h);

      store_in_cache (context, map->texture.shape.w, map->texture.shape.h,
		      str_key, int_key, texture);
    }

  return texture;
}

GLuint
mod_gl_utils_map2texture_wh_using_cache (MOD_GL_UTILS_CONTEXT * context,
					 LW6MAP_MAP * map, int w, int h,
					 char *str_key, int int_key)
{
  GLuint texture = 0;

  texture =
    find_in_cache (context, map->texture.shape.w, map->texture.shape.h,
		   str_key, int_key);

  if (!texture)
    {
      texture = mod_gl_utils_map2texture_wh (context, map, w, h);

      store_in_cache (context, map->texture.shape.w, map->texture.shape.h,
		      str_key, int_key, texture);
    }

  return texture;
}

GLuint
mod_gl_utils_map2texture_using_cache (MOD_GL_UTILS_CONTEXT * context,
				      LW6MAP_MAP * map, char *str_key,
				      int int_key)
{
  GLuint texture = 0;

  texture =
    find_in_cache (context, map->texture.shape.w, map->texture.shape.h,
		   str_key, int_key);

  if (!texture)
    {
      texture = mod_gl_utils_map2texture (context, map);

      store_in_cache (context, map->texture.shape.w, map->texture.shape.h,
		      str_key, int_key, texture);
    }

  return texture;
}

/*
 * Call this after each display, in case we need to clean up the cache.
 */
extern void
mod_gl_utils_optimize_texturecache (MOD_GL_UTILS_CONTEXT * context)
{
  int i;
  MOD_GL_UTILS_TEXTURECACHE_ARRAY *texturecache_array;
  MOD_GL_UTILS_TEXTURECACHE_ITEM *texturecache_item;

  texturecache_array = &(context->texturecache_array);

  for (i = 0; i < MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE; ++i)
    {
      texturecache_item = &(texturecache_array->item_array[i]);
      if (texturecache_item->used == 0 &&
	  texturecache_item->surface_w > 0 &&
	  texturecache_item->surface_h > 0 && texturecache_item->texture != 0)
	{
	  texturecache_item->surface_w = 0;
	  texturecache_item->surface_h = 0;
	  texturecache_item->int_key = 0;
	  if (texturecache_item->str_key != NULL)
	    {
	      LW6SYS_FREE (texturecache_item->str_key);
	    }
	  texturecache_item->str_key = NULL;
	  mod_gl_utils_schedule_delete_texture (context,
						texturecache_item->texture);
	  texturecache_item->texture = 0;
	}
      texturecache_item->used = 0;
    }
}

/*
 * Call this when quitting the game.
 */
extern void
mod_gl_utils_clear_texturecache (MOD_GL_UTILS_CONTEXT * context)
{
  int i;
  MOD_GL_UTILS_TEXTURECACHE_ARRAY *texturecache_array;
  MOD_GL_UTILS_TEXTURECACHE_ITEM *texturecache_item;

  texturecache_array = &(context->texturecache_array);

  for (i = 0; i < MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE; ++i)
    {
      texturecache_item = &(texturecache_array->item_array[i]);
      texturecache_item->used = 0;
      texturecache_item->surface_w = 0;
      texturecache_item->surface_h = 0;
      texturecache_item->int_key = 0;
      if (texturecache_item->str_key != NULL)
	{
	  LW6SYS_FREE (texturecache_item->str_key);
	}
      texturecache_item->str_key = NULL;
      if (texturecache_item->texture != 0)
	{
	  mod_gl_utils_schedule_delete_texture (context,
						texturecache_item->texture);
	  texturecache_item->texture = 0;
	}
    }

  memset (texturecache_array, 0, sizeof (MOD_GL_UTILS_TEXTURECACHE_ARRAY));
}
