/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6GFX_MOD_GL_UTILS_H
#define LIQUIDWAR6GFX_MOD_GL_UTILS_H

#include <GL/gl.h>
#include <SDL/SDL.h>
#include <SDL/SDL_ttf.h>

#include "../../gfx.h"

#define MOD_GL_UTILS_MENU_TEXTURE_W 512
#define MOD_GL_UTILS_MENU_TEXTURE_H 128

/*
 * Internal event buffer. One might wonder why we recode
 * some stuff which could be handled by SDL directly.
 * Answer: we need to do some basic processing before
 * sending the info to guile/scheme. For instance we
 * want key strokes to arrive with all the keycode/scancode/label
 * to be sent. We might also wish to send virtual key events
 * such as "mouse up" or "joy up".
 */
#define MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE 256

/*
 * Size of the buffer we use to keep track of which key is pressed
 * or not. The index in this array is a keysym, not an ascii code.
 */
#define MOD_GL_UTILS_KEYPRESS_STATE_SIZE 512

/*
 * We keep 16 menu bitmaps in memory, to avoid redrawing them
 * at each display frame.
 */
#define MOD_GL_UTILS_MENUCACHE_ARRAY_SIZE 16

/*
 * We keep 32 textures in memory, this is to avoid transfering
 * bitmaps from SDL structures to the OpenGL world at each frame
 * display.
 */
#define MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE 24


/*
 * OpenGL should handle at least this texture size. Theorically
 * we should support any MAX_TEXTURE_SIZE value, but in practise,
 * most recent drivers support 512x512, and the "liquidwarish" 
 * answer to this problem is -> use another backend.
 */
#define MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE 512

/*
 * How many different colors for armies.
 */
#define MOD_GL_SHADES_FOR_FIGHTERS_SCALE 64

typedef struct MOD_GL_UTILS_MEM_COUNTER_STRUCT
{
  int new_counter;
  int delete_counter;
}
MOD_GL_UTILS_MEM_COUNTER;

typedef struct MOD_GL_UTILS_CAPS_STRUCT
{
  GLint max_texture_size;
}
MOD_GL_UTILS_CAPS;

typedef struct MOD_GL_UTILS_SCREEN_STRUCT
{
  int width;
  int height;
  int bpp;
  int fullscreen;
}
MOD_GL_UTILS_SCREEN;

typedef enum MOD_GL_UTILS_RENDER_MODE_ENUM
{
  MOD_GL_UTILS_RENDER_NONE = 0,
  MOD_GL_UTILS_RENDER_2D = 1,
  MOD_GL_UTILS_RENDER_2D_BLEND = 2,
  MOD_GL_UTILS_RENDER_3D_MAP = 3,
  MOD_GL_UTILS_RENDER_3D_MENU = 4,
  MOD_GL_UTILS_RENDER_3D_PICK = 5
}
MOD_GL_UTILS_RENDER_MODE;

typedef struct MOD_GL_UTILS_RECT_ARRAY_STRUCT
{
  int n_w;
  int n_h;
  int *x0;
  int *y0;
  int *w;
  int *h;
}
MOD_GL_UTILS_RECT_ARRAY;

typedef struct MOD_GL_UTILS_SURFACE_ARRAY_STRUCT
{
  MOD_GL_UTILS_RECT_ARRAY layout;
  SDL_Surface **surfaces;
}
MOD_GL_UTILS_SURFACE_ARRAY;

typedef struct MOD_GL_UTILS_TEXTURE_ARRAY_STRUCT
{
  MOD_GL_UTILS_RECT_ARRAY layout;
  GLuint *textures;
}
MOD_GL_UTILS_TEXTURE_ARRAY;

typedef struct MOD_GL_UTILS_KEYBOARD_STATE_STRUCT
{
  LW6GFX_KEYPRESS keypress_buffer[MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE];
  int keypress_current;
  int keypress_queue;
  int keypress_state[MOD_GL_UTILS_KEYPRESS_STATE_SIZE];
}
MOD_GL_UTILS_KEYBOARD_STATE;

typedef struct MOD_GL_UTILS_MOUSE_BUTTON_STATE_STRUCT
{
  int pressed;
  int queue;
}
MOD_GL_UTILS_MOUSE_BUTTON_STATE;

typedef struct MOD_GL_UTILS_MOUSE_STATE_STRUCT
{
  int x;
  int y;
  int moved;
  MOD_GL_UTILS_MOUSE_BUTTON_STATE button_left;
  MOD_GL_UTILS_MOUSE_BUTTON_STATE button_right;
}
MOD_GL_UTILS_MOUSE_STATE;

typedef struct MOD_GL_UTILS_QUIT_STATE_STRUCT
{
  int quit;
}
MOD_GL_UTILS_QUIT_STATE;

typedef struct MOD_GL_UTILS_TEXTURE_DATA_STRUCT
{
  LW6SYS_LIST *to_delete;
}
MOD_GL_UTILS_TEXTURE_DATA;

typedef struct MOD_GL_UTILS_FONT_DATA_STRUCT
{
  TTF_Font *system;
  TTF_Font *menu;
}
MOD_GL_UTILS_FONT_DATA;

typedef struct MOD_GL_UTILS_MENUCACHE_ITEM_STRUCT
{
  char *key;
  SDL_Surface *surface;
}
MOD_GL_UTILS_MENUCACHE_ITEM;

typedef struct MOD_GL_UTILS_MENUCACHE_ARRAY_STRUCT
{
  int last_stored_item;
  MOD_GL_UTILS_MENUCACHE_ITEM item_array[MOD_GL_UTILS_MENUCACHE_ARRAY_SIZE];
}
MOD_GL_UTILS_MENUCACHE_ARRAY;

typedef struct MOD_GL_UTILS_TEXTURECACHE_ITEM_STRUCT
{
  int used;
  int surface_w;
  int surface_h;
  char *str_key;
  int int_key;
  GLuint texture;
}
MOD_GL_UTILS_TEXTURECACHE_ITEM;

typedef struct MOD_GL_UTILS_TEXTURECACHE_ARRAY_STRUCT
{
  MOD_GL_UTILS_TEXTURECACHE_ITEM
    item_array[MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE];
}
MOD_GL_UTILS_TEXTURECACHE_ARRAY;

typedef struct MOD_GL_UTILS_CONST_DATA_STRUCT
{
  /*
   * Set by the background module
   */
  LW6SYS_COLOR_F menu_fg;
  LW6SYS_COLOR_F menu_bg;
  LW6SYS_COLOR_F menu_fg_selected;
  LW6SYS_COLOR_F menu_bg_selected;
  /*
   * Set by the hud module
   */
  float viewport_x;
  float viewport_y;
  float viewport_w;
  float viewport_h;
  /*
   * Set by the utils module itself
   */
  float quit_sleep;
  float persp_fovy;
  float persp_znear;
  float persp_zfar;
  int menu_max_displayed_items;
  LW6SYS_COLOR_F system_font_fg;
  LW6SYS_COLOR_F system_font_shade;
  int menu_font_size;
  int system_font_size;
  int system_font_dh;
  int system_font_dw;
  float system_font_hcoef;
  float gradient_opacity;
  float zones_opacity;
}
MOD_GL_UTILS_CONST_DATA;

typedef struct MOD_GL_UTILS_TEAM_COLOR_MAP_STRUCT
{
  Uint32 team_colors[LW6OPT_MAX_NB_TEAMS][MOD_GL_SHADES_FOR_FIGHTERS_SCALE +
					  1];
}
MOD_GL_UTILS_TEAM_COLOR_MAP;

typedef struct MOD_GL_UTILS_TIMER_DATA_STRUCT
{
  int ticks_offset;
}
MOD_GL_UTILS_TIMER_DATA;

typedef struct MOD_GL_UTILS_LAST_ACTION_STRUCT
{
  int game_texture_array_update_id;
  int game_texture_array_update_rounds;
}
MOD_GL_UTILS_LAST_ACTION;

typedef struct MOD_GL_UTILS_CONTEXT_STRUCT
{
  void (*resize_callback) (int width, int height, int fullscreen);
  MOD_GL_UTILS_MEM_COUNTER surface_counter;
  MOD_GL_UTILS_MEM_COUNTER texture_counter;
  MOD_GL_UTILS_CAPS caps;
  MOD_GL_UTILS_SCREEN screen;
  MOD_GL_UTILS_RENDER_MODE render_mode;
  MOD_GL_UTILS_TEXTURE_DATA texture_data;
  MOD_GL_UTILS_FONT_DATA font_data;
  MOD_GL_UTILS_KEYBOARD_STATE keyboard_state;
  MOD_GL_UTILS_MOUSE_STATE mouse_state;
  MOD_GL_UTILS_QUIT_STATE quit_state;
  MOD_GL_UTILS_CONST_DATA const_data;
  MOD_GL_UTILS_MENUCACHE_ARRAY menucache_array;
  MOD_GL_UTILS_TEXTURECACHE_ARRAY texturecache_array;
  MOD_GL_UTILS_TEAM_COLOR_MAP team_color_map;
  MOD_GL_UTILS_TIMER_DATA timer_data;
  MOD_GL_UTILS_LAST_ACTION last_action;
}
MOD_GL_UTILS_CONTEXT;

/*
 * In color.c
 */
extern SDL_Color mod_gl_utils_color_sys2sdl (LW6SYS_COLOR_F * color_f);
extern void mod_gl_utils_update_team_color_map (MOD_GL_UTILS_TEAM_COLOR_MAP *
						team_color_map,
						LW6MAP_STYLE * map_style);
inline static Uint32
mod_gl_utils_get_shaded_color_for_fighter (MOD_GL_UTILS_CONTEXT *
					   utils_context, int team_id,
					   int health)
{
  return utils_context->team_color_map.
    team_colors[team_id][(health * MOD_GL_SHADES_FOR_FIGHTERS_SCALE) /
			 LW6OPT_MAX_FIGHTER_HEALTH];
}


/*
 * In data.c
 */
extern SDL_Surface *mod_gl_utils_load_image (MOD_GL_UTILS_CONTEXT *
					     utils_context, char *dirname,
					     char *filename);
extern void mod_gl_utils_unload_image (MOD_GL_UTILS_CONTEXT * utils_context,
				       SDL_Surface * image);
extern GLuint mod_gl_utils_load_texture (MOD_GL_UTILS_CONTEXT * utils_context,
					 SDL_Surface * surface);
extern void mod_gl_utils_unload_texture (MOD_GL_UTILS_CONTEXT * utils_context,
					 int texture);
extern int mod_gl_utils_load_data (MOD_GL_UTILS_CONTEXT * utils_context);
extern void mod_gl_utils_unload_data (MOD_GL_UTILS_CONTEXT * utils_context);

/*
 * In draw.c
 */
extern void mod_gl_utils_draw_rectfill (SDL_Surface * surface, int x1, int y1,
					int x2, int y2, Uint32 color);

/*
 * In event.c
 */
extern int mod_gl_utils_poll_keypress (MOD_GL_UTILS_CONTEXT * utils_context,
				       LW6GFX_KEYPRESS * keypress);
extern int mod_gl_utils_is_key_esc (MOD_GL_UTILS_CONTEXT * utils_context,
				    int keysym);
extern int mod_gl_utils_is_key_up (MOD_GL_UTILS_CONTEXT * utils_context,
				   int keysym);
extern int mod_gl_utils_is_key_down (MOD_GL_UTILS_CONTEXT * utils_context,
				     int keysym);
extern int mod_gl_utils_is_key_left (MOD_GL_UTILS_CONTEXT * utils_context,
				     int keysym);
extern int mod_gl_utils_is_key_right (MOD_GL_UTILS_CONTEXT * utils_context,
				      int keysym);
extern int mod_gl_utils_is_key_enter (MOD_GL_UTILS_CONTEXT * utils_context,
				      int keysym);
extern int mod_gl_utils_is_key_del (MOD_GL_UTILS_CONTEXT * utils_context,
				    int keysym);
extern int mod_gl_utils_is_key_backspace (MOD_GL_UTILS_CONTEXT *
					  utils_context, int keysym);
extern int mod_gl_utils_is_key_help (MOD_GL_UTILS_CONTEXT * utils_context,
				     int keysym);
extern int mod_gl_utils_is_key_quit (MOD_GL_UTILS_CONTEXT * utils_context,
				     int keysym);
extern int mod_gl_utils_poll_quit (MOD_GL_UTILS_CONTEXT * utils_context);
extern void mod_gl_utils_send_quit (MOD_GL_UTILS_CONTEXT * utils_context);
extern int mod_gl_utils_get_key_state (MOD_GL_UTILS_CONTEXT * utils_context,
				       int keysym);
extern int mod_gl_utils_poll_mouse_move (MOD_GL_UTILS_CONTEXT *
					 utils_context, int *x, int *y);
extern int mod_gl_utils_poll_mouse_button (MOD_GL_UTILS_CONTEXT *
					   utils_context, int button);
extern void mod_gl_utils_get_mouse_state (MOD_GL_UTILS_CONTEXT *
					  utils_context, int *x, int *y,
					  int *button_left,
					  int *button_right);

/*
 * In font.c
 */
extern int mod_gl_utils_get_system_text_width (MOD_GL_UTILS_CONTEXT *
					       utils_context, char *text);
extern int mod_gl_utils_get_system_text_height (MOD_GL_UTILS_CONTEXT *
						utils_context, char *text);
extern void mod_gl_utils_draw_system_text (MOD_GL_UTILS_CONTEXT *
					   utils_context, char *text, int x,
					   int y);
extern void mod_gl_utils_draw_system_text_top_left (MOD_GL_UTILS_CONTEXT *
						    utils_context,
						    char **text);
extern void mod_gl_utils_draw_system_text_top_right (MOD_GL_UTILS_CONTEXT *
						     utils_context,
						     char **text);
extern void mod_gl_utils_draw_system_text_bottom_left (MOD_GL_UTILS_CONTEXT *
						       utils_context,
						       char **text);
extern void mod_gl_utils_draw_system_text_bottom_right (MOD_GL_UTILS_CONTEXT *
							utils_context,
							char **text);


/*
 * In game.c
 */
extern void mod_gl_utils_update_game_texture (MOD_GL_UTILS_CONTEXT *
					      utils_context,
					      SDL_Surface * surface,
					      GLuint texture,
					      LW6KER_GAME_STATE * game_state,
					      LW6MAP_PARAM * map_param,
					      LW6GFX_GAME_LOOK * game_look,
					      int layer_id);
extern void mod_gl_utils_update_game_texture_array (MOD_GL_UTILS_CONTEXT *
						    utils_context,
						    MOD_GL_UTILS_SURFACE_ARRAY
						    * surface_array,
						    MOD_GL_UTILS_TEXTURE_ARRAY
						    * texture_array,
						    LW6KER_GAME_STATE *
						    game_state,
						    LW6MAP_PARAM * map_param,
						    LW6GFX_GAME_LOOK *
						    game_look, int layer_id);

/*
 * In gradient.c
 */
extern SDL_Surface *mod_gl_utils_create_gradient_surface (MOD_GL_UTILS_CONTEXT
							  * utils_context,
							  LW6KER_MAP_STATE *
							  map_state,
							  int team_id);

/*
 * In info.c
 */
extern void mod_gl_utils_display_sysinfo (MOD_GL_UTILS_CONTEXT *
					  utils_context, int fps, int rps,
					  int bytes_in, int bytes_out);
extern void mod_gl_utils_display_log (MOD_GL_UTILS_CONTEXT * utils_context,
				      char **log);
extern void mod_gl_utils_display_time (MOD_GL_UTILS_CONTEXT * utils_context,
				       long long seconds);
extern void mod_gl_utils_display_version (MOD_GL_UTILS_CONTEXT *
					  utils_context);

/*
 * In light.c
 */
extern void mod_gl_utils_main_light (MOD_GL_UTILS_CONTEXT * utils_context);

/*
 * In mode.c
 */
extern int mod_gl_utils_set_video_mode (MOD_GL_UTILS_CONTEXT * utils_context,
					int width, int height, int fullscreen,
					void (*resize_callback) (int width,
								 int height,
								 int
								 fullscreen));
extern int mod_gl_utils_resize_video_mode (MOD_GL_UTILS_CONTEXT *
					   utils_context, int width,
					   int height);

/*
 * In menucache.c
 */
extern int mod_gl_utils_store_button_in_menucache (MOD_GL_UTILS_CONTEXT *
						   utils_context, char *text,
						   int select,
						   SDL_Surface * surface);
extern SDL_Surface
  * mod_gl_utils_get_button_from_menucache (MOD_GL_UTILS_CONTEXT * context,
					    char *text, int select);
extern void mod_gl_utils_clear_menucache (MOD_GL_UTILS_CONTEXT *
					  utils_context);

/*
 * In pixel.c
 */
extern Uint32 mod_gl_utils_getpixel (SDL_Surface * surface, int x, int y);
extern void mod_gl_utils_putpixel (SDL_Surface * surface, int x, int y,
				   Uint32 pixel);

/*
 * In rectarray.c
 */
extern int mod_gl_utils_rect_array_init (MOD_GL_UTILS_CONTEXT *
					 utils_context,
					 MOD_GL_UTILS_RECT_ARRAY * rect_array,
					 int w, int h, int tile_size);
extern void mod_gl_utils_rect_array_clear (MOD_GL_UTILS_CONTEXT *
					   utils_context,
					   MOD_GL_UTILS_RECT_ARRAY *
					   rect_array);

/*
 * In render.c
 */
extern void mod_gl_utils_prepare_buffer (MOD_GL_UTILS_CONTEXT *
					 utils_context);
extern void mod_gl_utils_swap_buffers (MOD_GL_UTILS_CONTEXT * utils_context);
extern void mod_gl_utils_set_render_mode_2d (MOD_GL_UTILS_CONTEXT *
					     utils_context);
extern void mod_gl_utils_set_render_mode_2d_blend (MOD_GL_UTILS_CONTEXT *
						   utils_context);
extern void mod_gl_utils_set_render_mode_3d_map (MOD_GL_UTILS_CONTEXT *
						 utils_context);
extern void mod_gl_utils_set_render_mode_3d_menu (MOD_GL_UTILS_CONTEXT *
						  utils_context);
extern void mod_gl_utils_set_render_mode_3d_pick (MOD_GL_UTILS_CONTEXT *
						  utils_context, int x,
						  int y);

/*
 * In setup.c
 */
extern MOD_GL_UTILS_CONTEXT *mod_gl_utils_init ();
extern int mod_gl_utils_set_video_mode (MOD_GL_UTILS_CONTEXT * utils_context,
					int width, int height, int fullscreen,
					void (*resize_callback) (int width,
								 int height,
								 int
								 fullscreen));
extern int mod_gl_utils_resize_video_mode (MOD_GL_UTILS_CONTEXT *
					   utils_context, int width,
					   int height);
extern void mod_gl_utils_quit (MOD_GL_UTILS_CONTEXT * utils_context);

/*
 * In surface.c
 */
extern SDL_Surface *mod_gl_utils_create_surface (MOD_GL_UTILS_CONTEXT *
						 utils_context, int w, int h);
extern void mod_gl_utils_delete_surface (MOD_GL_UTILS_CONTEXT * utils_context,
					 SDL_Surface * surface);
extern void mod_gl_utils_clear_surface_with_color (SDL_Surface * surface,
						   SDL_Color color);
extern void mod_gl_utils_clear_surface (SDL_Surface * surface);
extern SDL_Surface *mod_gl_utils_map2surface_xywh (MOD_GL_UTILS_CONTEXT *
						   utils_context,
						   LW6MAP_MAP * map, int x,
						   int y, int w, int h);
extern SDL_Surface *mod_gl_utils_map2surface_wh (MOD_GL_UTILS_CONTEXT *
						 utils_context,
						 LW6MAP_MAP * map, int w,
						 int h);
extern SDL_Surface *mod_gl_utils_map2surface (MOD_GL_UTILS_CONTEXT *
					      utils_context,
					      LW6MAP_MAP * map);

/*
 * In surfacearray.c
 */
extern int mod_gl_utils_surface_array_init (MOD_GL_UTILS_CONTEXT *
					    utils_context,
					    MOD_GL_UTILS_SURFACE_ARRAY *
					    surface_array, int w, int h,
					    int tile_size);
extern int mod_gl_utils_surface_array_init_from_surface (MOD_GL_UTILS_CONTEXT
							 * utils_context,
							 MOD_GL_UTILS_SURFACE_ARRAY
							 * surface_array,
							 SDL_Surface *
							 surface,
							 int tile_size);
extern int mod_gl_utils_surface_array_init_from_map (MOD_GL_UTILS_CONTEXT *
						     utils_context,
						     MOD_GL_UTILS_SURFACE_ARRAY
						     * surface_array,
						     LW6MAP_MAP * map,
						     int tile_size);
extern void mod_gl_utils_surface_array_clear (MOD_GL_UTILS_CONTEXT *
					      utils_context,
					      MOD_GL_UTILS_SURFACE_ARRAY *
					      surface_array);
static inline void
mod_gl_utils_set_surface_array_surface (MOD_GL_UTILS_SURFACE_ARRAY *
					surface_array, int n_x, int n_y,
					SDL_Surface * surface)
{
  surface_array->surfaces[n_y * surface_array->layout.n_w + n_x] = surface;
}
static inline SDL_Surface *
mod_gl_utils_get_surface_array_surface (MOD_GL_UTILS_SURFACE_ARRAY *
					surface_array, int n_x, int n_y)
{
  return (surface_array->surfaces[n_y * surface_array->layout.n_w + n_x]);
}

/*
 * In texture.c
 */
extern int mod_gl_utils_power_of_two_le (int size);
extern int mod_gl_utils_power_of_two_ge (int size);
extern float mod_gl_utils_texture_scale (int size);
extern GLuint mod_gl_utils_surface2texture_xywh (MOD_GL_UTILS_CONTEXT *
						 utils_context,
						 SDL_Surface * surface, int x,
						 int y, int w, int h);
extern GLuint mod_gl_utils_surface2texture_wh (MOD_GL_UTILS_CONTEXT *
					       utils_context,
					       SDL_Surface * surface, int w,
					       int h);
extern GLuint mod_gl_utils_surface2texture (MOD_GL_UTILS_CONTEXT *
					    utils_context,
					    SDL_Surface * surface);
extern GLuint mod_gl_utils_map2texture_xywh (MOD_GL_UTILS_CONTEXT *
					     utils_context, LW6MAP_MAP * map,
					     int x, int y, int w, int h);
extern GLuint mod_gl_utils_map2texture_wh (MOD_GL_UTILS_CONTEXT *
					   utils_context, LW6MAP_MAP * map,
					   int w, int h);
extern GLuint mod_gl_utils_map2texture (MOD_GL_UTILS_CONTEXT * utils_context,
					LW6MAP_MAP * map);
extern void mod_gl_utils_delete_texture (MOD_GL_UTILS_CONTEXT * utils_context,
					 GLuint texture);
extern void mod_gl_utils_schedule_delete_texture (MOD_GL_UTILS_CONTEXT *
						  utils_context,
						  GLuint texture);
extern void mod_gl_utils_delete_scheduled_textures (MOD_GL_UTILS_CONTEXT *
						    utils_context);
extern void mod_gl_utils_texture_update (MOD_GL_UTILS_CONTEXT *
					 utils_context,
					 GLuint texture,
					 SDL_Surface * surface);

/*
 * In texturearray.c
 */
extern int mod_gl_utils_texture_array_init (MOD_GL_UTILS_CONTEXT *
					    utils_context,
					    MOD_GL_UTILS_TEXTURE_ARRAY *
					    texture_array, int w, int h,
					    int tile_size);
extern void mod_gl_utils_texture_array_update (MOD_GL_UTILS_CONTEXT *
					       utils_context,
					       MOD_GL_UTILS_TEXTURE_ARRAY *
					       texture_array,
					       MOD_GL_UTILS_SURFACE_ARRAY *
					       surface_array);
extern void mod_gl_utils_texture_array_clear (MOD_GL_UTILS_CONTEXT *
					      utils_context,
					      MOD_GL_UTILS_TEXTURE_ARRAY *
					      texture_array);
extern void mod_gl_utils_schedule_delete_texture_array (MOD_GL_UTILS_CONTEXT *
							utils_context,
							MOD_GL_UTILS_TEXTURE_ARRAY
							* texture_array);
static inline void
mod_gl_utils_set_texture_array_texture (MOD_GL_UTILS_TEXTURE_ARRAY *
					texture_array, int n_x, int n_y,
					GLuint texture)
{
  texture_array->textures[n_y * texture_array->layout.n_w + n_x] = texture;
}
static inline GLuint
mod_gl_utils_get_texture_array_texture (MOD_GL_UTILS_TEXTURE_ARRAY *
					texture_array, int n_x, int n_y)
{
  return (texture_array->textures[n_y * texture_array->layout.n_w + n_x]);
}

/*
 * In texturecache.c
 */
extern GLuint
mod_gl_utils_surface2texture_xywh_using_cache (MOD_GL_UTILS_CONTEXT * context,
					       SDL_Surface * surface, int x,
					       int y, int w, int h,
					       char *str_key, int int_key);
extern GLuint
mod_gl_utils_surface2texture_wh_using_cache (MOD_GL_UTILS_CONTEXT * context,
					     SDL_Surface * surface, int w,
					     int h, char *str_key,
					     int int_key);
extern GLuint mod_gl_utils_surface2texture_using_cache (MOD_GL_UTILS_CONTEXT *
							utils_context,
							SDL_Surface * surface,
							char *str_key,
							int int_key);
extern GLuint mod_gl_utils_map2texture_xywh_using_cache (MOD_GL_UTILS_CONTEXT
							 * context,
							 LW6MAP_MAP * map,
							 int x, int y, int w,
							 int h, char *str_key,
							 int int_key);
extern GLuint mod_gl_utils_map2texture_wh_using_cache (MOD_GL_UTILS_CONTEXT *
						       context,
						       LW6MAP_MAP * map,
						       int w, int h,
						       char *str_key,
						       int int_key);
extern GLuint mod_gl_utils_map2texture_using_cache (MOD_GL_UTILS_CONTEXT *
						    utils_context,
						    LW6MAP_MAP * map,
						    char *str_key,
						    int int_key);
extern void mod_gl_utils_optimize_texturecache (MOD_GL_UTILS_CONTEXT *
						utils_context);
extern void mod_gl_utils_clear_texturecache (MOD_GL_UTILS_CONTEXT *
					     utils_context);

/*
 * In timer.c
 */
extern void mod_gl_utils_timer_init (MOD_GL_UTILS_CONTEXT * utils_context,
				     int ticks);
extern void mod_gl_utils_delay (MOD_GL_UTILS_CONTEXT * utils_context, int ms);
extern int mod_gl_utils_get_ticks (MOD_GL_UTILS_CONTEXT * utils_context);

/*
 * In viewport.c
 */
extern void mod_gl_utils_viewport_calc_view_area (MOD_GL_UTILS_CONTEXT *
						  utils_context, int *view_x1,
						  int *view_y1, int *view_x2,
						  int *view_y2);
extern void mod_gl_utils_viewport_calc_map_area (MOD_GL_UTILS_CONTEXT *
						 utils_context, float *map_x1,
						 float *map_y1, float *map_x2,
						 float *map_y2, int shape_w,
						 int shape_h,
						 LW6MAP_PARAM * map_param);

/*
 * In zones.c
 */
extern SDL_Surface *mod_gl_utils_create_zones_surface (MOD_GL_UTILS_CONTEXT *
						       utils_context,
						       LW6KER_MAP_STRUCT *
						       map_struct);

#endif
