/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <string.h>

#include "config.h"
#include "../gfx.h"
#include "mod-gl-internal.h"

static void
clear_background_backend (_MOD_GL_CONTEXT * gfx_context)
{
  if (gfx_context->background_backend)
    {
      mod_gl_background_quit (&(gfx_context->utils_context),
			      gfx_context->background_backend);
      mod_gl_background_destroy_backend (&(gfx_context->utils_context),
					 gfx_context->background_backend);
      gfx_context->background_backend = NULL;
    }

  mod_gl_utils_clear_texturecache (&(gfx_context->utils_context));
  mod_gl_utils_clear_menucache (&(gfx_context->utils_context));
}

static void
clear_hud_backend (_MOD_GL_CONTEXT * gfx_context)
{
  if (gfx_context->hud_backend)
    {
      mod_gl_hud_quit (&(gfx_context->utils_context),
		       gfx_context->hud_backend);
      mod_gl_hud_destroy_backend (&(gfx_context->utils_context),
				  gfx_context->hud_backend);
      gfx_context->hud_backend = NULL;
    }

  mod_gl_utils_clear_texturecache (&(gfx_context->utils_context));
  mod_gl_utils_clear_menucache (&(gfx_context->utils_context));
}

static void
clear_menu_backend (_MOD_GL_CONTEXT * gfx_context)
{
  if (gfx_context->menu_backend)
    {
      mod_gl_menu_quit (&(gfx_context->utils_context),
			gfx_context->menu_backend);
      mod_gl_menu_destroy_backend (&(gfx_context->utils_context),
				   gfx_context->menu_backend);
      gfx_context->menu_backend = NULL;
    }

  mod_gl_utils_clear_texturecache (&(gfx_context->utils_context));
  mod_gl_utils_clear_menucache (&(gfx_context->utils_context));
}

static void
clear_view_backend (_MOD_GL_CONTEXT * gfx_context)
{
  if (gfx_context->view_backend)
    {
      mod_gl_view_quit (&(gfx_context->utils_context),
			gfx_context->view_backend);
      mod_gl_view_destroy_backend (&(gfx_context->utils_context),
				   gfx_context->view_backend);
      gfx_context->view_backend = NULL;
    }

  mod_gl_utils_clear_texturecache (&(gfx_context->utils_context));
  mod_gl_utils_clear_menucache (&(gfx_context->utils_context));
}

int
_mod_gl_set_background_backend (_MOD_GL_CONTEXT * gfx_context,
				char *background_style)
{
  int ret = 0;

  if ((!background_style) || strlen (background_style) == 0)
    {
      background_style = LW6MAP_STYLE_DEFAULT_BACKGROUND_STYLE;
    }

  if ((!gfx_context->background_backend)
      || strcmp (gfx_context->background_backend->backend_name,
		 background_style))
    {
      clear_background_backend (gfx_context);

      gfx_context->background_backend =
	mod_gl_background_create_backend (&(gfx_context->utils_context),
					  background_style);
      if (gfx_context->background_backend)
	{
	  ret =
	    mod_gl_background_init (&(gfx_context->utils_context),
				    gfx_context->background_backend);
	}
    }

  return ret;
}

int
_mod_gl_set_hud_backend (_MOD_GL_CONTEXT * gfx_context, char *hud_style)
{
  int ret = 0;

  if ((!hud_style) || strlen (hud_style) == 0)
    {
      hud_style = LW6MAP_STYLE_DEFAULT_HUD_STYLE;
    }

  if ((!gfx_context->hud_backend)
      || strcmp (gfx_context->hud_backend->backend_name, hud_style))
    {
      clear_hud_backend (gfx_context);

      gfx_context->hud_backend =
	mod_gl_hud_create_backend (&(gfx_context->utils_context), hud_style);
      if (gfx_context->hud_backend)
	{
	  ret =
	    mod_gl_hud_init (&(gfx_context->utils_context),
			     gfx_context->hud_backend);
	}
    }

  return ret;
}

int
_mod_gl_set_menu_backend (_MOD_GL_CONTEXT * gfx_context, char *menu_style)
{
  int ret = 0;

  if ((!menu_style) || strlen (menu_style) == 0)
    {
      menu_style = LW6MAP_STYLE_DEFAULT_MENU_STYLE;
    }

  if ((!gfx_context->menu_backend)
      || strcmp (gfx_context->menu_backend->backend_name, menu_style))
    {
      clear_menu_backend (gfx_context);

      gfx_context->menu_backend =
	mod_gl_menu_create_backend (&(gfx_context->utils_context),
				    menu_style);
      if (gfx_context->menu_backend)
	{
	  ret =
	    mod_gl_menu_init (&(gfx_context->utils_context),
			      gfx_context->menu_backend);
	}
    }

  return ret;
}

int
_mod_gl_set_view_backend (_MOD_GL_CONTEXT * gfx_context, char *view_style)
{
  int ret = 0;

  if ((!view_style) || strlen (view_style) == 0)
    {
      view_style = LW6MAP_STYLE_DEFAULT_VIEW_STYLE;
    }

  if ((!gfx_context->view_backend)
      || strcmp (gfx_context->view_backend->backend_name, view_style))
    {
      clear_view_backend (gfx_context);

      gfx_context->view_backend =
	mod_gl_view_create_backend (&(gfx_context->utils_context),
				    view_style);
      if (gfx_context->view_backend)
	{
	  ret =
	    mod_gl_view_init (&(gfx_context->utils_context),
			      gfx_context->view_backend);
	}
    }

  return ret;
}

/*
 * Low-level SDL initialisation.
 */
_MOD_GL_CONTEXT *
_mod_gl_init (int width, int height, int fullscreen,
	      void (*resize_callback) (int width, int height, int fullscreen),
	      int ticks)
{
  _MOD_GL_CONTEXT *gfx_context = NULL;
  int background_ok = 0;
  int hud_ok = 0;
  int menu_ok = 0;
  int view_ok = 0;
  int sdl_ok = 1;

  gfx_context = (_MOD_GL_CONTEXT *) LW6SYS_MALLOC (sizeof (_MOD_GL_CONTEXT));
  if (!gfx_context)
    {
      /*
       * No use to continue if this basic malloc fails...
       */
      exit (1);
    }
  memset (gfx_context, 0, sizeof (_MOD_GL_CONTEXT));


  if (lw6sys_sdl_register ())
    {
      sdl_ok = !SDL_Init (0);
    }

  sdl_ok = sdl_ok && (SDL_WasInit (SDL_INIT_TIMER)
		      || !SDL_InitSubSystem (SDL_INIT_TIMER));
  //sdl_ok = sdl_ok && (SDL_WasInit(SDL_INIT_JOYSTICK) || !SDL_InitSubSystem(SDL_INIT_JOYSTICK));
  sdl_ok = sdl_ok && (SDL_WasInit (SDL_INIT_VIDEO)
		      || !SDL_InitSubSystem (SDL_INIT_VIDEO));

  if (sdl_ok)
    {
      lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("SDL Init"));
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl", _("SDL init error: \"%s\""),
		  SDL_GetError ());
      exit (1);
    }

  SDL_EnableUNICODE (1);

  if (TTF_Init () == -1)
    {
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		  _("SDL_ttf init error: \"%s\""), TTF_GetError ());
      exit (1);
    }

  srandom (time (NULL));

  if (mod_gl_utils_set_video_mode
      (&(gfx_context->utils_context), width, height, fullscreen,
       resize_callback))
    {
      if (mod_gl_utils_load_data (&(gfx_context->utils_context)))
	{
	  background_ok = _mod_gl_set_background_backend (gfx_context, NULL);
	  hud_ok = _mod_gl_set_hud_backend (gfx_context, NULL);
	  menu_ok = _mod_gl_set_menu_backend (gfx_context, NULL);
	  view_ok = _mod_gl_set_view_backend (gfx_context, NULL);
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		      _("unable to load data"));
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_ERROR, "gfx-mod-gl",
		  _("unable to set video mode"));
    }

  if (gfx_context && !(background_ok && hud_ok && menu_ok && view_ok))
    {
      _mod_gl_quit (gfx_context);
      gfx_context = NULL;
    }

  mod_gl_utils_timer_init (&(gfx_context->utils_context), ticks);

  return gfx_context;
}

/*
 * Ends-up all SDL stuff.
 */
void
_mod_gl_quit (_MOD_GL_CONTEXT * gfx_context)
{
  float quit_sleep;

  /*
   * Keep this value locally since it can disappear
   * when freeing stuff.
   */
  quit_sleep = gfx_context->utils_context.const_data.quit_sleep;

  clear_background_backend (gfx_context);
  clear_hud_backend (gfx_context);
  clear_menu_backend (gfx_context);
  clear_view_backend (gfx_context);

  mod_gl_utils_unload_data (&(gfx_context->utils_context));

  glFinish ();
  TTF_Quit ();

  SDL_QuitSubSystem (SDL_INIT_VIDEO);
  //SDL_QuitSubSystem(SDL_INIT_JOYSTICK);
  SDL_QuitSubSystem (SDL_INIT_TIMER);

  if (lw6sys_sdl_unregister ())
    {
      lw6sys_log (LW6SYS_LOG_INFO, "gfx-mod-gl", _("SDL Quit"));
      SDL_Quit ();
    }

  /*
   * For some reason, I suspect some segfaults occur when
   * "dlclosing" mod-gl just after SDL_Quit. Might be a handler
   * or callback called afterwards, whatever. So I prefer
   * "wasting" a little time when closing, one never knows,
   * it might better things.
   */
  lw6sys_sleep (quit_sleep);

  if (gfx_context->utils_context.surface_counter.new_counter !=
      gfx_context->utils_context.surface_counter.delete_counter)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _
		  ("possible memory leak, %d calls to SDL_CreateSurface, IMG_Load or TTF_RenderUTF8, and %d calls to SDL_FreeSurface"),
		  gfx_context->utils_context.surface_counter.
		  new_counter,
		  gfx_context->utils_context.surface_counter.delete_counter);
    }

  if (gfx_context->utils_context.texture_counter.new_counter !=
      gfx_context->utils_context.texture_counter.delete_counter)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _
		  ("possible memory leak, %d calls to glGenTexture and %d calls to glDeleteTextures"),
		  gfx_context->utils_context.texture_counter.
		  new_counter,
		  gfx_context->utils_context.texture_counter.delete_counter);
    }

  LW6SYS_FREE (gfx_context);
}
