/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <string.h>

#include "config.h"
#include "ker.h"
#include "ker-internal.h"

/*
 * This is used to stamp game_structs as they are created.
 * Note that there's no race condition due to the fact that this
 * is global here, in fact even when 2 processes would share
 * this sequence id, it would not matter for they would then
 * try and identify the objects in their on per-process lists,
 * structures, Guile object, whatever they use.
 */
static int seq_id = 0;

LW6KER_GAME_STATE *
lw6ker_game_state_new (LW6KER_GAME_STRUCT * game_struct)
{
  LW6KER_GAME_STATE *ret = NULL;

  ret = (LW6KER_GAME_STATE *) LW6SYS_CALLOC (sizeof (LW6KER_GAME_STATE));
  if (ret)
    {
      ret->id = ++seq_id;
      ret->game_struct = game_struct;
      lw6opt_dynamic_init (&(ret->options), ret->game_struct->options);
      _lw6ker_map_state_init (&(ret->map), &(ret->game_struct->map),
			      &(ret->game_struct->options));
    }

  return ret;
}

void
lw6ker_game_state_free (LW6KER_GAME_STATE * game_state)
{
  /*
   * IMPORTANT NOTE: it's important *not* to reference game_struct
   * in this free function. The idea is that game_struct can be
   * freed *before* game_state. Why is that is linked with arcane
   * garbage collection wizardry. Theorically the cleanest way would
   * be to make Guile aware that a given game_state requires a game_struct.
   * In a first approach, it's easier (but quirk and dirty, that's true)
   * not to care about this reference stuff, and simply manage
   * things so that whenever we *use* game_state, game_struct is still
   * alive. But we don't control what the garbage collector does,
   * and once it decides to purge objects (we have no more refs on
   * them that is) it can purge them in any order...
   */
  _lw6ker_map_state_clear (&(game_state->map));

  LW6SYS_FREE (game_state);
}

int
lw6ker_game_state_memory_footprint (LW6KER_GAME_STATE * game_state)
{
  int ret = 0;

  // todo

  return ret;
}

char *
lw6ker_game_state_repr (LW6KER_GAME_STATE * game_state)
{
  char *ret = NULL;

  if (game_state)
    {
      ret =
	lw6sys_new_sprintf ("%d (%dx%d)", game_state->id,
			    game_state->game_struct->map.shape.w,
			    game_state->game_struct->map.shape.h);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "game_state",
		  _("can't generate string id for NULL game_state"));
    }

  return ret;
}

/*
 * Fundamental function, used to carbon copy a game state to another,
 * this is intensively used to keep too tracks of the game state, one
 * most-up-to-date but probably wrong, the one we use to display on the
 * screen, and one slightly outdated (or very outdated if network is
 * slow) but that we're sure of, something 100% bullet proof we can
 * rely on.
 */
LW6SYS_BOOL
lw6ker_game_state_copy (LW6KER_GAME_STATE * dst, LW6KER_GAME_STATE * src)
{
  LW6SYS_BOOL ret = 0;
  int i, j;

  /*
   * Sanity check, this function is designed to copy game states
   * which correspond to the same game struct. Any other use is
   * useless in LW6 context.
   */
  if (dst && src && dst->game_struct && src->game_struct
      && dst->game_struct == src->game_struct)
    {
      if (dst == src)
	{
	  /*
	   * Special case where source and destination are the
	   * same, this can happen when testing, or when
	   * handling demo maps for instance, in that case
	   * we just do... nothing!
	   */
	  ret = 1;
	}
      else
	{
	  ret = _lw6ker_map_state_copy (&dst->map, &src->map);
	  lw6opt_dynamic_copy (&dst->options, &src->options);
	  dst->moves = src->moves;
	  dst->spreads = src->spreads;
	  dst->rounds = src->rounds;
	  for (i = 0; i < LW6OPT_MAX_NB_TEAMS; ++i)
	    {
	      for (j = 0; j < LW6OPT_MAX_CURSORS_PER_TEAM; ++j)
		{
		  dst->bots[i][j] = src->bots[i][j];
		}
	    }
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		  _
		  ("game_state_copy only works if dst and src point to the same game_struct"));
    }

  return ret;
}

void
_lw6ker_game_state_update_checksum (LW6KER_GAME_STATE *
				    game_state, LW6SYS_UINT32 * checksum)
{
  int i, j;

  _lw6ker_game_struct_update_checksum (game_state->game_struct, checksum);
  _lw6ker_map_state_update_checksum (&(game_state->map), checksum);
  lw6opt_dynamic_update_checksum (&(game_state->options), checksum);
  lw6sys_checksum_update_int32 (checksum, game_state->moves);
  lw6sys_checksum_update_int32 (checksum, game_state->spreads);
  lw6sys_checksum_update_int32 (checksum, game_state->rounds);
  for (i = 0; i < LW6OPT_MAX_NB_TEAMS; ++i)
    {
      for (j = 0; j < LW6OPT_MAX_CURSORS_PER_TEAM; ++j)
	{
	  _lw6ker_bot_update_checksum (&(game_state->bots[i][j]), checksum);
	}
    }
}

LW6SYS_UINT32
lw6ker_game_state_checksum (LW6KER_GAME_STATE * game_state)
{
  LW6SYS_UINT32 ret = 0;

  _lw6ker_game_state_update_checksum (game_state, &ret);

  return ret;
}

LW6SYS_BOOL
lw6ker_game_state_add_team (LW6KER_GAME_STATE * game_state,
			    LW6SYS_INT32 team_id, LW6SYS_INT32 nb_cursors)
{
  LW6SYS_BOOL ret = 0;
  LW6SYS_INT32 i;
  /*
   * Some values as INT64 to force casts to long long and
   * avoid integer capacity overflow.
   */
  LW6SYS_INT64 total_fighters;
  LW6SYS_INT64 total_fighters_to_remove;

  if (!game_state->map.teams[team_id].active)
    {
      LW6SYS_INT32 nb_fighters;
      LW6SYS_XY desired_center;

      lw6ker_map_state_start_xy (&(game_state->map), &desired_center,
				 team_id);

      nb_fighters =
	lw6ker_per1000 (game_state->map.map_struct->room_for_armies,
			game_state->game_struct->options.single_army_surface);
      if (nb_fighters + game_state->map.armies.active_fighters >
	  game_state->map.armies.max_fighters)
	{
	  /*
	   * OK there will be too many fighters, now the rule is:
	   * a new team gets 1/n of total allowed fighters.
	   */
	  nb_fighters =
	    game_state->map.armies.max_fighters /
	    (lw6ker_map_state_get_nb_active_teams (&(game_state->map)) + 1);
	}
      /*
       * Remove fighters if needed
       */
      if (nb_fighters + game_state->map.armies.active_fighters >
	  game_state->map.armies.max_fighters)
	{
	  total_fighters = game_state->map.armies.active_fighters;
	  total_fighters_to_remove = nb_fighters - (game_state->map.armies.
						    max_fighters -
						    total_fighters);
	  for (i = 0; i < LW6OPT_MAX_NB_TEAMS; ++i)
	    {
	      lw6ker_map_state_remove_team_fighters (&(game_state->map),
						     i,
						     (total_fighters_to_remove
						      *
						      game_state->map.armies.
						      fighters_per_team[i]) /
						     total_fighters);
	    }
	  /*
	   * Note that this value has probably changed since we last
	   * calculated it...
	   */
	  total_fighters = game_state->map.armies.active_fighters;
	  total_fighters_to_remove = nb_fighters - (game_state->map.armies.
						    max_fighters -
						    total_fighters);
	  lw6ker_map_state_remove_fighters (&(game_state->map),
					    total_fighters_to_remove);
	}

      lw6ker_map_state_populate_team (&game_state->map, team_id,
				      nb_fighters, desired_center,
				      nb_cursors,
				      game_state->game_struct->options);
      ret = 1;
    }
  else
    {
      int free_team_id =
	lw6ker_map_state_get_free_team_id (&(game_state->map));

      if (free_team_id >= 0)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		      _
		      ("can't add team (team_id=%d), it's already active, however team %d is free"),
		      team_id, free_team_id);
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		      _
		      ("can't add team (team_id=%d) no free teams left, game is full"),
		      team_id);
	}
    }

  return ret;
}

LW6SYS_BOOL
lw6ker_game_state_remove_team (LW6KER_GAME_STATE * game_state,
			       LW6SYS_INT32 team_id)
{
  LW6SYS_BOOL ret = 0;

  if (game_state->map.teams[team_id].active)
    {
      LW6SYS_INT32 nb_active_teams;
      /*
       * Some values as INT64 to force casts to long long and
       * avoid integer capacity overflow.
       */
      LW6SYS_INT64 nb_active_fighters;
      LW6SYS_INT64 new_average_fighters_per_team;
      LW6SYS_INT64 new_active_fighters;
      LW6SYS_INT64 single_average_fighters_per_team;
      LW6SYS_INT64 total_fighters_to_distribute;
      LW6SYS_INT32 i;

      nb_active_teams =
	lw6ker_map_state_get_nb_active_teams (&(game_state->map));
      nb_active_fighters = game_state->map.armies.active_fighters;

      if (nb_active_teams <= 1)
	{
	  /*
	   * Weird situation, not sure it should happen, except when in
	   * a spectator mode -> not one single fighter left!
	   */
	  lw6ker_map_state_remove_fighters (&(game_state->map),
					    nb_active_fighters);
	}
      else
	{
	  new_average_fighters_per_team =
	    nb_active_fighters / (nb_active_teams - 1);
	  single_average_fighters_per_team =
	    lw6ker_per1000 (game_state->map.map_struct->room_for_armies,
			    game_state->game_struct->options.
			    single_army_surface);
	  if (new_average_fighters_per_team >
	      single_average_fighters_per_team)
	    {
	      lw6ker_map_state_remove_team_fighters (&(game_state->map),
						     team_id,
						     nb_active_fighters -
						     single_average_fighters_per_team
						     * (nb_active_teams - 1));
	    }

	  /*
	   * At this stage we have suppressed any fighter that needed
	   * to be, that is there are not enough teams to reach the
	   * absolute limit for fighters in this map.
	   */
	  new_active_fighters = game_state->map.armies.active_fighters;

	  /*
	   * Now, second step, we redistribute the remaining fighters 
	   * to other teams
	   */
	  total_fighters_to_distribute =
	    game_state->map.armies.fighters_per_team[team_id];
	  for (i = 0; i < LW6OPT_MAX_NB_TEAMS; ++i)
	    {
	      if (i != team_id)
		{
		  lw6ker_map_state_redistribute_team (&(game_state->map), i,
						      team_id,
						      (game_state->map.
						       armies.
						       fighters_per_team[i]
						       *
						       total_fighters_to_distribute)
						      / new_active_fighters,
						      game_state->
						      game_struct->options);
		}
	    }

	  /*
	   * We might get a few fighters left due to rounding/approximations,
	   * we get rid of them.
	   */
	  while (game_state->map.armies.fighters_per_team[team_id] > 0)
	    {
	      for (i = 0;
		   i < LW6OPT_MAX_NB_TEAMS
		   && game_state->map.armies.fighters_per_team[team_id] > 0;
		   ++i)
		{
		  if (game_state->map.teams[i].active)
		    {
		      lw6ker_map_state_redistribute_team (&(game_state->map),
							  i, team_id, 1,
							  game_state->
							  game_struct->
							  options);
		    }
		}
	    }

	  lw6ker_map_state_cancel_team (&(game_state->map), team_id);
	}

      ret = 1;
    }

  return ret;
}

static void
update_bots (LW6KER_GAME_STATE * game_state)
{
  int i, j;

  for (i = 0; i < LW6OPT_MAX_NB_TEAMS; ++i)
    {
      for (j = 0; j < LW6OPT_MAX_CURSORS_PER_TEAM; ++j)
	{
	  if (game_state->bots[i][j].active)
	    {
	      _lw6ker_bot_update_pos (&(game_state->bots[i][j]), game_state);
	    }
	}
    }
}

static void
update_cursors_with_bots (LW6KER_GAME_STATE * game_state)
{
  int i, j;

  for (i = 0; i < LW6OPT_MAX_NB_TEAMS; ++i)
    {
      for (j = 0; j < LW6OPT_MAX_CURSORS_PER_TEAM; ++j)
	{
	  if (game_state->bots[i][j].active)
	    {
	      if (j < game_state->map.teams[i].cursor_array.nb_cursors)
		{
		  game_state->map.teams[i].cursor_array.cursors[j].pos =
		    game_state->bots[i][j].pos;
		}
	      else
		{
		  lw6sys_log (LW6SYS_LOG_WARNING, "ker",
			      _
			      ("strange, bot %d/%d is active, however team %d has only %d cursors"),
			      i, j, i,
			      game_state->map.teams[i].cursor_array.
			      nb_cursors);
		}
	    }
	}
    }
}

/*
 * This is a fundamental function, it's called at each round,
 * it fires all the complex calculations in the game, the
 * real core algorithm. Every time this function is called,
 * the round is "over" and the game state is ready for the
 * next... round.
 */
void
lw6ker_game_state_do_round (LW6KER_GAME_STATE * game_state)
{
  update_bots (game_state);
  update_cursors_with_bots (game_state);

  lw6ker_map_state_spread_gradient (&(game_state->map),
				    &(game_state->game_struct->options),
				    game_state->game_struct->options.
				    spreads_per_round);

  lw6ker_map_state_move_fighters (&(game_state->map),
				  lw6sys_checksum_int32 (game_state->
							 rounds) %
				  LW6KER_NB_PARITIES,
				  &(game_state->game_struct->options),
				  game_state->game_struct->options.
				  moves_per_round);

  game_state->moves += game_state->game_struct->options.moves_per_round;
  game_state->spreads += game_state->game_struct->options.spreads_per_round;
  game_state->rounds++;
}

LW6SYS_UINT32
lw6ker_game_state_get_moves (LW6KER_GAME_STATE * game_state)
{
  return (game_state->moves);
}

LW6SYS_UINT32
lw6ker_game_state_get_spreads (LW6KER_GAME_STATE * game_state)
{
  return (game_state->spreads);
}

LW6SYS_UINT32
lw6ker_game_state_get_rounds (LW6KER_GAME_STATE * game_state)
{
  return (game_state->rounds);
}

LW6SYS_BOOL
lw6ker_game_state_enable_bot (LW6KER_GAME_STATE * game_state,
			      LW6SYS_INT32 team_id)
{
  LW6SYS_BOOL ret = 0;
  int i;

  if (team_id >= 0 && team_id < LW6OPT_MAX_NB_TEAMS
      && game_state->map.teams[team_id].active)
    {
      for (i = 0; i < game_state->map.teams[team_id].cursor_array.nb_cursors;
	   ++i)
	{
	  if (game_state->bots[team_id][i].active)
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
			  _
			  ("trying to enable bot %d on team %d, but it's already enabled"),
			  i, team_id);
	    }
	  _lw6ker_bot_init (&(game_state->bots[team_id][i]),
			    &(game_state->map.teams[team_id].cursor_array.
			      cursors[i]), game_state->rounds);
	}
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		  _("unable to enable bot on team %d"), team_id);
    }

  return ret;
}

LW6SYS_BOOL
lw6ker_game_state_disable_bot (LW6KER_GAME_STATE * game_state,
			       LW6SYS_INT32 team_id)
{
  LW6SYS_BOOL ret = 0;

  int i;

  if (team_id >= 0 && team_id < LW6OPT_MAX_NB_TEAMS
      && game_state->map.teams[team_id].active)
    {
      for (i = 0; i < game_state->map.teams[team_id].cursor_array.nb_cursors;
	   ++i)
	{
	  if (!game_state->bots[team_id][i].active)
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
			  _
			  ("trying to disable bot %d on team %d, but it's not disabled"),
			  i, team_id);
	    }
	  _lw6ker_bot_clear (&(game_state->bots[team_id][i]));
	}
      ret = 1;
    }
  else
    {
      /*
       * Note: we fire this message even if the only problem is that
       * the team is not active, the idea is that bots must be disabled
       * before the team is unactivated, final dot. We don't want ghost bots.
       */
      lw6sys_log (LW6SYS_LOG_WARNING, "ker",
		  _("unable to disable bot on team %d"), team_id);
    }

  return ret;
}
