/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>

#include <libpng12/png.h>

#include "config.h"
#include "map.h"
#include "map-internal.h"

void
lw6map_depth_block_up_down (LW6MAP_DEPTH * depth)
{
  int x;

  for (x = 0; x < depth->shape.w; ++x)
    {
      lw6map_depth_set (depth, x, 0, 0);
      lw6map_depth_set (depth, x, depth->shape.h - 1, 0);
    }
}

void
lw6map_depth_block_left_right (LW6MAP_DEPTH * depth)
{
  int y;

  for (y = 0; y < depth->shape.h; ++y)
    {
      lw6map_depth_set (depth, 0, y, 0);
      lw6map_depth_set (depth, depth->shape.w - 1, y, 0);
    }
}

static int
read_png (LW6MAP_DEPTH * depth, char *map_dot_png,
	  LW6OPT_STATIC * static_options)
{
  int ret = 0;
  _LW6MAP_IMAGE_BW image;

  memset (&image, 0, sizeof (_LW6MAP_IMAGE_BW));

  if (_lw6map_bw_read (&image, map_dot_png))
    {
      LW6MAP_RESAMPLER resampler;
      lw6map_resampler_init (&resampler, static_options, &(image.shape));

      depth->shape = resampler.target_shape;
      depth->data =
	(unsigned char *) LW6SYS_MALLOC (depth->shape.w * depth->shape.h *
					 sizeof (unsigned char *));

      if (depth->data)
	{
	  int row, col, row2, col2;
	  unsigned int value;

	  for (row = 0; row < depth->shape.h; ++row)
	    {
	      for (col = 0; col < depth->shape.w; ++col)
		{
		  lw6map_resampler_target2source (&resampler, &col2, &row2,
						  col, row);

		  value = image.data[row2][col2 * image.step];
		  /*
		   * For now we do not handle depth the full way. There's
		   * a big TODO here, one could read in another file (say
		   * an XML file) a parameter that would state "maximum depth
		   * is 5" and maybe even the various thresholds for the
		   * different levels. For now using depth 0 or 1 is just like
		   * in Liquid War 5, we have flat levels, will be fine for
		   * a some time anyway.
		   */
		  value = (value > 127) ? 1 : 0;
		  lw6map_depth_set (depth, col, row, value);
		}
	    }

	  ret = 1;
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_ERROR, "map",
		      _("unable to allocate memory for depth"));
	}

      _lw6map_bw_clear (&image);
    }

  if (!ret)
    {
      lw6map_depth_clear (depth);
    }

  return ret;
}

static int
find_first_free_point (LW6MAP_DEPTH * depth, int *found_x, int *found_y)
{
  int ret = 0;
  int x, y;

  *found_x = 0;
  *found_y = 0;
  for (y = 1; y < depth->shape.h - 1 && !ret; ++y)
    {
      for (x = 1; x < depth->shape.w - 1 && !ret; ++x)
	{
	  if (lw6map_depth_get (depth, x, y) > 0)
	    {
	      ret = 1;
	      *found_x = x;
	      *found_y = y;
	    }
	}
    }

  return ret;
}

/*
 * Updates a depth point if needed, returns true if value was changed
 */
static int
update_if_needed (LW6MAP_DEPTH * dst, LW6MAP_DEPTH * src, int x, int y)
{
  int ret = 0;

  if (lw6map_depth_get (dst, x, y) != lw6map_depth_get (src, x, y))
    {
      lw6map_depth_set (dst, x, y, lw6map_depth_get (src, x, y));
      ret = 1;
    }

  return ret;
}

/*
 * Checks the maps for "holes", that is fight zones that wouldn't
 * be connected with the main area, and fills them with walls.
 */
static int
check_and_fix_holes (LW6MAP_DEPTH * depth)
{
  int ret = 0;
  LW6MAP_DEPTH fixed_depth;
  int x, y;

  fixed_depth.shape.w = depth->shape.w;
  fixed_depth.shape.h = depth->shape.h;
  fixed_depth.data =
    (unsigned char *) LW6SYS_CALLOC (depth->shape.w * depth->shape.h *
				     sizeof (unsigned char *));
  fixed_depth.max = depth->max;

  if (fixed_depth.data)
    {
      int found;

      /*
       * We first set one point, which will spread all over.
       */
      ret = find_first_free_point (depth, &x, &y)
	&& update_if_needed (&fixed_depth, depth, x, y);
      found = ret ? 1 : 0;

      while (found)
	{
	  found = 0;

	  for (y = 1; y < depth->shape.h - 1; ++y)
	    {
	      for (x = 1; x < depth->shape.w - 1; ++x)
		{
		  if (lw6map_depth_get (&fixed_depth, x, y) > 0)
		    {
		      found +=
			update_if_needed (&fixed_depth, depth, x + 1, y);
		      found +=
			update_if_needed (&fixed_depth, depth, x + 1, y + 1);
		      found +=
			update_if_needed (&fixed_depth, depth, x, y + 1);
		      found +=
			update_if_needed (&fixed_depth, depth, x - 1, y + 1);
		    }
		}
	    }

	  for (y = depth->shape.h - 2; y >= 1; --y)
	    {
	      for (x = depth->shape.w - 2; x >= 1; --x)
		{
		  if (lw6map_depth_get (&fixed_depth, x, y) > 0)
		    {
		      found +=
			update_if_needed (&fixed_depth, depth, x - 1, y);
		      found +=
			update_if_needed (&fixed_depth, depth, x - 1, y - 1);
		      found +=
			update_if_needed (&fixed_depth, depth, x, y - 1);
		      found +=
			update_if_needed (&fixed_depth, depth, x + 1, y - 1);
		    }
		}
	    }
	}
      LW6SYS_FREE (depth->data);
      depth->data = fixed_depth.data;
    }

  return ret;
}

/*
 * Read a depth map. Pointer to depth must be valid,
 * it's modified in-place.
 */
int
lw6map_depth_read (LW6MAP_DEPTH * depth, char *dirname,
		   LW6OPT_STATIC * static_options)
{
  int ret = 0;
  char *map_dot_png;

  lw6map_depth_clear (depth);

  map_dot_png = lw6sys_str_concat (dirname, LW6MAP_FILE_MAP_PNG);
  if (map_dot_png)
    {
      ret = read_png (depth, map_dot_png, static_options);
      if (ret)
	{
	  if (static_options->x_polarity == 0)
	    {
	      lw6map_depth_block_left_right (depth);
	    }
	  if (static_options->y_polarity == 0)
	    {
	      lw6map_depth_block_up_down (depth);
	    }
	  ret = check_and_fix_holes (depth);
	}

      LW6SYS_FREE (map_dot_png);
    }

  return ret;
}

/*
 * Clears a depth map.
 */
void
lw6map_depth_clear (LW6MAP_DEPTH * depth)
{
  if (depth->data)
    {
      LW6SYS_FREE (depth->data);
    }

  memset (depth, 0, sizeof (LW6MAP_DEPTH));
}

void
lw6map_depth_fill (LW6MAP_DEPTH * depth, unsigned char value)
{
  int x, y;

  for (y = 0; y < depth->shape.h; ++y)
    {
      for (x = 0; x < depth->shape.w; ++x)
	{
	  lw6map_depth_set (depth, x, y, value);
	}
    }
}
