/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "sys.h"

#define CHECKSUM_POLY 0x04c11db7

/*
 * Creates a checksum from byte array. This could be mapped on
 * any standard CRC-32 and/or MD5 algorithm, but licence issues
 * for those are such a headache that for the sake of simplicity,
 * it's wrapped here. In LW5 context, we do not really really
 * fear any attack for these checksums are used internally to
 * track bugs and check, for instance, that two game states are
 * actually the same on two distant computers in a network game.
 * Data encryption and security of network links is another debate.
 * Additionnally, this function returns an integer, easier to
 * handle in standard C than any malloc'ed stuff.
 */
LW6SYS_UINT32
lw6sys_checksum (unsigned char *data, int len)
{
  LW6SYS_UINT32 ret = 0;

  lw6sys_checksum_update (&ret, data, len);

  return ret;
}

LW6SYS_UINT32
lw6sys_checksum_int32 (LW6SYS_UINT32 value)
{
  LW6SYS_UINT32 ret = 0;

  lw6sys_checksum_update_int32 (&ret, value);

  return ret;
}

LW6SYS_UINT32
lw6sys_checksum_wh (LW6SYS_WH * wh)
{
  LW6SYS_UINT32 ret = 0;

  lw6sys_checksum_update_wh (&ret, wh);

  return ret;
}

LW6SYS_UINT32
lw6sys_checksum_xy (LW6SYS_XY * xy)
{
  LW6SYS_UINT32 ret = 0;

  lw6sys_checksum_update_xy (&ret, xy);

  return ret;
}

/*
 * Updates a checksum with more data.
 */
void
lw6sys_checksum_update (LW6SYS_UINT32 * checksum, unsigned char *data,
			int len)
{
  /*
   * Algorithm copied from Linux kernel source (lib/crc32.c).
   * The code used to be public domain, was GPL'ed, and is however
   * way less than 15 lines long... We don't care wether it's a real
   * bullet proof CRC32, all we want is consistent calculus on
   * every platform, hopefully this code fits.
   */
  int i;
  while (len--)
    {
      (*checksum) ^= ((LW6SYS_UINT32) (*data++)) << 24;
      for (i = 0; i < 8; i++)
	(*checksum) =
	  ((*checksum) << 1) ^ (((*checksum) & 0x80000000) ? CHECKSUM_POLY :
				0);
    }
}

void
lw6sys_checksum_update_int32 (LW6SYS_UINT32 * checksum, LW6SYS_INT32 value)
{
  unsigned char buffer[LW6SYS_SIZEOF_INT32];

  lw6sys_serialize_int32 (buffer, value);
  lw6sys_checksum_update (checksum, buffer, LW6SYS_SIZEOF_INT32);
}

void
lw6sys_checksum_update_wh (LW6SYS_UINT32 * checksum, LW6SYS_WH * wh)
{
  lw6sys_checksum_update_int32 (checksum, wh->w);
  lw6sys_checksum_update_int32 (checksum, wh->h);
}

void
lw6sys_checksum_update_xy (LW6SYS_UINT32 * checksum, LW6SYS_XY * xy)
{
  lw6sys_checksum_update_int32 (checksum, xy->x);
  lw6sys_checksum_update_int32 (checksum, xy->y);
}
