/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include "config.h"
#include "dyn.h"

static lt_dlhandle
verbose_dlopen (char *file)
{
  lt_dlhandle ret = NULL;
  int nb_errs = 0;

  if ((nb_errs = lt_dlinit ()) > 0)
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("couldn't initialize libltdl, %d errors"), nb_errs);
    }
  else
    {
      ret = lt_dlopen (file);
    }

  if (ret)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("dlopen \"%s\""), file);
    }
  else
    {
      if (lw6sys_file_exists (file))
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("couldn't open shared library \"%s\""), file);
	}
      else
	{
	  /*
	   * File doesn't exist, different error message, and not
	   * as a warning, only an information.
	   */
	  lw6sys_log (LW6SYS_LOG_INFO,
		      _("shared library \"%s\" doesn't exist"), file);
	}
    }

  return ret;
}

/**
 * lw6dyn_dlopen_backend:
 *
 * @argc: the number of command-line arguments as passed to @main
 * @arvg: an array of command-line arguments as passed to @main
 * @top_level_lib: the top-level library concerned, this means is it
 *   "cli", "gfx", "snd" or "srv". This will tell the function to search
 *   for the .so file in the correct subdirectory. Think of this as a
 *   category.
 * @backend: the actual name of the backend, this is the name of the
 *   .so file, between "libmod_" and ".so". For instance, to find
 *   "libmod_gl.so", the right argument is "gl".
 *
 * Opens a .so file corresponding to the given backend,
 * it is capable to search for system libraries installed after "make install"
 * but if not found, it will also search in the current build directory,
 * finding the .so files in hidden .libs subdirectories.
 *
 * Return value: a handle to the module, once it's opened. You might still
 *   need to call a module specific @init() function, but it's another story.
 */
lw6dyn_dl_handle_t *
lw6dyn_dlopen_backend (int argc, char *argv[], char *top_level_lib,
		       char *backend_name)
{
  lw6dyn_dl_handle_t *ret = NULL;

  ret = LW6SYS_CALLOC (sizeof (lw6dyn_dl_handle_t));
  if (ret)
    {
      ret->library_path =
	lw6dyn_path_find_backend (argc, argv, top_level_lib, backend_name);
      if (ret->library_path)
	{
	  ret->handle = verbose_dlopen (ret->library_path);
	  if (!ret->handle)
	    {
	      LW6SYS_FREE (ret->library_path);
	      LW6SYS_FREE (ret);
	      ret = NULL;
	    }
	}
      else
	{
	  LW6SYS_FREE (ret);
	  ret = NULL;
	}
    }

  return ret;
}

int
lw6dyn_dlclose_backend (lw6dyn_dl_handle_t * handle)
{
  int nb_errs;
  int ret = 0;

  if (handle)
    {
      /*
       * Just for this log line, it's interesting to wrap
       * the libtool lt_dlclose in our own function: when freeing
       * a loaded shared library, we know *what* we're freeing,
       * which can be pretty usefull when debugging.
       */
      lw6sys_log (LW6SYS_LOG_INFO, _("dlclose \"%s\""), handle->library_path);
      if (handle->handle)
	{
	  if (lt_dlclose (handle->handle))
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _("couldn't unload shared library"));
	      ret = 1;
	    }
	}

      if ((nb_errs = lt_dlexit ()) > 0)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("couldn't exit libltdl, %d errors"), nb_errs);
	  ret = 0;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("trying to free NULL dl_handle"));
    }

  LW6SYS_FREE (handle->library_path);
  LW6SYS_FREE (handle);

  return ret;
}

void *
lw6dyn_dlsym (lw6dyn_dl_handle_t * handle, char *func_name)
{
  return lt_dlsym (handle->handle, func_name);
}
