/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>

#include "config.h"
#include "gui.h"

/*
 * This is used to stamp menuitems as they are created.
 * Theorically, there could be a race condition if several threads used
 * this, in practice, very rare, and, moreover, few consequences.
 */
static int seq_id = 0;

/**
 * lw6gui_menuitem_new
 *
 * @label: the string to be displayed, what the user sees. Can be freed after
 *   the call is done, function will make a copy internally.
 * @value: the value. No GUI function uses this, this is the "real" value
 *   associated to the item.
 * @enabled: wether the menu item can be selected, used, and so on
 * @selected: wether the menu item is the item selected among all menu items.
 * @colored: wetherr the menu item must, when drawn, be colored according
 *   to its value.
 *
 * Constructs a new menuitem object. Note that you can always call other
 * functions to modify these values afterwards, this might change rendering
 * since @lw6gui_menuitem_set_value or @lw6gui_menuitem_set_label will,
 * for instance, modify the "when was that item last modified" information.
 *
 * Return value: a pointer to the newly allocated object.
 */
lw6gui_menuitem_t *
lw6gui_menuitem_new (char *label, int value, int enabled, int selected,
		     int colored)
{
  lw6gui_menuitem_t *menuitem = NULL;

  menuitem = (lw6gui_menuitem_t *) LW6SYS_CALLOC (sizeof (lw6gui_menuitem_t));
  if (menuitem)
    {
      menuitem->label = lw6sys_str_copy (label);
      if (menuitem->label)
	{
	  menuitem->id = ++seq_id;
	  menuitem->value = value;
	  menuitem->enabled = enabled;
	  menuitem->selected = selected;
	  menuitem->colored = colored;
	}
      else
	{
	  LW6SYS_FREE (menuitem);
	  menuitem = NULL;
	}
    }

  return menuitem;
}

/**
 * lw6gui_menuitem_free
 *
 * @menuitem: a pointer to the menuitem.
 *
 * Frees the menuitem, checking if things are OK before doing so.
 *
 * Return value: none.
 */
void
lw6gui_menuitem_free (lw6gui_menuitem_t * menuitem)
{
  if (menuitem)
    {
      if (menuitem->label)
	{
	  LW6SYS_FREE (menuitem->label);
	  menuitem->label = NULL;
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, _("menuitem with NULL label"));
	}
      LW6SYS_FREE (menuitem);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("trying to free NULL menuitem"));
    }
}

/**
 * lw6gui_menuitem_memory_footprint
 *
 * @menuitem: a pointer to the menuitem.
 *
 * Gets the memory occupied by the menuitem. Could be usefull to help
 * a garbage collector taking decisions or reporting erros, for instance.
 *
 * Return value: the number of bytes used.
 */
int
lw6gui_menuitem_memory_footprint (lw6gui_menuitem_t * menuitem)
{
  int memory_footprint = 0;

  if (menuitem)
    {
      memory_footprint += sizeof (lw6gui_menuitem_t);
      memory_footprint += strlen (menuitem->label) + 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("calling memory_footprint on NULL menuitem"));
    }

  return memory_footprint;
}

/**
 * lw6gui_menuitem_repr
 *
 * @menuitem: a pointer to the menuitem.
 *
 * Constructs a readable description of the object. Usefull for
 * debugging, or to introspect things using scripts, at run-time.
 * Does not necessarly describe all the informations about the
 * object, but helps knowing what it is.
 *
 * Return value: a string describing the object, must be freed.
 */
char *
lw6gui_menuitem_repr (lw6gui_menuitem_t * menuitem)
{
  char *repr;

  repr =
    lw6sys_new_sprintf (_("%d \"%s\" (value=%d)"), menuitem->id,
			menuitem->label, menuitem->value);

  return repr;
}

/**
 * lw6gui_menuitem_set_label
 *
 * @menuitem: a pointer to the menuitem.
 * @label: the new label, you can free it after calling the function,
 *   an internal copy will be made. 
 * @now: the current time, as a timestamp.
 *
 * Change the label of the menu item. That is to say, what the user sees.
 * Use this function to change the menuitem 
 * value, don't try to access the struct directly. The idea is 1) to have safe 
 * memory management and 2) to keep the @last_change member up to date. 
 * It can be later used for eye-candy effects.
 *
 * Return value: none
 */
void
lw6gui_menuitem_set_label (lw6gui_menuitem_t * menuitem, char *label, int now)
{
  if (strcmp (label, menuitem->label))
    {
      menuitem->last_change = now;
    }
  LW6SYS_FREE (menuitem->label);
  menuitem->label = lw6sys_str_copy (label);
  if (!(menuitem->label))
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("couldn't set menu item label \"%s\""), label);
    }
}

/**
 * lw6gui_menuitem_set_value
 *
 * @menuitem: a pointer to the menuitem.
 * @label: the new value.
 * @now: the current time, as a timestamp.
 * 
 * Changes the value of a menuitem. This is the internal value, not what
 * the user sees. Use this function to change the menuitem 
 * value, don't try to access the struct directly. The idea is to keep 
 * the @last_change member up to date. 
 * It can be later used for eye-candy effects.
 *
 * Return value: none
 */
void
lw6gui_menuitem_set_value (lw6gui_menuitem_t * menuitem, int value, int now)
{
  if (value != menuitem->value)
    {
      menuitem->last_change = now;
    }
  menuitem->value = value;
}

/**
 * lw6gui_menuitem_select
 *
 * @menuitem: a pointer to the menuitem.
 * @now: the current time, as a timestamp.
 * 
 * Switches the menuitem to selected state. Use this function, don't try
 * to modify the struct members directly. The idea is to have
 * the @last_select parameter up to date.
 * It can be later used for eye-candy effects.
 *
 * Return value: none
 */
void
lw6gui_menuitem_select (lw6gui_menuitem_t * menuitem, int now)
{
  if (!(menuitem->selected))
    {
      menuitem->selected = 1;
      menuitem->last_select = now;
    }
}

/**
 * lw6gui_menuitem_select
 *
 * @menuitem: a pointer to the menuitem.
 * @now: the current time, as a timestamp.
 * 
 * Switches the menuitem to unselected state. Use this function, don't try
 * to modify the struct members directly. The idea is to have
 * the @last_unselect parameter up to date.
 * It can be later used for eye-candy effects.
 *
 * Return value: none
 */
void
lw6gui_menuitem_unselect (lw6gui_menuitem_t * menuitem, int now)
{
  if (menuitem->selected)
    {
      menuitem->selected = 0;
      menuitem->last_unselect = now;
    }
}

/**
 * lw6gui_menuitem_checksum
 *
 * @menuitem: the menuitem we want to identify
 *
 * Returns a checksum which can be used to know, for
 * instance, wether the menuitem has changed or not, and if we should
 * redraw it.
 *
 * Return value: a checksum.
 */
int
lw6gui_menuitem_checksum (lw6gui_menuitem_t * menuitem)
{
  int ret;

  ret =
    lw6sys_checksum ((unsigned char *) menuitem, sizeof (lw6gui_menuitem_t));

  return ret;
}
