/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>
#include <math.h>

#include "config.h"
#include "ker.h"
#include "ker-internal.h"

typedef struct polarity_data_s
{
  int32_t fighter_x;
  int32_t fighter_y;
  int32_t map_w;
  int32_t map_h;
} polarity_data_t;

int
_lw6ker_map_state_init (lw6ker_map_state_t * map_state,
			lw6ker_map_struct_t * map_struct,
			lw6map_options_t * options)
{
  int ret = 0;
  int32_t i;

  map_state->map_struct = map_struct;
  map_state->shape = map_struct->shape;

  ret = 1;

  ret = ret
    && _lw6ker_armies_init (&(map_state->armies), map_struct, options);

  map_state->nb_layers = map_struct->max_depth;
  for (i = 0; i < map_state->nb_layers; ++i)
    {
      ret = ret
	&& _lw6ker_layer_init (&(map_state->layers[i]), map_state->map_struct,
			       &(map_state->armies), options);
    }

  map_state->max_nb_teams = options->max_nb_teams;
  for (i = 0; i < map_state->max_nb_teams; ++i)
    {
      ret = ret
	&& _lw6ker_team_init (&(map_state->teams[i]), map_struct, options);
    }

  return ret;
}

void
_lw6ker_map_state_clear (lw6ker_map_state_t * map_state)
{
  int32_t i;

  for (i = 0; i < map_state->max_nb_teams; ++i)
    {
      _lw6ker_team_clear (&(map_state->teams[i]));
    }

  for (i = 0; i < map_state->nb_layers; ++i)
    {
      _lw6ker_layer_clear (&(map_state->layers[i]));
    }
  map_state->nb_layers = 0;

  _lw6ker_armies_clear (&(map_state->armies));

  map_state->shape.w = 0;
  map_state->shape.h = 0;
  map_state->map_struct = NULL;
}

int
_lw6ker_map_state_copy (lw6ker_map_state_t * dst, lw6ker_map_state_t * src)
{
  int ret = 0;

  if (dst && src && dst->map_struct && src->map_struct
      && dst->map_struct == src->map_struct)
    {
      int i;

      dst->shape = src->shape;
      ret = _lw6ker_armies_copy (&(dst->armies), &(src->armies));
      dst->max_nb_teams = src->max_nb_teams;
      for (i = 0; i < src->max_nb_teams; ++i)
	{
	  ret = ret && _lw6ker_team_copy (&(dst->teams[i]), &(src->teams[i]));
	}
      dst->nb_layers = src->nb_layers;
      for (i = 0; i < src->nb_layers; ++i)
	{
	  ret = ret
	    && _lw6ker_layer_copy (&(dst->layers[i]), &(src->layers[i]));
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("map_state_copy only works if dst and src point to the same map_struct"));
    }

  return ret;
}

void
_lw6ker_map_state_update_checksum (lw6ker_map_state_t * map_state,
				   u_int32_t * checksum)
{
  int i;
  /*
   * No need to compute map_struct checksum again, it's done
   * within game_state_update_checksum which itself calls 
   * game_struct_update_checksum.
   */
  _lw6ker_armies_update_checksum (&(map_state->armies), checksum);
  lw6sys_checksum_update_int32 (checksum, map_state->max_nb_teams);
  for (i = 0; i < map_state->max_nb_teams; ++i)
    {
      _lw6ker_team_update_checksum (&(map_state->teams[i]), checksum);
    }
  lw6sys_checksum_update_int32 (checksum, map_state->nb_layers);
  for (i = 0; i < map_state->nb_layers; ++i)
    {
      _lw6ker_layer_update_checksum (&(map_state->layers[i]), checksum);
    }
}

int32_t
lw6ker_map_state_get_free_team_id (lw6ker_map_state_t * map_state)
{
  int32_t i;
  int32_t ret = -1;

  for (i = 0; i < LW6MAP_MAX_NB_TEAMS && ret < 0; ++i)
    {
      if (!map_state->teams[i].active)
	{
	  ret = i;
	}
    }

  return ret;
}

void
lw6ker_map_state_start_xy (lw6ker_map_state_t * map_state,
			   lw6sys_xy_t * start_xy, int32_t team_id)
{
  int32_t angle;
  int32_t team_ip, team_fp;

  team_ip = (team_id / 2) * 2;	// unoptimized, but well, is "& 0xFFFFFFFE" cleaner? 
  team_fp = team_id % 2;

  angle =
    (LW6KER_TRIGO_PI * team_ip) / LW6MAP_MAX_NB_TEAMS +
    LW6KER_TRIGO_PI * team_fp + (5 * LW6KER_TRIGO_PI) / 4;
  start_xy->x =
    map_state->shape.w / 2 +
    (lw6ker_cos (angle) * ((map_state->shape.w / 2) - 1)) /
    LW6KER_TRIGO_RADIUS;
  start_xy->y =
    map_state->shape.h / 2 +
    (lw6ker_sin (angle) * ((map_state->shape.h / 2) - 1)) /
    LW6KER_TRIGO_RADIUS;
}

int32_t
lw6ker_map_state_get_nb_active_teams (lw6ker_map_state_t * map_state)
{
  int32_t i;
  int32_t ret = 0;

  for (i = 0; i < LW6MAP_MAX_NB_TEAMS; ++i)
    {
      if (map_state->teams[i].active)
	{
	  ret++;
	}
    }

  return ret;
}

int32_t
lw6ker_map_state_populate_team (lw6ker_map_state_t * map_state,
				int32_t team_id,
				int32_t nb_fighters,
				lw6sys_xy_t desired_center,
				int32_t nb_cursors, lw6map_options_t options)
{
  lw6sys_xy_t real_center;
  int32_t angle, radius;
  int32_t max_radius, max_angle;
  int32_t x, y;
  int32_t depth;
  int32_t nb_fighters_added = 0;
  int32_t layer;
  int32_t new_fighter_id;
  lw6ker_fighter_t new_fighter;

  _lw6ker_fighter_clear (&new_fighter);
  lw6ker_map_struct_find_free_slot_near (map_state->map_struct, &real_center,
					 desired_center);

  max_radius = map_state->map_struct->shape.w + map_state->map_struct->shape.h;	// +, not *
  for (radius = 1; radius < max_radius && nb_fighters_added < nb_fighters;
       ++radius)
    {
      max_angle = radius * M_PI * 2;
      for (angle = 0; angle < max_angle && nb_fighters_added < nb_fighters;
	   ++angle)
	{
	  x =
	    real_center.x +
	    (lw6ker_cos ((angle * LW6KER_TRIGO_2PI) / max_angle) * radius) /
	    LW6KER_TRIGO_RADIUS;
	  y =
	    real_center.y -
	    (lw6ker_sin ((angle * LW6KER_TRIGO_2PI) / max_angle) * radius) /
	    LW6KER_TRIGO_RADIUS;
	  if (x >= 0 && x < map_state->map_struct->shape.w && y >= 0
	      && y < map_state->map_struct->shape.h)
	    {
	      depth =
		lw6ker_map_struct_get_depth (map_state->map_struct, x, y);
	      for (layer = 0; layer < map_state->nb_layers && layer < depth;
		   ++layer)
		{
		  if (lw6ker_map_state_get_fighter_id (map_state, layer, x, y)
		      < 0)
		    {
		      new_fighter.team_id = team_id;
		      new_fighter.layer = layer;
		      new_fighter.pos.x = x;
		      new_fighter.pos.y = y;
		      new_fighter.health = LW6MAP_MAX_FIGHTER_HEALTH;
		      new_fighter.last_direction = 0;
		      new_fighter_id =
			lw6ker_armies_add_fighter (&(map_state->armies),
						   new_fighter);
		      if (new_fighter_id >= 0)
			{
			  nb_fighters_added++;
			  lw6ker_map_state_set_fighter_id (map_state, layer,
							   x, y,
							   new_fighter_id);
			}
		    }
		}
	    }
	}
    }

  lw6ker_team_activate (&(map_state->teams[team_id]), nb_cursors,
			real_center);

  return nb_fighters_added;
}

int
lw6ker_map_state_redistribute_team (lw6ker_map_state_t * map_state,
				    int32_t dst_team_id,
				    int32_t src_team_id,
				    int32_t nb_fighters,
				    lw6map_options_t options)
{
  int ret = 0;
  int32_t i, j;
  int32_t nb_fighters_redistributed = 0;

  if (nb_fighters <= map_state->armies.fighters_per_team[src_team_id])
    {
      while (nb_fighters_redistributed < nb_fighters)
	{
	  for (i = 0;
	       i < LW6MAP_MAX_NB_TEAMS
	       && nb_fighters_redistributed < nb_fighters; ++i)
	    {
	      for (j = i;
		   j < map_state->armies.active_fighters
		   && nb_fighters_redistributed < nb_fighters;
		   j += LW6MAP_MAX_NB_TEAMS)
		{
		  if (map_state->armies.fighters[j].team_id == src_team_id)
		    {
		      map_state->armies.fighters[j].team_id = dst_team_id;
		      (map_state->armies.fighters_per_team[src_team_id])--;
		      (map_state->armies.fighters_per_team[dst_team_id])++;
		      map_state->armies.fighters[j].health =
			LW6MAP_MAX_FIGHTER_HEALTH;
		      nb_fighters_redistributed++;
		    }
		}
	    }
	}
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("can't redistribute %d fighters from team %d which has only %d"),
		  nb_fighters, src_team_id,
		  map_state->armies.fighters_per_team[src_team_id]);
    }

  return ret;
}

int
lw6ker_map_state_cancel_team (lw6ker_map_state_t * map_state, int32_t team_id)
{
  int ret = 0;

  if (map_state->teams[team_id].active)
    {
      if (map_state->armies.fighters_per_team[team_id] == 0)
	{
	  lw6ker_team_unactivate (&(map_state->teams[team_id]));
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("trying to cancel non-zeroed team %d, still has %d fighters"),
		      team_id, map_state->armies.fighters_per_team[team_id]);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("trying to cancel inactive team %d"), team_id);
    }

  return ret;
}

int
lw6ker_map_state_remove_fighter (lw6ker_map_state_t * map_state,
				 int32_t fighter_id)
{
  int ret = 0;
  int32_t last_fighter_id;

  last_fighter_id = map_state->armies.active_fighters - 1;
  if (fighter_id >= 0 && fighter_id <= last_fighter_id)
    {
      lw6ker_fighter_t fighter;
      lw6ker_fighter_t last_fighter;

      fighter = map_state->armies.fighters[fighter_id];
      last_fighter = map_state->armies.fighters[last_fighter_id];

      if (fighter_id < last_fighter_id)
	{
	  lw6ker_map_state_set_fighter_id (map_state, last_fighter.layer,
					   last_fighter.pos.x,
					   last_fighter.pos.y, fighter_id);
	  /*
	   * It's important to really *exchange* the fighters, for
	   * deletion in the armies struct will read the last fighter's
	   * team_id for instance, to maintain the list of active
	   * fighters in each team. So affecting the remaining one
	   * is not enough.
	   */
	  map_state->armies.fighters[fighter_id] = last_fighter;
	  map_state->armies.fighters[last_fighter_id] = fighter;
	}

      lw6ker_map_state_set_fighter_id (map_state, fighter.layer,
				       fighter.pos.x, fighter.pos.y, -1);
      lw6ker_armies_remove_fighter (&(map_state->armies));

      ret = 1;
    }

  return ret;
}

int
lw6ker_map_state_remove_fighters (lw6ker_map_state_t * map_state,
				  int32_t nb_fighters)
{
  int32_t i, j;
  int ret = 0;
  int32_t nb_fighters_removed = 0;

  if (nb_fighters <= map_state->armies.active_fighters)
    {
      while (nb_fighters_removed < nb_fighters)
	{
	  /*
	   * To remove fighters, we simply pseudo-randomly
	   * pass the current map, trying not to process
	   * low-numbered fighters systematically to avoid
	   * disadvantaging "first" teams, and whenever we
	   * encounter a fighter, we delete it. We count on
	   * statistics so that deletion is globally fair
	   * and no team should especially be adavantaged
	   * by this. But it's true there's some luck and
	   * a slight probability that all of the fighters
	   * of a given team are deleted at once...
	   */
	  for (i = 0;
	       i < LW6MAP_MAX_NB_TEAMS && nb_fighters_removed < nb_fighters;
	       ++i)
	    {
	      for (j = i;
		   j < map_state->armies.active_fighters
		   && nb_fighters_removed < nb_fighters;
		   j += LW6MAP_MAX_NB_TEAMS)
		{
		  lw6ker_map_state_remove_fighter (map_state, j);
		  nb_fighters_removed++;
		}
	    }
	}
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("can't remove %d fighters, map only has %d"),
		  nb_fighters, map_state->armies.active_fighters);
    }

  return ret;
}

int
lw6ker_map_state_remove_team_fighters (lw6ker_map_state_t * map_state,
				       int32_t team_id, int32_t nb_fighters)
{
  int ret = 0;
  int32_t nb_fighters_removed = 0;
  int32_t i, j;

  nb_fighters =
    lw6sys_min (nb_fighters, map_state->armies.fighters_per_team[team_id]);

  while (nb_fighters_removed < nb_fighters)
    {
      for (i = 0;
	   i < LW6MAP_MAX_NB_TEAMS && nb_fighters_removed < nb_fighters; ++i)
	{
	  for (j = i;
	       j < map_state->armies.active_fighters
	       && nb_fighters_removed < nb_fighters; j += LW6MAP_MAX_NB_TEAMS)
	    {
	      if (map_state->armies.fighters[j].team_id == team_id)
		{
		  lw6ker_map_state_remove_fighter (map_state, j);
		  nb_fighters_removed++;
		}
	    }
	}
    }

  ret = 1;

  return ret;
}

lw6ker_fighter_t *
lw6ker_map_state_get_fighter_safe (lw6ker_map_state_t * map_state,
				   int32_t layer, int32_t x, int32_t y)
{
  lw6ker_fighter_t *ret = NULL;
  int fighter_id;

  if (layer >= 0 && layer < map_state->nb_layers && x >= 0
      && x <= map_state->shape.w && y >= 0 && y < map_state->shape.h)
    {
      fighter_id =
	map_state->layers[layer].slots[map_state->shape.w * y + x].fighter_id;
      if (fighter_id >= 0)
	{
	  ret = &(map_state->armies.fighters[fighter_id]);
	}
    }

  return ret;
};

void
lw6ker_map_state_print_debug (lw6ker_map_state_t * map_state)
{
  int32_t i;

  lw6sys_log (LW6SYS_LOG_DEBUG, _("active_fighters = %d"),
	      map_state->armies.active_fighters);
  lw6sys_log (LW6SYS_LOG_DEBUG, _("max_fighters = %d"),
	      map_state->armies.max_fighters);
  if (map_state->armies.active_fighters > 0)
    {
      for (i = 0; i < LW6MAP_MAX_NB_TEAMS; ++i)
	{
	  lw6sys_log (LW6SYS_LOG_DEBUG,
		      _("team %d has %d fighters (%2.1f%%)"), i,
		      map_state->armies.fighters_per_team[i],
		      ((float) map_state->armies.fighters_per_team[i]) /
		      ((float) map_state->armies.active_fighters) * 100.0f);
	}
    }
}

int
lw6ker_map_state_sanity_check (lw6ker_map_state_t * map_state)
{
  int ret = 1;
  int32_t i;
  int32_t real_fighters_per_team[LW6MAP_MAX_NB_TEAMS];
  int32_t fighter_id;
  lw6ker_fighter_t fighter;

  if (map_state->armies.active_fighters > map_state->armies.max_fighters)
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("active_fighters (%d) > max_fighters (%d)"),
		  map_state->armies.active_fighters,
		  map_state->armies.max_fighters);
      ret = 0;
    }
  for (i = 0; i < LW6MAP_MAX_NB_TEAMS; ++i)
    {
      real_fighters_per_team[i] = 0;
    }
  for (i = 0; i < map_state->armies.active_fighters; ++i)
    {
      fighter = map_state->armies.fighters[i];
      if (fighter.team_id >= LW6MAP_MAX_NB_TEAMS)	// <0 check useless, unsigned
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("fighter.team_id out of range (%d) for fighter %d"),
		      fighter.team_id, i);
	  ret = 0;
	}
      (real_fighters_per_team[fighter.team_id])++;
      fighter_id =
	lw6ker_map_state_get_fighter_id (map_state, fighter.layer,
					 fighter.pos.x, fighter.pos.y);
      if (i != fighter_id)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("fighter %d in armies array pretends to be at layer=%d,x=%d,y=%d, but in fact there is fighter %d there from slots point of view"),
		      i, fighter.layer, fighter.pos.x, fighter.pos.y,
		      fighter_id);
	  ret = 0;
	}
    }
  for (i = 0; i < LW6MAP_MAX_NB_TEAMS; ++i)
    {
      if (map_state->armies.fighters_per_team[i] > 0
	  && !map_state->teams[i].active)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("team %d pretends to have %d fighters but is inactive"),
		      i, map_state->armies.fighters_per_team[i]);
	  ret = 0;
	}
      if (map_state->armies.fighters_per_team[i] != real_fighters_per_team[i])
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("team %d pretends to have %d fighters but counting them one founds %d"),
		      i, map_state->armies.fighters_per_team[i],
		      real_fighters_per_team[i]);
	  ret = 0;
	}
    }

  return ret;
}

/*
 * Applies the cursors before spreading the gradient.
 */
void
lw6ker_map_state_spread_gradient (lw6ker_map_state_t * map_state,
				  lw6map_options_t * options,
				  int32_t nb_spreads)
{
  int i, j;

  for (i = 0; i < map_state->max_nb_teams; ++i)
    {
      /*
       * Ideas for optimization:
       * - on SMP machines, the various spread_gradient fonctions
       *   could be called in separated threads. One needs to
       *   check wether the cost of creating the threads and
       *   joigning them is worth it, but it could.
       * - spread gradient could be called "less often" when
       *   cursors do not move. Ideally if there's a big move,
       *   we'd call it many times to have accurate results,
       *   but if it didn't really move, well, we'll just spread
       *   once in a while.
       */
      if (map_state->teams[i].active)
	{
	  /*
	   * Note for optimization: on SMP machines, one could decide
	   * to fire several threads, say one per team, and compute
	   * each team on a separate thread (possibly a separate CPU)
	   * and this way one could use more than one single CPU.
	   */
	  lw6ker_team_apply_cursors (&(map_state->teams[i]), options);
	  for (j = 0; j < nb_spreads; ++j)
	    {
	      lw6ker_team_spread_gradient (&(map_state->teams[i]));
	    }
	}
    }
}

int
_lw6ker_map_state_is_slot_free (lw6ker_map_state_t * map_state,
				int32_t layer, int32_t x, int32_t y)
{
  return (lw6ker_map_struct_get_zone_id (map_state->map_struct, x, y) >= 0 &&
	  lw6ker_map_state_get_fighter_id (map_state, layer, x, y) < 0);
}

int
_lw6ker_map_state_is_enemy_there (lw6ker_map_state_t * map_state,
				  int32_t team_id,
				  int32_t layer, int32_t x, int32_t y)
{
  int ret = 0;

  if (lw6ker_map_struct_get_zone_id (map_state->map_struct, x, y) >= 0)
    {
      int32_t enemy_id;

      enemy_id = lw6ker_map_state_get_fighter_id (map_state, layer, x, y);
      if (enemy_id >= 0)
	{
	  ret = (map_state->armies.fighters[enemy_id].team_id != team_id);
	}
    }

  return ret;
}

int
_lw6ker_map_state_is_ally_there (lw6ker_map_state_t * map_state,
				 int32_t team_id,
				 int32_t layer, int32_t x, int32_t y)
{
  int ret = 0;

  if (lw6ker_map_struct_get_zone_id (map_state->map_struct, x, y) >= 0)
    {
      int32_t ally_id;

      ally_id = lw6ker_map_state_get_fighter_id (map_state, layer, x, y);
      if (ally_id >= 0)
	{
	  ret = (map_state->armies.fighters[ally_id].team_id == team_id);
	}
    }

  return ret;
}

static int32_t
find_straight_dir (lw6sys_xy_t from, lw6sys_xy_t to, int parity)
{
  int32_t ret;
  int32_t straight_dir = 0;

  if (to.y < from.y)
    {
      straight_dir = straight_dir | _LW6KER_STRAIGHT_DIR_UP;
    }
  if (to.x > from.x)
    {
      straight_dir = straight_dir | _LW6KER_STRAIGHT_DIR_RIGHT;
    }
  if (to.y > from.y)
    {
      straight_dir = straight_dir | _LW6KER_STRAIGHT_DIR_DOWN;
    }
  if (to.x < from.x)
    {
      straight_dir = straight_dir | _LW6KER_STRAIGHT_DIR_LEFT;
    }

  ret = _LW6KER_TABLES_STRAIGHT_DIRS[parity][straight_dir];

  return ret;
}

/*
 * Higher potential = closer to the cursor
 */
static int32_t
find_best_dir (lw6ker_map_state_t * map_state, lw6ker_fighter_t * fighter,
	       int parity)
{
  int32_t ret = fighter->last_direction;
  int32_t zone_id =
    lw6ker_map_struct_get_zone_id (map_state->map_struct, fighter->pos.x,
				   fighter->pos.y);
  lw6ker_zone_state_t *zone_states =
    map_state->teams[fighter->team_id].gradient;

  if (zone_id >= 0)
    {
      if (zone_states[zone_id].direction_to_cursor >= 0)
	{
	  /*
	   * OK, it was cached...
	   */
	  ret = zone_states[zone_id].direction_to_cursor;
	}
      else
	{
	  /*
	   * Nothing in cache, we calculate it
	   */
	  int32_t i;
	  lw6ker_zone_struct_t *zone_structs = map_state->map_struct->zones;
	  int32_t neighbour_zone_id;
	  lw6ker_zone_struct_t *fighter_zone_struct;
	  fighter_zone_struct = &(zone_structs[zone_id]);
	  int32_t best_potential = zone_states[zone_id].potential;

	  ret = -1;

	  if (parity)
	    {
	      for (i = 0; i < LW6KER_NB_DIRS; ++i)
		{
		  neighbour_zone_id = fighter_zone_struct->link[i];
		  if (neighbour_zone_id >= 0)
		    {
		      if (zone_states[neighbour_zone_id].potential >
			  best_potential)
			{
			  best_potential =
			    zone_states[neighbour_zone_id].potential;
			  ret = i;
			}
		    }
		}
	    }
	  else
	    {
	      for (i = LW6KER_NB_DIRS - 1; i >= 0; --i)
		{
		  neighbour_zone_id = fighter_zone_struct->link[i];
		  if (neighbour_zone_id >= 0)
		    {
		      if (zone_states[neighbour_zone_id].potential >
			  best_potential)
			{
			  best_potential =
			    zone_states[neighbour_zone_id].potential;
			  ret = i;
			}
		    }
		}
	    }

	  if (ret < 0)
	    {
	      /*
	       * Now, in most cases, if we can't find an adjacent square
	       * which is closer, it just means we are in the square
	       * where the cursor is. So we go straight to it.
	       * Fortunately this information is cached in
	       * closest_cursor_pos.
	       */
	      ret =
		find_straight_dir (fighter->pos,
				   zone_states[zone_id].closest_cursor_pos,
				   parity);
	    }

	  if (ret < 0)
	    {
	      /*
	       * this can happen, for find_straight_dir is very likely
	       * to return -1
	       */
	      ret = fighter->last_direction;
	    }
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("fighter with incorrect zone_id=%d (pos=%d,%d)"), zone_id,
		  (int) fighter->pos.x, (int) fighter->pos.y);
    }

  return ret;
}

/*
 * All "polarity" functions follow a uniform scheme:
 * x means x is wrapped (left connected to right)
 * ix means x is wrapped and inversed
 * y means y is wrapped (top connected to bottom)
 * iy means y is wrapped and inversed
 */
static void
polarity_func_nowrap (int32_t * x, int32_t * y,
		      polarity_data_t * polarity_data, int32_t move_dir)
{
  /*
   * Not that in this case there's no border check, in fact we
   * rely on the fact that maps are "correctly built" and
   * have walls at the right place, that is, in this case,
   * on all borders.
   */
  (*x) = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  (*y) = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];
}

static void
polarity_func_x (int32_t * x, int32_t * y,
		 polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_x;

  tmp_x = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  (*y) = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_x < 0)
    {
      (*x) = polarity_data->map_w - 1;
    }
  else if (tmp_x >= polarity_data->map_w)
    {
      (*x) = 0;
    }
  else
    {
      (*x) = tmp_x;
    }
}

static void
polarity_func_y (int32_t * x, int32_t * y,
		 polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_y;

  (*x) = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  tmp_y = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_y < 0)
    {
      (*y) = polarity_data->map_h - 1;
    }
  else if (tmp_y >= polarity_data->map_h)
    {
      (*y) = 0;
    }
  else
    {
      (*y) = tmp_y;
    }
}

static void
polarity_func_ix (int32_t * x, int32_t * y,
		  polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_x, tmp_y;

  tmp_x = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  tmp_y = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_x < 0)
    {
      (*x) = polarity_data->map_w - 1;
      (*y) = polarity_data->map_h - 1 - tmp_y;
    }
  else if (tmp_x >= polarity_data->map_w)
    {
      (*x) = 0;
      (*y) = polarity_data->map_h - 1 - tmp_y;
    }
  else
    {
      (*x) = tmp_x;
      (*y) = tmp_y;
    }
}

static void
polarity_func_iy (int32_t * x, int32_t * y,
		  polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_x, tmp_y;

  tmp_x = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  tmp_y = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_y < 0)
    {
      (*x) = polarity_data->map_w - 1 - tmp_x;
      (*y) = polarity_data->map_h - 1;
    }
  else if (tmp_y >= polarity_data->map_h)
    {
      (*x) = polarity_data->map_w - 1 - tmp_x;
      (*y) = 0;
    }
  else
    {
      (*x) = tmp_x;
      (*y) = tmp_y;
    }
}

static void
polarity_func_x_y (int32_t * x, int32_t * y,
		   polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_x, tmp_y;

  tmp_x = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  tmp_y = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_x < 0)
    {
      (*x) = polarity_data->map_w - 1;
    }
  else if (tmp_x >= polarity_data->map_w)
    {
      (*x) = 0;
    }
  else
    {
      (*x) = tmp_x;
    }

  if (tmp_y < 0)
    {
      (*y) = polarity_data->map_h - 1;
    }
  else if (tmp_y >= polarity_data->map_h)
    {
      (*y) = 0;
    }
  else
    {
      (*y) = tmp_y;
    }
}

static void
polarity_func_ix_y (int32_t * x, int32_t * y,
		    polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_x, tmp_y;

  tmp_x = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  tmp_y = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_y < 0)
    {
      tmp_y = polarity_data->map_h - 1;
    }
  else if (tmp_y >= polarity_data->map_h)
    {
      tmp_y = 0;
    }

  if (tmp_x < 0)
    {
      (*x) = polarity_data->map_w - 1;
      (*y) = polarity_data->map_h - 1 - tmp_y;
    }
  else if (tmp_x >= polarity_data->map_w)
    {
      (*x) = 0;
      (*y) = polarity_data->map_h - 1 - tmp_y;
    }
  else
    {
      (*x) = tmp_x;
      (*y) = tmp_y;
    }
}

static void
polarity_func_x_iy (int32_t * x, int32_t * y,
		    polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_x, tmp_y;

  tmp_x = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  tmp_y = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_x < 0)
    {
      tmp_x = polarity_data->map_w - 1;
    }
  else if (tmp_x >= polarity_data->map_w)
    {
      tmp_x = 0;
    }

  if (tmp_y < 0)
    {
      (*x) = polarity_data->map_w - 1 - tmp_x;
      (*y) = polarity_data->map_h - 1;
    }
  else if (tmp_y >= polarity_data->map_h)
    {
      (*x) = polarity_data->map_w - 1 - tmp_x;
      (*y) = 0;
    }
  else
    {
      (*x) = tmp_x;
      (*y) = tmp_y;
    }
}

static void
polarity_func_ix_iy (int32_t * x, int32_t * y,
		     polarity_data_t * polarity_data, int32_t move_dir)
{
  int32_t tmp_x, tmp_y;

  tmp_x = polarity_data->fighter_x + _LW6KER_TABLES_MOVE_X_OFFSET[move_dir];
  tmp_y = polarity_data->fighter_y + _LW6KER_TABLES_MOVE_Y_OFFSET[move_dir];

  if (tmp_x < 0)
    {
      tmp_x = polarity_data->map_w - 1;
      tmp_y = polarity_data->map_h - 1 - tmp_y;
    }
  else if (tmp_x >= polarity_data->map_w)
    {
      tmp_x = 0;
      tmp_y = polarity_data->map_h - 1 - tmp_y;
    }

  if (tmp_y < 0)
    {
      tmp_x = polarity_data->map_w - 1 - tmp_x;
      tmp_y = polarity_data->map_h - 1;
    }
  else if (tmp_y >= polarity_data->map_h)
    {
      tmp_x = polarity_data->map_w - 1 - tmp_x;
      tmp_y = 0;
    }

  (*x) = tmp_x;
  (*y) = tmp_y;
}

/*
 * Probably the most time greedy function of the algorithm, and
 * one of the hardest one to optimize. It simply moves all the
 * fighters on the battlefield, once the gradient has been calculated.
 *
 * It's the direct equivalent of move_fighters in src/fighter.c
 * in Liquid War 5.
 */
void
lw6ker_map_state_move_fighters (lw6ker_map_state_t * map_state,
				int parity, lw6map_options_t * options,
				int32_t nb_moves)
{
  int i, j, k;
  lw6ker_armies_t *armies = &(map_state->armies);
  int32_t active_fighters = armies->active_fighters;
  lw6ker_fighter_t *fighter = NULL;
  int32_t layer, x, y;
  int32_t best_dir, test_dir;
  int32_t fighter_team_id, fighter_layer, fighter_x, fighter_y;
  int32_t loop_init, loop_step;
  int done_with_fighter;
  int32_t fighter_attack;
  int32_t fighter_defense;
  int32_t fighter_new_health;
  int32_t side_attack_factor;
  int32_t side_defense_factor;
  int32_t nb_move_tries;
  int32_t nb_attack_tries;
  int32_t nb_defense_tries;
  int32_t fighter_side_attack;
  int32_t fighter_side_defense;
  polarity_data_t polarity_data;
  void (*polarity_func) (int32_t * x, int32_t * y,
			 polarity_data_t * polarity_data,
			 int32_t move_dir) = NULL;

  fighter_attack = options->fighter_attack;
  fighter_defense = options->fighter_defense;
  fighter_new_health = options->fighter_new_health;
  side_attack_factor = options->side_attack_factor;
  side_defense_factor = options->side_defense_factor;
  nb_move_tries = options->nb_move_tries;
  nb_attack_tries = options->nb_attack_tries;
  nb_defense_tries = options->nb_defense_tries;
  fighter_side_attack = lw6ker_percent (fighter_attack, side_attack_factor);
  fighter_side_defense =
    lw6ker_percent (fighter_defense, side_defense_factor);

  switch (options->y_polarity)
    {
    case 1:
      switch (options->x_polarity)
	{
	case 1:
	  polarity_func = polarity_func_x_y;
	  break;
	case -1:
	  polarity_func = polarity_func_ix_y;
	  break;
	default:
	  polarity_func = polarity_func_y;
	  break;
	}
      break;
    case -1:
      switch (options->x_polarity)
	{
	case 1:
	  polarity_func = polarity_func_x_iy;
	  break;
	case -1:
	  polarity_func = polarity_func_ix_iy;
	  break;
	default:
	  polarity_func = polarity_func_iy;
	  break;
	}
      break;
    default:
      switch (options->x_polarity)
	{
	case 1:
	  polarity_func = polarity_func_x;
	  break;
	case -1:
	  polarity_func = polarity_func_ix;
	  break;
	default:
	  polarity_func = polarity_func_nowrap;
	  break;
	}
      break;
    }
  polarity_data.map_w = map_state->map_struct->shape.w;
  polarity_data.map_h = map_state->map_struct->shape.h;

  for (k = 0; k < nb_moves; ++k)
    {
      loop_init = parity ? active_fighters - 1 : 0;
      loop_step = parity ? -1 : 1;
      for (i = loop_init; i >= 0 && i < active_fighters; i += loop_step)
	{
	  done_with_fighter = 0;
	  fighter = &(map_state->armies.fighters[i]);
	  fighter_team_id = fighter->team_id;
	  fighter_x = fighter->pos.x;
	  fighter_y = fighter->pos.y;
	  fighter_layer = fighter->layer;
	  polarity_data.fighter_x = fighter_x;
	  polarity_data.fighter_y = fighter_y;

	  best_dir = find_best_dir (map_state, fighter, parity);
	  // we assume best_dir is equal to something valid now

	  layer = fighter_layer;	// not multilayer yet
	  for (j = 0; j < nb_move_tries; ++j)
	    {
	      test_dir = _LW6KER_TABLES_MOVE_DIR[parity][best_dir][j];
	      polarity_func (&x, &y, &polarity_data, test_dir);
	      if (_lw6ker_map_state_is_slot_free
		  (map_state, fighter_layer, x, y))
		{
		  done_with_fighter = 1;
		  _lw6ker_fighter_move (fighter, i, fighter_layer, x, y,
					map_state);
		  break;
		}
	    }

	  if (!done_with_fighter)
	    {
	      for (j = 0; j < nb_attack_tries; ++j)
		{
		  test_dir = _LW6KER_TABLES_MOVE_DIR[parity][best_dir][j];
		  polarity_func (&x, &y, &polarity_data, test_dir);
		  if (_lw6ker_map_state_is_enemy_there
		      (map_state, fighter_team_id, fighter_layer, x, y))
		    {
		      done_with_fighter = 1;
		      _lw6ker_fighter_attack (fighter, i, fighter_layer, x, y,
					      map_state,
					      j >
					      0 ? fighter_attack :
					      fighter_side_attack,
					      fighter_new_health);
		      break;
		    }
		}
	    }

	  if (!done_with_fighter)
	    {
	      for (j = 0; j < nb_defense_tries; ++j)
		{
		  test_dir = _LW6KER_TABLES_MOVE_DIR[parity][best_dir][j];
		  polarity_func (&x, &y, &polarity_data, test_dir);
		  if (_lw6ker_map_state_is_ally_there
		      (map_state, fighter_team_id, fighter_layer, x, y))
		    {
		      done_with_fighter = 1;
		      _lw6ker_fighter_defend (fighter, i, fighter_layer, x, y,
					      map_state,
					      j >
					      0 ? fighter_defense :
					      fighter_side_defense);
		      break;
		    }
		}
	    }
	}
    }
}
