
/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6KER_H
#define LIQUIDWAR6KER_H

#include "../sys/sys.h"
#include "../map/map.h"

#define LW6KER_NB_DIRS 12

#define LW6KER_DIR_NNE 0
#define LW6KER_DIR_NE 1
#define LW6KER_DIR_ENE 2
#define LW6KER_DIR_ESE 3
#define LW6KER_DIR_SE 4
#define LW6KER_DIR_SSE 5
#define LW6KER_DIR_SSW 6
#define LW6KER_DIR_SW 7
#define LW6KER_DIR_WSW 8
#define LW6KER_DIR_WNW 9
#define LW6KER_DIR_NW 10
#define LW6KER_DIR_NNW 11

#define LW6KER_NB_PARITIES 2

#define LW6KER_PARITY_EVEN 0
#define LW6KER_PARITY_ODD 1

#include "ker-trigo.h"

typedef struct lw6ker_zone_struct_s
{
  lw6sys_xy_t pos;
  u_int32_t size;
  int32_t link[LW6KER_NB_DIRS];
}
lw6ker_zone_struct_t;

typedef struct lw6ker_slot_struct_s
{
  int32_t zone_id;
  int32_t depth;
}
lw6ker_slot_struct_t;

/*
 * To some extent, this structure is very similar to lw6map_level_t.
 * Let's explain why it exists separately: 
 * - this structure is not only used to store the information, but
 *   also to present it in a form which is adapted to (fast) treatments.
 *   This means it can be redundant, akward, whereas lw6map_level_t is meant
 *   to just convey informations. lw6ker_map_struct_t is used when the
 *   game is running, to inform the core algorithm of tha map structure,
 *   whereas lw6map_level_t is used to intialize lw6ker_map_struct_t, to send
 *   the maps structure to other players over the network, to initialize
 *   the graphics artifacts which will be used for rendering, and so on.
 * - having a loading/serialization (that is lw6map_level_t) separated from
 *   algorithmic wizardry related to shortest path insanities might be
 *   what could prevent LW6 from being as bloated as LW5.
 */
typedef struct lw6ker_map_struct_s
{
  lw6sys_wh_t shape;
  int32_t nb_zones;
  int32_t nb_slots;
  int32_t nb_usable_slots;
  int32_t room_for_armies;
  int32_t max_depth;
  int32_t max_zone_size;
  lw6ker_zone_struct_t *zones;
  lw6ker_slot_struct_t *slots;
}
lw6ker_map_struct_t;

/*
 * Game struct contains all the data which never change during a game.
 * If it was feasible in C, we would change it to "read-only" when game
 * starts.
 */
typedef struct lw6ker_game_struct_s
{
  int id;
  lw6map_level_t *level;
  lw6ker_map_struct_t map_struct;
  lw6map_options_t options;
}
lw6ker_game_struct_t;

/*
 * Completes lw6ker_zone_struct_t with per-team information.
 */
typedef struct lw6ker_zone_state_s
{
  int32_t potential:24;
  int32_t direction_to_cursor:8;
  lw6sys_xy_t closest_cursor_pos;
}
lw6ker_zone_state_t;

typedef struct lw6ker_slot_state_s
{
  int32_t fighter_id;
}
lw6ker_slot_state_t;

typedef struct lw6ker_fighter_s
{
  u_int32_t team_id:4;
  u_int32_t layer:4;
  u_int32_t last_direction:8;
  u_int32_t health:16;
  lw6sys_xy_t pos;
}
lw6ker_fighter_t;

typedef struct lw6ker_cursor_s
{
  lw6sys_xy_t pos;
  lw6sys_xy_t last_applied_pos;
  int32_t pot_offset;
}
lw6ker_cursor_t;

typedef struct lw6ker_cursor_array_s
{
  int32_t nb_cursors;
  lw6ker_cursor_t cursors[LW6MAP_MAX_CURSORS_PER_TEAM];
}
lw6ker_cursor_array_t;

typedef struct lw6ker_team_s
{
  int active;
  lw6ker_map_struct_t *map_struct;
  lw6ker_zone_state_t *gradient;
  int32_t cursor_ref_pot;
  int32_t last_spread_dir;
  lw6ker_cursor_array_t cursor_array;
}
lw6ker_team_t;

typedef struct lw6ker_armies_s
{
  lw6ker_map_struct_t *map_struct;
  /*
   * The maximum number or fighters in the armies, in fact
   * it's more a less the amount of free space in the map, minus
   * some standard percentage to avoid bloating a map completely
   * (playing with 100% space filled doesn't really make sense).
   */
  int32_t max_fighters;
  /*
   * The number of active fighters, that is the ones that are
   * actually doing things on the map, moving, fighting. Other
   * ones are dormant fighters which will activate when a
   * network player connects for instance.
   */
  int32_t active_fighters;
  /*
   * This is highly redundant for one could get this information
   * by simply reading the fighters themselves, however this is
   * inefficient, and having the exact count is usefull at every
   * game refreh, to display the information on the screen.
   */
  int32_t fighters_per_team[LW6MAP_MAX_NB_TEAMS];
  /*
   * The actual data, a pointer to all the fighters in the map.
   * It's more convenient to access them this way, instead of
   * having to read a w*h array, which can be very big and 90%
   * empty. This optimization is not usefull on small and/or
   * crowded maps, but experience shows that:
   * a) only small maps can be crowded anyway
   * b) small maps are fast, only big maps require optimization.
   */
  lw6ker_fighter_t *fighters;
}
lw6ker_armies_t;

typedef struct lw6ker_layer_s
{
  lw6ker_map_struct_t *map_struct;
  int32_t nb_slots;		// redundant but convenient
  lw6ker_armies_t *armies;
  lw6ker_slot_state_t *slots;
} lw6ker_layer_t;

typedef struct lw6ker_map_state_s
{
  lw6ker_map_struct_t *map_struct;
  lw6sys_wh_t shape;		// redundant but convenient
  lw6ker_armies_t armies;
  int32_t max_nb_teams;
  lw6ker_team_t teams[LW6MAP_MAX_NB_TEAMS];
  int32_t nb_layers;
  lw6ker_layer_t layers[LW6MAP_MAX_DEPTH];
}
lw6ker_map_state_t;

typedef struct lw6ker_bot_s
{
  int active;
  lw6sys_xy_t pos;
  int32_t rounds_at_end_of_last_move;
  int moving;
  int32_t move_nb_steps;
  int32_t move_current_step;
  lw6sys_xy_t move_pos_from;
  lw6sys_xy_t move_pos_to;
} lw6ker_bot_t;

/*
 * Game state contains all the data which is verstatile, stuff that
 * changes, this includes "where fighters are" of course but also
 * some parameters which might change in-game (such as an option
 * which can be changeable, an alliance between teams, and so on...)
 */
typedef struct lw6ker_game_state_s
{
  int id;
  lw6ker_game_struct_t *game_struct;
  lw6ker_map_state_t map_state;
  u_int32_t moves;
  u_int32_t spreads;
  u_int32_t rounds;
  lw6ker_bot_t bots[LW6MAP_MAX_NB_TEAMS][LW6MAP_MAX_CURSORS_PER_TEAM];
}
lw6ker_game_state_t;

/*
 * in armies.c
 */
extern int32_t lw6ker_armies_add_fighter (lw6ker_armies_t * armies,
					  lw6ker_fighter_t fighter);
extern int lw6ker_armies_remove_fighter (lw6ker_armies_t * armies);

/*
 * in cursor.c
 */
extern int lw6ker_cursor_set (lw6ker_cursor_t * cursor, int32_t x,
			      int32_t y, int32_t pot_offset,
			      lw6ker_map_struct_t * map_struct,
			      int32_t max_cursor_pot_offset);
extern void lw6ker_cursor_clear (lw6ker_cursor_t * cursor);

/*
 * in cursorarray.c
 */
extern void lw6ker_cursor_array_clear (lw6ker_cursor_array_t * cursor_array);
extern int lw6ker_cursor_array_add (lw6ker_cursor_array_t *
				    cursor_array, lw6ker_cursor_t * cursor);
extern int lw6ker_cursor_array_set (lw6ker_cursor_array_t *
				    cursor_array, int32_t i,
				    int32_t x, int32_t y,
				    int32_t pot_offset,
				    lw6ker_map_struct_t * map_struct,
				    int32_t max_cursor_pot_offset);
extern int lw6ker_cursor_array_force_nb_cursors (lw6ker_cursor_array_t *
						 cursor_array,
						 int32_t nb_cursors);

/*
 * in gamestate.c
 */
extern lw6ker_game_state_t *lw6ker_game_state_new (lw6ker_game_struct_t *
						   game_struct);
extern void lw6ker_game_state_free (lw6ker_game_state_t * game_state);
extern int lw6ker_game_state_memory_footprint (lw6ker_game_state_t *
					       game_state);
extern char *lw6ker_game_state_repr (lw6ker_game_state_t * game_state);
extern int lw6ker_game_state_copy (lw6ker_game_state_t * dst,
				   lw6ker_game_state_t * src);
extern u_int32_t lw6ker_game_state_checksum (lw6ker_game_state_t *
					     game_state);
extern int lw6ker_game_state_add_team (lw6ker_game_state_t * game_state,
				       int32_t team_id, int32_t nb_cursors);
extern int lw6ker_game_state_remove_team (lw6ker_game_state_t * game_state,
					  int32_t team_id);
extern void lw6ker_game_state_do_round (lw6ker_game_state_t * game_state);
extern u_int32_t lw6ker_game_state_get_moves (lw6ker_game_state_t *
					      game_state);
extern u_int32_t lw6ker_game_state_get_spreads (lw6ker_game_state_t *
						game_state);
extern u_int32_t lw6ker_game_state_get_rounds (lw6ker_game_state_t *
					       game_state);
extern int lw6ker_game_state_enable_bot (lw6ker_game_state_t * game_state,
					 int32_t team_id);
extern int lw6ker_game_state_disable_bot (lw6ker_game_state_t * game_state,
					  int32_t team_id);

/*
 * In gamestruct.c
 */
extern lw6ker_game_struct_t *lw6ker_game_struct_new (lw6map_level_t * level);
extern void lw6ker_game_struct_free (lw6ker_game_struct_t * game_struct);
extern int lw6ker_game_struct_memory_footprint (lw6ker_game_struct_t *
						game_struct);
extern char *lw6ker_game_struct_repr (lw6ker_game_struct_t * game_struct);
extern u_int32_t lw6ker_game_struct_checksum (lw6ker_game_struct_t *
					      game_struct);

/*
 * In layer.c
 */

/*
 * in mapstate.c
 */
extern int32_t lw6ker_map_state_get_free_team_id (lw6ker_map_state_t *
						  map_state);
extern void lw6ker_map_state_start_xy (lw6ker_map_state_t * map_state,
				       lw6sys_xy_t * start_xy,
				       int32_t team_id);
extern int32_t lw6ker_map_state_get_nb_active_teams (lw6ker_map_state_t *
						     map_state);;
extern int32_t lw6ker_map_state_populate_team (lw6ker_map_state_t * map_state,
					       int32_t team_id,
					       int32_t nb_fighters,
					       lw6sys_xy_t desired_center,
					       int32_t nb_cursors,
					       lw6map_options_t options);
extern int lw6ker_map_state_redistribute_team (lw6ker_map_state_t * map_state,
					       int32_t dst_team_id,
					       int32_t src_team_id,
					       int32_t nb_fighters,
					       lw6map_options_t options);
extern int lw6ker_map_state_cancel_team (lw6ker_map_state_t * map_state,
					 int32_t team_id);
extern int lw6ker_map_state_remove_fighter (lw6ker_map_state_t * map_state,
					    int32_t fighter_id);
extern int lw6ker_map_state_remove_fighters (lw6ker_map_state_t * map_state,
					     int32_t nb_fighters);
extern int lw6ker_map_state_remove_team_fighters (lw6ker_map_state_t *
						  map_state, int32_t team_id,
						  int32_t nb_fighters);
static inline void
lw6ker_map_state_set_fighter_id (lw6ker_map_state_t * map_state,
				 int32_t layer, int32_t x,
				 int32_t y, int32_t fighter_id)
{
  map_state->layers[layer].slots[map_state->shape.w * y + x].fighter_id =
    fighter_id;
};
static inline int32_t
lw6ker_map_state_get_fighter_id (lw6ker_map_state_t * map_state,
				 int32_t layer, int32_t x, int32_t y)
{
  return (map_state->layers[layer].slots[map_state->shape.w * y + x].
	  fighter_id);
};

extern lw6ker_fighter_t *lw6ker_map_state_get_fighter_safe (lw6ker_map_state_t
							    * map_state,
							    int32_t layer,
							    int32_t x,
							    int32_t y);
/*
 * Carefull with this one, there must really be a fighter
 * or it segfaults directly.
 */
static inline lw6ker_fighter_t *
lw6ker_map_state_get_fighter_unsafe (lw6ker_map_state_t * map_state,
				     int32_t layer, int32_t x, int32_t y)
{
  return (&
	  (map_state->armies.
	   fighters[map_state->layers[layer].
		    slots[map_state->shape.w * y + x].fighter_id]));
};
extern void lw6ker_map_state_print_debug (lw6ker_map_state_t * map_state);
extern int lw6ker_map_state_sanity_check (lw6ker_map_state_t * map_state);
extern void lw6ker_map_state_spread_gradient (lw6ker_map_state_t * map_state,
					      lw6map_options_t * options,
					      int32_t nb_spreads);
extern void lw6ker_map_state_move_fighters (lw6ker_map_state_t * map_state,
					    int parity,
					    lw6map_options_t * options,
					    int32_t nb_moves);

/*
 * In mapstruct.c
 */
static inline void
lw6ker_map_struct_set_depth (lw6ker_map_struct_t * map_struct, int32_t x,
			     int32_t y, int32_t depth)
{
  map_struct->slots[map_struct->shape.w * y + x].depth = depth;
};
static inline int32_t
lw6ker_map_struct_get_depth (lw6ker_map_struct_t * map_struct, int32_t x,
			     int32_t y)
{
  return (map_struct->slots[map_struct->shape.w * y + x].depth);
};
static inline void
lw6ker_map_struct_set_zone_id (lw6ker_map_struct_t * map_struct, int32_t x,
			       int32_t y, int32_t zone_id)
{
  map_struct->slots[map_struct->shape.w * y + x].zone_id = zone_id;
};
static inline int32_t
lw6ker_map_struct_get_zone_id (lw6ker_map_struct_t * map_struct, int32_t x,
			       int32_t y)
{
  return (map_struct->slots[map_struct->shape.w * y + x].zone_id);
};
extern void lw6ker_map_struct_find_free_slot_near (lw6ker_map_struct_t *
						   map_struct,
						   lw6sys_xy_t * there,
						   lw6sys_xy_t here);

/*
 * In team.c
 */
extern void lw6ker_team_activate (lw6ker_team_t * team,
				  int32_t nb_cursors, lw6sys_xy_t pos);
extern void lw6ker_team_unactivate (lw6ker_team_t * team);
extern void lw6ker_team_apply_cursors (lw6ker_team_t * team,
				       lw6map_options_t * options);
extern void lw6ker_team_spread_gradient (lw6ker_team_t * team);
extern void lw6ker_team_normalize_pot (lw6ker_team_t * team,
				       lw6map_options_t * options);

/*
 * In test.c
 */
extern int lw6ker_test ();

/*
 * In trigo.c
 */
static inline int32_t
lw6ker_sin (int32_t alpha)
{
  alpha = alpha & LW6KER_TRIGO_2PI_MASK;
  return LW6KER_TRIGO_SIN_TABLE[alpha];
}
static inline int32_t
lw6ker_cos (int32_t alpha)
{
  alpha = (LW6KER_TRIGO_PI2 + alpha) & LW6KER_TRIGO_2PI_MASK;
  return LW6KER_TRIGO_SIN_TABLE[alpha];
}

/*
 * Various utils
 */
static inline int32_t
lw6ker_percent (int32_t n, int32_t percent)
{
  return (n * percent) / 100;
}
static inline int32_t
lw6ker_per1000 (int32_t n, int32_t per1000)
{
  return (n * per1000) / 1000;
}

#endif
