
/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <dirent.h>
#include <string.h>
#include <stdio.h>

#include "config.h"
#include "ldr.h"
#include "ldr-internal.h"

static lw6ldr_entry_t *
new_entry (char *absolute_path, char *relative_path, char *d_name)
{
  lw6ldr_entry_t *entry = NULL;
  char *map_filename = NULL;

  if (d_name[0] != '.')
    {
      entry = (lw6ldr_entry_t *) LW6SYS_CALLOC (sizeof (lw6ldr_entry_t));
      if (entry)
	{
	  entry->absolute_path = lw6sys_path_concat (absolute_path, d_name);
	  if (entry->absolute_path)
	    {
	      if (lw6sys_dir_exists (entry->absolute_path))
		{
		  entry->title = lw6sys_str_copy (d_name);
		  entry->relative_path =
		    lw6sys_path_concat (relative_path, d_name);
		  map_filename =
		    lw6sys_path_concat (entry->absolute_path,
					_LW6LDR_FILE_MAP_PNG);
		  if (map_filename)
		    {
		      if (!lw6sys_file_exists (map_filename))
			{
			  /*
			   * OK there's no map.png, let's assume there are
			   * some sub directories in it. At least, that's all
			   * we can do...
			   */
			  entry->has_subdirs = 1;
			}
		      LW6SYS_FREE (map_filename);
		    }
		}
	      else
		{
		  // not a directory, ignore
		}
	    }
	}
    }

  if (entry
      && (!entry->title || !entry->absolute_path || !entry->relative_path))
    {
      lw6ldr_free_entry (entry);
      entry = NULL;
    }

  return entry;
}

static void
free_entry_callback (void *data)
{
  lw6ldr_free_entry ((lw6ldr_entry_t *) data);
}

void
lw6ldr_free_entry (lw6ldr_entry_t * entry)
{
  if (entry)
    {
      // needs to be rock solid for construction bugs
      if (entry->title)
	{
	  LW6SYS_FREE (entry->title);
	}
      if (entry->absolute_path)
	{
	  LW6SYS_FREE (entry->absolute_path);
	}
      if (entry->relative_path)
	{
	  LW6SYS_FREE (entry->relative_path);
	}
    }
  LW6SYS_FREE (entry);
}

static void
add_subdirs (lw6sys_list_t ** entries, lw6sys_assoc_t ** entries_index,
	     char *absolute_path, char *relative_path)
{
  struct dirent **namelist = NULL;
  int n;

  n = scandir (absolute_path, &namelist, NULL, alphasort);
  if (n < 0)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "", _("couldn't read map dir \"%s\""),
		  absolute_path);
    }
  else
    {
      if (n > 0)
	{
	  while (n--)
	    {
	      if (1)
		{
		  lw6ldr_entry_t *entry =
		    new_entry (absolute_path, relative_path,
			       namelist[n]->d_name);
		  if (entry)
		    {
		      if (!lw6sys_assoc_has_key
			  (*entries_index, entry->relative_path))
			{
			  lw6sys_lifo_push (entries, (void *) entry);
			}
		      else
			{
			  lw6sys_log (LW6SYS_LOG_INFO,
				      _
				      ("duplicate map \"%s\", it is present in at least two places, last found (and ignored) in \"%s\""),
				      entry->relative_path,
				      entry->absolute_path);
			  lw6ldr_free_entry (entry);
			}
		      lw6sys_assoc_set (entries_index, entry->relative_path,
					NULL);
		    }
		}
	      free (namelist[n]);	// use of free instead of LW6SYS_FREE is "normal" here       
	    }
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("no maps in dir \"%s\""), absolute_path);
	}
      free (namelist);		// use of free instead of LW6SYS_FREE is "normal" here
    }
}

/*
 * Returns a list of lw6ldr_entry_t elements, which
 * describe what maps are in a directory.
 */
lw6sys_list_t *
lw6ldr_get_entries (char *map_path, char *relative_path)
{
  lw6sys_list_t *entries = NULL;
  lw6sys_assoc_t *entries_index = NULL;
  lw6sys_list_t *dirs = NULL;
  char *dir = NULL;
  char *absolute_path = NULL;

  entries_index = lw6sys_assoc_new (NULL);
  if (entries_index)
    {
      entries = lw6sys_list_new (free_entry_callback);
      if (entries)
	{
	  dirs = lw6sys_env_split (map_path);
	  if (dirs)
	    {
	      while (dirs && (dir = (char *) lw6sys_lifo_pop (&dirs)) != NULL)
		{
		  if (dir)
		    {
		      if (strlen (dir) > 0)
			{
			  absolute_path =
			    lw6sys_path_concat (dir, relative_path);
			  if (absolute_path)
			    {
			      if (lw6sys_dir_exists (absolute_path))
				{
				  add_subdirs (&entries, &entries_index,
					       absolute_path, relative_path);
				}
			      LW6SYS_FREE (absolute_path);
			    }
			}
		      LW6SYS_FREE (dir);
		    }
		}
	    }
	}
      lw6sys_assoc_free (entries_index);
    }

  return entries;
}
