/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>

#include "config.h"
#include "ldr.h"
#include "ldr-internal.h"

#define EXAMPLE_DIR "example"

/**
 * lw6ldr_print_example_options_xml
 *
 * @f: file to output content to
 *
 * Print to a file a typical map options.xml file.
 * 
 * Return value: none.
 */
void
lw6ldr_print_example_options_xml (FILE * f)
{
  lw6map_options_t options;
  lw6sys_assoc_t *assoc;
  int i;
  char *str;

  memset (&options, 0, sizeof (lw6map_options_t));
  lw6map_options_defaults (&options);

  lw6sys_print_xml_header (f,
			   _
			   ("This is an example Liquid War 6 'options.xml' file. This file can be put along with a 'map.png' file to set up some parameters to your own preferred values. However, it's not required, a map can live without any 'options.xml' file, defaults will be used, that's all. Note that this example file has every available entry redefined, which is likely not to be what you want. I suggest you remove all the entries you don't need, so that if players know better for these parameters, they use theirs. Happy hacking!"));

  assoc = lw6sys_assoc_new (lw6sys_free_callback);
  if (assoc)
    {
      for (i = 0; assoc && (LW6MAP_OPTIONS_LIST[i] != NULL); ++i)
	{
	  str =
	    lw6sys_itoa (lw6map_options_get_default (LW6MAP_OPTIONS_LIST[i]));
	  if (str)
	    {
	      lw6sys_assoc_set (&assoc, LW6MAP_OPTIONS_LIST[i], str);
	    }
	}
      lw6sys_assoc_sort_and_map (assoc,
				 (lw6sys_assoc_t_CALLBACK_FUNC)
				 lw6cfg_write_xml_guess_type, (void *) f);
      lw6sys_assoc_free (assoc);
    }

  lw6sys_print_xml_footer (f);

  lw6map_options_clear (&options);
}

/**
 * lw6ldr_print_example_style_xml
 *
 * @f: file to output content to
 *
 * Print to a file a typical map style.xml file.
 * 
 * Return value: none.
 */
void
lw6ldr_print_example_style_xml (FILE * f)
{
  lw6map_style_t style;

  memset (&style, 0, sizeof (lw6map_style_t));
  lw6map_style_defaults (&style);

  lw6sys_print_xml_header (f,
			   _
			   ("This is an example Liquid War 6 'style.xml' file. This file can be put along with a 'map.png' file to change the look of the map. It will not change the gameplay at all, all gameplay related parameters are in 'options.xml'. However, it's not required, a map can live without any 'style.xml' file, defaults will be used, that's all. Note that this example file has every available entry redefined, which is likely not to be what you want. I suggest you remove all the entries you don't need, so that if players know better for these parameters, they use theirs. Happy hacking!"));

  lw6cfg_write_xml_bool (f, "keep-ratio", style.keep_ratio);
  lw6cfg_write_xml_float (f, "zoom", style.zoom);
  lw6cfg_write_xml_string (f, "background-style", style.background_style);
  lw6cfg_write_xml_string (f, "hud-style", style.hud_style);
  lw6cfg_write_xml_string (f, "menu-style", style.menu_style);
  lw6cfg_write_xml_string (f, "view-style", style.view_style);
  lw6cfg_write_xml_color (f, "team-color-red",
			  style.team_colors[LW6MAP_TEAM_COLOR_RED]);
  lw6cfg_write_xml_color (f, "team-color-green",
			  style.team_colors[LW6MAP_TEAM_COLOR_GREEN]);
  lw6cfg_write_xml_color (f, "team-color-blue",
			  style.team_colors[LW6MAP_TEAM_COLOR_BLUE]);
  lw6cfg_write_xml_color (f, "team-color-yellow",
			  style.team_colors[LW6MAP_TEAM_COLOR_YELLOW]);
  lw6cfg_write_xml_color (f, "team-color-cyan",
			  style.team_colors[LW6MAP_TEAM_COLOR_CYAN]);
  lw6cfg_write_xml_color (f, "team-color-magenta",
			  style.team_colors[LW6MAP_TEAM_COLOR_MAGENTA]);
  lw6cfg_write_xml_color (f, "team-color-orange",
			  style.team_colors[LW6MAP_TEAM_COLOR_ORANGE]);
  lw6cfg_write_xml_color (f, "team-color-lightblue",
			  style.team_colors[LW6MAP_TEAM_COLOR_LIGHTBLUE]);
  lw6cfg_write_xml_color (f, "team-color-purple",
			  style.team_colors[LW6MAP_TEAM_COLOR_PURPLE]);
  lw6cfg_write_xml_color (f, "team-color-pink",
			  style.team_colors[LW6MAP_TEAM_COLOR_PINK]);

  lw6sys_print_xml_footer (f);

  lw6map_style_clear (&style);
}

/**
 * lw6ldr_print_examples
 *
 * @user_dir: the user directory or at least, a writable one
 *
 * Writes all example XML files in 'user_dir/example/', will
 * create the directory if needed.
 *
 * Return value: 1 if success, 0 if failed.
 */
int
lw6ldr_print_examples (char *user_dir)
{
  int ret = 0;
  char *example_dir = NULL;
  char *filename = NULL;
  FILE *f;
  int options_ok = 0;
  int style_ok = 0;

  example_dir = lw6sys_path_concat (user_dir, EXAMPLE_DIR);
  if (example_dir)
    {
      if (!lw6sys_dir_exists (example_dir))
	{
	  lw6sys_create_dir (example_dir);
	}
      if (lw6sys_dir_exists (example_dir))
	{
	  filename =
	    lw6sys_path_concat (example_dir, _LW6LDR_FILE_OPTIONS_XML);
	  if (filename)
	    {
	      f = fopen (filename, "w");
	      if (f)
		{
		  lw6sys_log (LW6SYS_LOG_INFO,
			      _("writing example map options file in \"%s\""),
			      filename);
		  lw6ldr_print_example_options_xml (f);
		  options_ok = 1;
		  fclose (f);
		}
	      LW6SYS_FREE (filename);
	    }
	  filename = lw6sys_path_concat (example_dir, _LW6LDR_FILE_STYLE_XML);
	  if (filename)
	    {
	      f = fopen (filename, "w");
	      if (f)
		{
		  lw6sys_log (LW6SYS_LOG_INFO,
			      _("writing example map style file in \"%s\""),
			      filename);
		  lw6ldr_print_example_style_xml (f);
		  style_ok = 1;
		  fclose (f);
		}
	      LW6SYS_FREE (filename);
	    }
	}
      LW6SYS_FREE (example_dir);
    }

  ret = (options_ok && style_ok);

  return ret;
}
