/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <libpng12/png.h>

#include "config.h"
#include "ldr.h"
#include "ldr-internal.h"

/**
 * lw6ldr_read
 *
 * @dirname: the directory containing the map
 * @default_param: default parameters, as strings
 * @forced_param: forced parameters, as strings
 * @display_shape: the shape of the display output (resolution)
 *
 * Loads a map from dist. The default_param and forced_param can contain
 * values corresponding to options.xml and style.xml entries. Parameters
 * are read in 4 steps. 1st, a default value is picked by the program.
 * 2nd, any value in @default_param replaces previous values. 3rd, any
 * value in options.xml or style.xml replaces previous values.
 * 4th, any value in @forced_param replaces previous values. In practice,
 * the @default_param allows the user to set defaults which can still
 * be overwritten by the map, while @forced_param is a definitive 'ignore
 * what is is defined in the map' way of doing things.
 * See also @lw6ldr_read_relative.
 *
 * Return value: 1 if success, 0 if failed.
 */
lw6map_level_t *
lw6ldr_read (char *dirname, lw6sys_assoc_t * default_param,
	     lw6sys_assoc_t * forced_param, lw6sys_wh_t * display_shape)
{
  lw6map_level_t *level = NULL;
  int ok = 0;
  float ratio = 1.0f;

  lw6sys_log (LW6SYS_LOG_INFO, _("loading map \"%s\""), dirname);

  level = lw6map_new ();
  if (level)
    {
      lw6map_param_defaults (&(level->param));

      ok = 1;

      ok = ok && lw6ldr_metadata_read (&level->metadata, dirname);
      lw6ldr_param_update (&(level->param), default_param);
      ok = ok && lw6ldr_param_read (&level->param, dirname);
      lw6ldr_param_update (&(level->param), forced_param);
      lw6map_auto_colors (&level->param.style);
      ok = ok
	&& lw6ldr_texture_read (&level->texture, dirname,
				&(level->param.style), display_shape, &ratio);
      ok = ok
	&& lw6ldr_depth_read (&level->depth, dirname, &(level->param.options),
			      display_shape, ratio,
			      level->param.style.fighter_scale);
      _lw6ldr_guess_colors (level);
      lw6map_auto_colors (&level->param.style);
    }

  if (!ok)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("unable to load map \"%s\""),
		  dirname);
      lw6map_free (level);
      level = NULL;
    }

  return level;
}

/**
 * lw6ldr_read_relative
 *
 * @map_path: a collection of paths where to find maps
 * @relative_path: something which will be appended to a @map_path member
 * @default_param: default parameters, as strings
 * @forced_param: forced parameters, as strings
 * @display_shape: the shape of the display output (resolution)
 *
 * Reads a map from disk, using the map-path value, which is a collection
 * of paths defined by the command-line, the environment variables, and
 * the config file. @default_param and @forced_param work as in the
 * function @lw6ldr_read.
 *
 * Return value: 1 if success, 0 if failure.
 */
lw6map_level_t *
lw6ldr_read_relative (char *map_path, char *relative_path,
		      lw6sys_assoc_t * default_param,
		      lw6sys_assoc_t * forced_param,
		      lw6sys_wh_t * display_shape)
{
  lw6map_level_t *ret = NULL;
  lw6sys_list_t *dirs = NULL;
  char *dir = NULL;
  char *full_dir = NULL;

  dirs = lw6sys_env_split (map_path);
  if (dirs)
    {
      while (dirs && (dir = (char *) lw6sys_lifo_pop (&dirs)) != NULL)
	{
	  full_dir = lw6sys_path_concat (dir, relative_path);
	  if (full_dir)
	    {
	      if (lw6sys_dir_exists (full_dir))
		{
		  if (!ret)
		    {
		      ret =
			lw6ldr_read (full_dir, default_param, forced_param,
				     display_shape);
		    }
		}
	      LW6SYS_FREE (full_dir);
	    }
	  LW6SYS_FREE (dir);
	}
    }

  return ret;
}
