
/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>

#include <libpng12/png.h>
#include <jpeglib.h>

#include "config.h"
#include "ldr.h"
#include "ldr-internal.h"

int
_lw6ldr_rgba_read_png (_lw6ldr_image_rgba_t * image, char *png_file)
{
  png_structp png_ptr = NULL;
  png_infop info_ptr = NULL;
  png_infop end_info = NULL;
  /* 
   * no need to use end_info for we use the high level interface
   * with png_read_png.
   */
  int ret = 0;
  FILE *f = NULL;

  png_ptr = png_create_read_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (png_ptr)
    {
      info_ptr = png_create_info_struct (png_ptr);
      if (info_ptr)
	{
	  end_info = png_create_info_struct (png_ptr);
	  if (end_info)
	    {
	      f = fopen (png_file, "rb");
	      if (f)
		{
		  png_uint_32 width;
		  png_uint_32 height;
		  int bit_depth;
		  int color_type;
		  unsigned char **buf = NULL;
		  int row;
		  int memory_ok = 1;
		  int format_ok = 1;

		  png_init_io (png_ptr, f);

		  png_read_info (png_ptr, info_ptr);

		  png_get_IHDR (png_ptr, info_ptr, &width, &height,
				&bit_depth, &color_type, NULL, NULL, NULL);

		  png_set_expand (png_ptr);
		  png_set_strip_16 (png_ptr);
		  png_set_packswap (png_ptr);

		  if (!(color_type & PNG_COLOR_MASK_COLOR))
		    {
		      png_set_gray_to_rgb (png_ptr);
		    }

		  if (!(color_type & PNG_COLOR_MASK_ALPHA))
		    {
		      png_set_filler (png_ptr, 255, PNG_FILLER_AFTER);
		    }

		  png_read_update_info (png_ptr, info_ptr);

		  png_get_IHDR (png_ptr, info_ptr, &width, &height,
				&bit_depth, &color_type, NULL, NULL, NULL);

		  if (color_type & PNG_COLOR_MASK_PALETTE)
		    {
		      lw6sys_log (LW6SYS_LOG_WARNING,
				  _
				  ("can't load RGBA PNG file \"%s\", it is still paletted after filtering"),
				  png_file);
		      format_ok = 0;
		    }


		  if (info_ptr->rowbytes != info_ptr->width * 4 ||
		      bit_depth > 8)
		    {
		      lw6sys_log (LW6SYS_LOG_WARNING,
				  _
				  ("can't load RGBA PNG file \"%s\", memory footprint is inconsistent, color_type=%d, rowbytes=%d, width=%d, step=%d, bit_depth=%d"),
				  png_file, color_type,
				  info_ptr->rowbytes, width, 4, bit_depth);
		      format_ok = 0;
		    }

		  if (format_ok)
		    {
		      buf =
			(unsigned char **) LW6SYS_MALLOC (height *
							  sizeof (unsigned
								  char *));
		      if (buf)
			{
			  for (row = 0; row < height; ++row)
			    {
			      buf[row] =
				(unsigned char *) LW6SYS_MALLOC (info_ptr->
								 rowbytes *
								 sizeof
								 (unsigned
								  char));
			      if (!buf[row])
				{
				  memory_ok = 0;
				}
			    }
			}
		      else
			{
			  memory_ok = 0;
			}

		      if (memory_ok)
			{
			  png_read_image (png_ptr, buf);
			  png_read_end (png_ptr, end_info);

			  image->shape.w = width;
			  image->shape.h = height;
			  image->data = buf;
			  ret = 1;
			}
		      else
			{
			  lw6sys_log (LW6SYS_LOG_ERROR,
				      _
				      ("unable to allocate memory for RGBA PNG file"));
			}
		    }


		  fclose (f);
		}
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _("couldn't create png end info struct"));
	    }
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("couldn't create png info struct"));
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("couldn't create png read struct"));
    }

  png_destroy_read_struct (&png_ptr, &info_ptr, &end_info);

  if (!ret)
    {
      _lw6ldr_rgba_clear (image);
    }

  return ret;
}

int
_lw6ldr_rgba_read_jpeg (_lw6ldr_image_rgba_t * image, char *jpeg_file)
{
  int ret = 0;
  FILE *f = NULL;
  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;
  JSAMPARRAY buffer;
  int row_stride;
  int i;
  int j;
  unsigned char **buf;
  int memory_ok;

  /*
   * This function probably lacks many error checks.
   */

  f = fopen (jpeg_file, "rb");
  if (f != NULL)
    {
      memset (&cinfo, 0, sizeof (cinfo));
      jpeg_create_decompress (&cinfo);
      cinfo.err = jpeg_std_error (&jerr);
      jpeg_stdio_src (&cinfo, f);
      jpeg_read_header (&cinfo, TRUE);
      jpeg_start_decompress (&cinfo);

      memory_ok = 1;
      buf =
	(unsigned char **) LW6SYS_MALLOC (cinfo.output_height *
					  sizeof (unsigned char *));
      if (buf)
	{
	  for (j = 0; j < cinfo.output_height; ++j)
	    {
	      buf[j] =
		(unsigned char *) LW6SYS_MALLOC (cinfo.output_width * 4 *
						 sizeof (unsigned char));
	      if (!buf[j])
		{
		  memory_ok = 0;
		}
	    }
	}
      else
	{
	  memory_ok = 0;
	}

      if (memory_ok)
	{
	  switch (cinfo.output_components)
	    {
	    case 3:		// color
	      row_stride = cinfo.output_width * cinfo.output_components;
	      buffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr) & cinfo, JPOOL_IMAGE, row_stride, 1);
	      while (cinfo.output_scanline < cinfo.output_height)
		{
		  j =
		    lw6sys_max (lw6sys_min
				(cinfo.output_scanline,
				 cinfo.output_height - 1), 0);

		  jpeg_read_scanlines (&cinfo, buffer, 1);
		  for (i = 0; i < cinfo.output_width; i++)
		    {
		      buf[j][i * 4 + 0] = buffer[0][i * 3 + 0];
		      buf[j][i * 4 + 1] = buffer[0][i * 3 + 1];
		      buf[j][i * 4 + 2] = buffer[0][i * 3 + 2];
		      buf[j][i * 4 + 3] = 255;
		    }
		}
	      image->shape.w = cinfo.output_width;
	      image->shape.h = cinfo.output_height;
	      image->data = buf;
	      ret = 1;
	      break;

	    case 1:		// greyscale
	      row_stride = cinfo.output_width * cinfo.output_components;
	      buffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr) & cinfo, JPOOL_IMAGE, row_stride, 1);
	      while (cinfo.output_scanline < cinfo.output_height)
		{
		  j =
		    lw6sys_max (lw6sys_min
				(cinfo.output_scanline,
				 cinfo.output_height - 1), 0);

		  jpeg_read_scanlines (&cinfo, buffer, 1);
		  for (i = 0; i < cinfo.output_width; i++)
		    {
		      buf[j][i * 4 + 0] =
			buf[j][i * 4 + 1] = buf[j][i * 4 + 2] = buffer[0][i];
		      buf[j][i * 4 + 3] = 255;
		    }
		}
	      image->shape.w = cinfo.output_width;
	      image->shape.h = cinfo.output_height;
	      image->data = buf;
	      ret = 1;
	      break;

	    default:
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _
			  ("unable to handle jpeg file output_components must be 3 (RGB) but is %d"),
			  cinfo.output_components);
	    }
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_ERROR,
		      _("unable to allocate memory for RGBA JPEG file"));
	}

      jpeg_finish_decompress (&cinfo);
      jpeg_destroy_decompress (&cinfo);

      fclose (f);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to read jpeg file \"%s\""), jpeg_file);
    }

  if (!ret)
    {
      _lw6ldr_rgba_clear (image);
    }

  return ret;
}


/*
 * Clear everything and free data pointers.
 */
extern void
_lw6ldr_rgba_clear (_lw6ldr_image_rgba_t * image)
{
  int row;

  if (image->data)
    {
      for (row = 0; row < image->shape.h; ++row)
	{
	  if (image->data[row])
	    {
	      LW6SYS_FREE (image->data[row]);
	    }
	}
      LW6SYS_FREE (image->data);
    }

  memset (image, 0, sizeof (_lw6ldr_image_rgba_t));
}
