/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>
#include <stdio.h>

#include "liquidwar6.h"

/*
 * Helper function, creates a 0 terminated string
 * from a Guile string. A very common task.
 * Returned pointer must be freed.
 */
static char *
to_0str (SCM string)
{
  char *c_string;
  int length;

  SCM_ASSERT (scm_is_string (string), string, SCM_ARG1, "");

  /*
   * See the comment in sys/sys-str.c to see why we use
   * 2 trailing '\0' at the end of the string.
   */
  length = scm_c_string_length (string);
  c_string = (char *) LW6SYS_MALLOC ((length + 2) * sizeof (char));

  if (c_string)
    {
      memcpy ((void *) c_string, (void *) scm_i_string_chars (string),
	      length * sizeof (char));
      c_string[length] = c_string[length + 1] = '\0';
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("unable to convert a guile SCM string to a standard C \"'\\0' terminated\" string"));
    }

  return c_string;
}

/*
 * For a GNU gettext-like behavior of scheme code.
 */
static SCM
scm_gettext (SCM string)
{
  char *c_string;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (string), string, SCM_ARG1, "c-gettext");

  c_string = to_0str (string);
  if (c_string)
    {
      ret = scm_makfrom0str (gettext (c_string));
      LW6SYS_FREE (c_string);
    }

  return ret;
}

/*
 * In liquidwar6sys
 */

/*
 * In bazooka.c
 */
static SCM
scm_lw6sys_set_memory_bazooka_size (SCM size)
{
  SCM ret = SCM_BOOL_F;
  int c_size;

  SCM_ASSERT (scm_is_integer (size), size, SCM_ARG1,
	      "c-lw6sys-set-memory-bazooka-size");

  c_size = scm_to_int (size);

  ret = lw6sys_set_memory_bazooka_size (c_size) ? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

static SCM
scm_lw6sys_set_memory_bazooka_eraser (SCM state)
{
  SCM ret = SCM_BOOL_F;
  int c_state;

  SCM_ASSERT (scm_is_bool (state), state, SCM_ARG1,
	      "c-lw6sys-set-memory-bazooka-eraser");

  c_state = scm_to_bool (state);

  ret = lw6sys_set_memory_bazooka_eraser (c_state) ? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

/*
 * build.c
 */
static SCM
scm_lw6sys_build_get_package_tarname ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_package_tarname ());

  return ret;
}

static SCM
scm_lw6sys_build_get_package_name ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_package_name ());

  return ret;
}

static SCM
scm_lw6sys_build_get_package_string ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_package_string ());

  return ret;
}

static SCM
scm_lw6sys_build_get_version ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_version ());

  return ret;
}

static SCM
scm_lw6sys_build_get_codename ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_codename ());

  return ret;
}

static SCM
scm_lw6sys_build_get_stamp ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_stamp ());

  return ret;
}

static SCM
scm_lw6sys_build_get_md5sum ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_md5sum ());

  return ret;
}

static SCM
scm_lw6sys_build_get_copyright ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_copyright ());

  return ret;
}

static SCM
scm_lw6sys_build_get_license ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_license ());

  return ret;
}

static SCM
scm_lw6sys_build_get_configure_args ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_configure_args ());

  return ret;
}

static SCM
scm_lw6sys_build_get_cflags ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_cflags ());

  return ret;
}

static SCM
scm_lw6sys_build_get_ldflags ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_ldflags ());

  return ret;
}

static SCM
scm_lw6sys_build_get_hostname ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_hostname ());

  return ret;
}

static SCM
scm_lw6sys_build_get_date ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_date ());

  return ret;
}

static SCM
scm_lw6sys_build_get_time ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_time ());

  return ret;
}

static SCM
scm_lw6sys_build_get_target_cpu ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_target_cpu ());

  return ret;
}

static SCM
scm_lw6sys_build_get_target_os ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_target_os ());

  return ret;
}

static SCM
scm_lw6sys_build_get_top_srcdir ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_top_srcdir ());

  return ret;
}

static SCM
scm_lw6sys_build_get_prefix ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_prefix ());

  return ret;
}

static SCM
scm_lw6sys_build_get_datadir ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_datadir ());

  return ret;
}

static SCM
scm_lw6sys_build_get_libdir ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_libdir ());

  return ret;
}

static SCM
scm_lw6sys_build_get_includedir ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_includedir ());

  return ret;
}

static SCM
scm_lw6sys_build_get_localedir ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_makfrom0str (lw6sys_build_get_localedir ());

  return ret;
}

/*
 * In dump.c
 */

static SCM
scm_lw6sys_dump (SCM content)
{
  char *c_content;
  char *user_dir = NULL;

  SCM_ASSERT (scm_is_string (content), content, SCM_ARG1, "c-lw6sys-dump");

  c_content = to_0str (content);
  if (c_content)
    {
      user_dir =
	lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
      if (user_dir)
	{
	  lw6sys_dump (user_dir, c_content);
	  LW6SYS_FREE (user_dir);
	}
      LW6SYS_FREE (c_content);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6sys_dump_clear ()
{
  char *user_dir = NULL;

  user_dir = lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (user_dir)
    {
      lw6sys_dump_clear (user_dir);
      LW6SYS_FREE (user_dir);
    }

  return SCM_UNDEFINED;
}

/*
 * In env.c
 */
static SCM
scm_lw6sys_get_username ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_username ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_hostname ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_hostname ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

/*
 * In log.c
 */

static SCM
scm_lw6sys_log (SCM level, SCM message)
{
  int c_level;
  char *c_message;

  SCM_ASSERT (scm_is_integer (level), level, SCM_ARG1, "c-lw6sys-log");
  SCM_ASSERT (scm_is_string (message), message, SCM_ARG2, "c-lw6sys-log");

  c_level = scm_to_int (level);
  c_message = to_0str (message);
  if (c_message)
    {
      lw6sys_log (c_level, __FILE__, __LINE__, "%s", c_message);
      LW6SYS_FREE (c_message);
    }

  return SCM_UNDEFINED;
}

/*
 * In option.c
 */
static SCM
scm_lw6sys_get_default_user_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_user_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_config_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_config_file ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_log_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_log_file ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_prefix ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_prefix ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_mod_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_mod_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_data_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_data_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_map_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_map_dir ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_map_path ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_map_path ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_default_script_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_default_script_file ();
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_user_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_config_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_config_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_log_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_log_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_prefix ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_prefix (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_mod_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_mod_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_data_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_data_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_map_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_map_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_map_path ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_map_path (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6sys_get_script_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6sys_get_script_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

/*
 * In path.c
 */
static SCM
scm_lw6sys_path_concat (SCM path1, SCM path2)
{
  SCM ret = SCM_BOOL_F;
  char *c_path1 = NULL;
  char *c_path2 = NULL;
  char *c_ret = NULL;

  SCM_ASSERT (scm_is_string (path1), path1, SCM_ARG1, "c-lw6sys-path-concat");
  SCM_ASSERT (scm_is_string (path2), path2, SCM_ARG2, "c-lw6sys-path-concat");

  c_path1 = to_0str (path1);
  if (c_path1)
    {
      c_path2 = to_0str (path2);
      if (c_path2)
	{
	  c_ret = lw6sys_path_concat (c_path1, c_path2);
	  if (c_ret)
	    {
	      ret = scm_makfrom0str (c_ret);
	      LW6SYS_FREE (c_ret);
	    }
	  LW6SYS_FREE (c_path2);
	}
      LW6SYS_FREE (c_path1);
    }

  return ret;
}

static SCM
scm_lw6sys_path_parent (SCM path)
{
  SCM ret = SCM_BOOL_F;
  char *c_path = NULL;
  char *c_ret = NULL;

  SCM_ASSERT (scm_is_string (path), path, SCM_ARG1, "c-lw6sys-path-parent");

  c_path = to_0str (path);
  if (c_path)
    {
      c_ret = lw6sys_path_parent (c_path);
      if (c_ret)
	{
	  ret = scm_makfrom0str (c_ret);
	  LW6SYS_FREE (c_ret);
	}
      LW6SYS_FREE (c_path);
    }

  return ret;
}

/*
 * In time.c
 */
static SCM
scm_lw6sys_timestamp ()
{
  SCM ret = SCM_BOOL_F;

  ret = scm_long_long2num (lw6sys_timestamp ());

  return ret;
}

static SCM
scm_lw6sys_sleep (SCM seconds)
{
  SCM ret = SCM_BOOL_F;
  float c_seconds;

  SCM_ASSERT (scm_is_number (seconds), seconds, SCM_ARG1, "c-lw6sys-sleep");

  c_seconds = scm_to_double (seconds);

  lw6sys_sleep (c_seconds);

  return ret;
}

/*
 * In load.c
 */
static SCM
scm_lw6cfg_load (SCM filename)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_filename;

      SCM_ASSERT (scm_is_string (filename),
		  filename, SCM_ARG1, "c-lw6cfg-load");

      c_filename = to_0str (filename);
      if (c_filename)
	{
	  if (lw6cfg_load (lw6_global.cfg_context, c_filename))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_filename);
	}
    }

  return ret;
}

/*
 * In option.c
 */
static SCM
scm_lw6cfg_option_exists (SCM key)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1,
		  "c-lw6cfg-option-exists");

      c_key = to_0str (key);
      if (c_key)
	{
	  if (lw6cfg_option_exists (lw6_global.cfg_context, c_key))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_key);
	}
    }

  return ret;
}

static SCM
scm_lw6cfg_get_option (SCM key)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, "c-lw6cfg-get-option");

      c_key = to_0str (key);
      if (c_key)
	{
	  char *value = NULL;

	  value = lw6cfg_get_option (lw6_global.cfg_context, c_key);
	  if (value)
	    {
	      ret =
		scm_makfrom0str (lw6cfg_get_option
				 (lw6_global.cfg_context, c_key));
	    }

	  LW6SYS_FREE (c_key);
	}
    }

  return ret;
}

static SCM
scm_lw6cfg_set_option (SCM key, SCM value)
{
  if (lw6_global.cfg_context != NULL)
    {
      char *c_key;
      char *c_value;

      SCM_ASSERT (scm_is_string (key), key, SCM_ARG1, "c-lw6cfg-set-option");

      SCM_ASSERT (scm_is_string (value),
		  value, SCM_ARG2, "c-lw6cfg-set-option");

      c_key = to_0str (key);
      if (c_key)
	{
	  c_value = to_0str (value);
	  if (c_value)
	    {
	      lw6cfg_set_option (lw6_global.cfg_context, c_key, c_value);

	      LW6SYS_FREE (c_value);
	    }
	  LW6SYS_FREE (c_key);
	}
    }

  return SCM_UNDEFINED;
}

/*
 * In save.c
 */
static SCM
scm_lw6cfg_save (SCM filename)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.cfg_context != NULL)
    {
      char *c_filename;

      SCM_ASSERT (scm_is_string (filename),
		  filename, SCM_ARG1, "c-lw6cfg-save");

      c_filename = to_0str (filename);
      if (c_filename)
	{
	  if (lw6cfg_save (lw6_global.cfg_context, c_filename))
	    {
	      ret = SCM_BOOL_T;
	    }
	  else
	    {
	      ret = SCM_BOOL_F;
	    }
	  LW6SYS_FREE (c_filename);
	}
    }

  return ret;
}

/*
 * In setup.c
 */
static SCM
scm_lw6cfg_init ()
{
  SCM ret = SCM_BOOL_F;

  lw6_global.cfg_context = lw6cfg_init (lw6_global.argc, lw6_global.argv);
  lw6_global.cfg_initialized = lw6_global.cfg_context ? 1 : 0;
  ret = lw6_global.cfg_context ? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

static SCM
scm_lw6cfg_quit ()
{
  if (lw6_global.cfg_context != NULL)
    {
      lw6cfg_quit (lw6_global.cfg_context);
    }

  lw6_global.cfg_context = NULL;
  lw6_global.cfg_initialized = 0;

  return SCM_UNDEFINED;
}

/*
 * Unified
 */

static SCM
scm_lw6cfg_unified_get_user_dir ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6cfg_unified_get_log_file ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6cfg_unified_get_log_file (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}

static SCM
scm_lw6cfg_unified_get_map_path ()
{
  SCM ret = SCM_BOOL_F;
  char *buf = NULL;

  buf = lw6cfg_unified_get_map_path (lw6_global.argc, lw6_global.argv);
  if (buf)
    {
      ret = scm_makfrom0str (buf);
      LW6SYS_FREE (buf);
    }

  return ret;
}



/*
 * In liquidwar6opt
 */

/*
 * In static.c
 */
static SCM
scm_lw6map_options_get_list ()
{
  SCM ret = SCM_EOL;
  int i = 0;

  while (LW6MAP_OPTIONS_LIST[i])
    {
      ret = scm_cons (scm_makfrom0str (LW6MAP_OPTIONS_LIST[i]), ret);
      ++i;
    }
  ret = scm_reverse (ret);

  return ret;
}

static SCM
scm_lw6map_options_get_int (SCM game_struct, SCM key)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_struct_t *c_game_struct;
      char *c_key;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_struct,
		   game_struct), game_struct, SCM_ARG1,
		  "c-lw6map-options-get-int");
      SCM_ASSERT (scm_is_string (key), key, SCM_ARG2,
		  "c-lw6map-options-get-int");

      c_game_struct = lw6_scm_to_game_struct (game_struct);
      if (c_game_struct)
	{
	  c_key = to_0str (key);
	  if (c_key)
	    {
	      ret =
		scm_int2num (lw6map_options_get_int
			     (&(c_game_struct->options), c_key));
	      LW6SYS_FREE (c_key);
	    }
	}
    }

  return ret;
}

/*
 * In liquidwar6gui
 */
static SCM
scm_lw6gui_menu_new (SCM title)
{
  char *c_title;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (title), title, SCM_ARG1, "c-lw6gui-menu-new");

  c_title = to_0str (title);
  if (c_title)
    {
      lw6gui_menu_t *c_menu;

      c_menu = lw6gui_menu_new (c_title);
      if (c_menu)
	{
	  ret = lw6_make_scm_menu (c_menu);
	}
      LW6SYS_FREE (c_title);
    }

  return ret;
}

static SCM
scm_lw6gui_menu_append (SCM menu, SCM menuitem)
{
  char *c_label;
  int c_value;
  int c_enabled;
  int c_selected;
  int c_colored;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, "c-lw6gui-menu-append");
  SCM_ASSERT (SCM_CONSP (menuitem)
	      || menuitem == SCM_EOL, menuitem, SCM_ARG2,
	      "c-lw6gui-menu-append");

  c_label = to_0str (scm_assoc_ref (menuitem, scm_makfrom0str ("label")));

  if (c_label)
    {
      lw6gui_menu_t *c_menu;
      int now = 0;

      c_value =
	scm_to_int (scm_assoc_ref (menuitem, scm_makfrom0str ("value")));
      c_enabled =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("enabled")));
      c_selected =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("selected")));
      c_colored =
	scm_to_bool (scm_assoc_ref (menuitem, scm_makfrom0str ("colored")));

      c_menu = lw6_scm_to_menu (menu);

      if (lw6_global.gfx_backend != NULL)
	{
	  now = lw6gfx_get_ticks (lw6_global.gfx_backend);
	}

      ret =
	scm_int2num (lw6gui_menu_append_for_id_use
		     (c_menu, c_label, c_value, c_enabled, c_selected,
		      c_colored, now));

      LW6SYS_FREE (c_label);
    }

  return ret;
}

static SCM
scm_lw6gui_menu_select (SCM menu, SCM position)
{
  lw6gui_menu_t *c_menu;
  int c_position;
  int now = 0;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, "c-lw6gui-menu-select");
  SCM_ASSERT (scm_is_integer (position), position, SCM_ARG2,
	      "c-lw6gui-menu-select");

  c_menu = lw6_scm_to_menu (menu);
  c_position = scm_to_int (position);

  if (lw6_global.gfx_backend != NULL)
    {
      now = lw6gfx_get_ticks (lw6_global.gfx_backend);
    }

  ret =
    lw6gui_menu_select (c_menu, c_position, now) ? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

static SCM
scm_lw6gui_menu_scroll_up (SCM menu)
{
  lw6gui_menu_t *c_menu;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, "c-lw6gui-menu-scroll-up");

  c_menu = lw6_scm_to_menu (menu);

  ret = lw6gui_menu_scroll_up (c_menu) ? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

static SCM
scm_lw6gui_menu_scroll_down (SCM menu)
{
  lw6gui_menu_t *c_menu;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.menu,
	       menu), menu, SCM_ARG1, "c-lw6gui-menu-scroll-down");

  c_menu = lw6_scm_to_menu (menu);

  ret = lw6gui_menu_scroll_down (c_menu) ? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

static SCM
scm_lw6gui_default_look ()
{
  lw6gui_look_t *c_look;
  SCM ret = SCM_BOOL_F;

  c_look = lw6gui_look_new (NULL);
  if (c_look)
    {
      ret = lw6_make_scm_look (c_look);
    }

  return ret;
}

static SCM
scm_lw6gui_look_set (SCM look, SCM key, SCM value)
{
  lw6gui_look_t *c_look;
  char *c_key;
  char *c_value;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.look,
	       look), look, SCM_ARG1, "c-lw6gui-look-set");
  SCM_ASSERT (scm_is_string (key), key, SCM_ARG2, "c-lw6gui-look-set");
  SCM_ASSERT (scm_is_string (value), value, SCM_ARG3, "c-lw6gui-look-set");

  c_look = lw6_scm_to_look (look);
  c_key = to_0str (key);
  if (c_key)
    {
      c_value = to_0str (value);
      if (c_value)
	{
	  ret = lw6gui_look_set (c_look, c_key, c_value) ? SCM_BOOL_T : SCM_BOOL_F;	// todo
	  LW6SYS_FREE (c_value);
	}
      LW6SYS_FREE (c_key);
    }

  return ret;
}

/*
 * In liquidwar6gfx
 */

/*
 * In backend.c
 */
static SCM
scm_lw6gfx_get_backends ()
{
  SCM ret = SCM_BOOL_F;
  lw6sys_assoc_t *backends;
  lw6sys_list_t *keys;
  lw6sys_list_t *key;
  char *module_id;
  char *module_name;

  backends = lw6gfx_get_backends (lw6_global.argc, lw6_global.argv);
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  return ret;
}

static SCM
scm_lw6gfx_create_backend (SCM backend_name)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend == NULL)
    {
      char *c_backend_name;

      SCM_ASSERT (scm_is_string (backend_name),
		  backend_name, SCM_ARG1, "c-lw6gfx-create-backend");

      c_backend_name = to_0str (backend_name);
      if (c_backend_name)
	{
	  lw6_global.gfx_backend =
	    lw6gfx_create_backend (lw6_global.argc, lw6_global.argv,
				   c_backend_name);
	  LW6SYS_FREE (c_backend_name);
	}
    }

  if (lw6_global.gfx_backend)
    {
      ret = SCM_BOOL_T;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_destroy_backend ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      if (lw6_global.gfx_initialized)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("destroying gfx backend, but backend has not been previously cleaned up"));
	  lw6gfx_quit (lw6_global.gfx_backend);
	}

      lw6gfx_destroy_backend (lw6_global.gfx_backend);
    }
  lw6_global.gfx_backend = NULL;
  lw6_global.gfx_initialized = 0;

  return SCM_UNDEFINED;
}

/*
 * In background.c
 */
static SCM
scm_lw6gfx_display_background (SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG1, "c-lw6gfx-display-background");

      c_look = lw6_scm_to_look (look);

      lw6gfx_display_background (lw6_global.gfx_backend, c_look);
    }

  return SCM_UNDEFINED;
}

/*
 * In event.c
 */
static SCM
scm_lw6gfx_poll_key ()
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_keypress_t keypress;

      if (lw6gfx_poll_key (lw6_global.gfx_backend, &keypress))
	{
	  ret =
	    scm_list_3 (scm_cons
			(scm_makfrom0str ("keysym"),
			 scm_int2num (keypress.keysym)),
			scm_cons (scm_makfrom0str ("unicode"),
				  scm_int2num (keypress.unicode)),
			scm_cons (scm_makfrom0str ("label"),
				  scm_makfrom0str (keypress.label)));
	}
      else
	{
	  ret = SCM_BOOL_F;
	}
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_esc (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-esc");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_esc (lw6_global.gfx_backend,
			   c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_enter (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-enter");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_enter (lw6_global.gfx_backend,
			     c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_up (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-up");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_up (lw6_global.gfx_backend,
			  c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_down (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-down");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_down (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_left (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-left");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_left (lw6_global.gfx_backend,
			    c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_is_key_right (SCM keysym)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;

      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-is-key-right");

      c_keysym = scm_to_int (keysym);

      ret =
	lw6gfx_is_key_right (lw6_global.gfx_backend,
			     c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_quit ()
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_quit (lw6_global.gfx_backend) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_send_quit ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_send_quit (lw6_global.gfx_backend);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_get_key_state (SCM keysym)
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_keysym;
      SCM_ASSERT (scm_is_integer (keysym), keysym, SCM_ARG1,
		  "c-lw6gfx-get-key-state");
      c_keysym = scm_to_int (keysym);
      ret =
	lw6gfx_get_key_state (lw6_global.gfx_backend,
			      c_keysym) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_mouse_move ()
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_x;
      int c_y;

      if (lw6gfx_poll_mouse_move (lw6_global.gfx_backend, &c_x, &c_y))
	{
	  ret = scm_list_2 (scm_cons
			    (scm_makfrom0str ("x"), scm_int2num (c_x)),
			    scm_cons (scm_makfrom0str ("y"),
				      scm_int2num (c_y)));
	}
      else
	{
	  ret = SCM_BOOL_F;
	}
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_mouse_button_left ()
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_button_left (lw6_global.
				       gfx_backend) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_mouse_button_right ()
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_button_right (lw6_global.
					gfx_backend) ? SCM_BOOL_T :
	SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_mouse_wheel_up ()
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_wheel_up (lw6_global.
				    gfx_backend) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_mouse_wheel_down ()
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      ret =
	lw6gfx_poll_mouse_wheel_down (lw6_global.
				      gfx_backend) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_get_mouse_state ()
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_x;
      int c_y;
      int c_button_left;
      int c_button_right;
      lw6gfx_get_mouse_state (lw6_global.gfx_backend,
			      &c_x, &c_y, &c_button_left, &c_button_right);
      ret =
	scm_list_4 (scm_cons
		    (scm_makfrom0str ("x"),
		     scm_int2num (c_x)),
		    scm_cons (scm_makfrom0str ("y"),
			      scm_int2num (c_y)),
		    scm_cons (scm_makfrom0str
			      ("button-left"),
			      c_button_left ? SCM_BOOL_T :
			      SCM_BOOL_F),
		    scm_cons (scm_makfrom0str
			      ("button-right"),
			      c_button_right ? SCM_BOOL_T : SCM_BOOL_F));
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_joystick_up (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, "c-lw6gfx-poll-joystick-up");

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_up (lw6_global.
				 gfx_backend, c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_joystick_down (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, "c-lw6gfx-poll-joystick-down");

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_down (lw6_global.
				   gfx_backend,
				   c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_joystick_left (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, "c-lw6gfx-poll-joystick-left");

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_left (lw6_global.
				   gfx_backend,
				   c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_joystick_right (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1,
	      "c-lw6gfx-poll-joystick-right");

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_right (lw6_global.
				    gfx_backend,
				    c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_joystick_button_a (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1,
	      "c-lw6gfx-poll-joystick-button-a");

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_button_a (lw6_global.
				       gfx_backend,
				       c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_poll_joystick_button_b (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1,
	      "c-lw6gfx-poll-joystick-button-b");

  if (lw6_global.gfx_backend != NULL)
    {
      c_i = scm_to_int (i);
      ret =
	lw6gfx_poll_joystick_button_b (lw6_global.
				       gfx_backend,
				       c_i) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_get_joystick_state (SCM i)
{
  SCM ret = SCM_BOOL_F;
  int c_i;

  SCM_ASSERT (scm_is_integer (i), i, SCM_ARG1, "c-lw6gfx-get-joystick-state");

  if (lw6_global.gfx_backend != NULL)
    {
      int c_x;
      int c_y;
      int c_button_left;
      int c_button_right;

      c_i = scm_to_int (i);
      lw6gfx_get_joystick_state (lw6_global.gfx_backend,
				 &c_x, &c_y, &c_button_left, &c_button_right,
				 c_i);
      ret =
	scm_list_4 (scm_cons (scm_makfrom0str ("x"), scm_int2num (c_x)),
		    scm_cons (scm_makfrom0str ("y"), scm_int2num (c_y)),
		    scm_cons (scm_makfrom0str ("button-a"),
			      c_button_left ? SCM_BOOL_T : SCM_BOOL_F),
		    scm_cons (scm_makfrom0str ("button-n"),
			      c_button_right ? SCM_BOOL_T : SCM_BOOL_F));
    }

  return ret;
}

/*
 * In hud.c
 */
static SCM
scm_lw6gfx_display_hud (SCM game_state, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6gfx-display-hud");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, "c-lw6gfx-display-hud");

      c_game_state = lw6_scm_to_game_state (game_state);
      c_look = lw6_scm_to_look (look);
      lw6gfx_display_hud (lw6_global.gfx_backend, c_look, c_game_state);
    }

  return SCM_UNDEFINED;
}

/*
 * In info.c
 */
static SCM
scm_lw6gfx_display_log (SCM log, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      int log_size;
      char **c_log;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_CONSP (log), log, SCM_ARG1, "c-lw6sys-display-log");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, "c-lw6gfx-display-log");

      c_look = lw6_scm_to_look (look);
      log_size = scm_to_int (scm_length (log));
      c_log = LW6SYS_MALLOC ((log_size + 1) * sizeof (char *));
      if (c_log)
	{
	  int i;
	  for (i = 0; i < log_size; ++i)
	    {
	      SCM str;
	      str = scm_list_ref (log, scm_int2num (i));
	      c_log[i] = to_0str (str);
	    }
	  c_log[log_size] = NULL;
	  lw6gfx_display_log (lw6_global.gfx_backend, c_look, c_log);
	  for (i = 0; i < log_size; ++i)
	    {
	      if (c_log[i])
		{
		  LW6SYS_FREE (c_log[i]);
		}
	    }

	  LW6SYS_FREE (c_log);
	}
    }

  return SCM_UNDEFINED;
}


static SCM
scm_lw6gfx_display_fps (SCM fps, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      int c_fps;
      lw6gui_look_t *c_look;

      SCM_ASSERT (scm_is_integer (fps), fps, SCM_ARG1,
		  "c-lw6sys-display-fps");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, "c-lw6gfx-display-fps");

      c_fps = scm_to_int (fps);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_fps (lw6_global.gfx_backend, c_look, c_fps);
    }

  return SCM_UNDEFINED;
}

/*
 * In map.c
 */
static SCM
scm_lw6gfx_display_map_preview (SCM level, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6map_level_t *c_level;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, level), level,
		  SCM_ARG1, "c-lw6gfx-display-map-preview");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, "c-lw6gfx-display-map-preview");

      c_level = lw6_scm_to_map (level);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_map_preview (lw6_global.gfx_backend, c_look, c_level);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_map_zones (SCM game_struct, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_struct_t *c_game_struct;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_struct,
		   game_struct), game_struct, SCM_ARG1,
		  "c-lw6gfx-display-map-zones");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, "c-lw6gfx-display-map-zones");

      c_game_struct = lw6_scm_to_game_struct (game_struct);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_map_zones (lw6_global.gfx_backend, c_look,
				c_game_struct);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_map_gradient (SCM game_state, SCM team_id, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      int c_team_id;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1,
		  "c-lw6gfx-display-map-gradient");
      SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
		  "c-lw6gfx-display-map-gradient");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG3, "c-lw6gfx-display-map-gradient");

      c_game_state = lw6_scm_to_game_state (game_state);
      c_team_id = scm_to_int (team_id);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_map_gradient (lw6_global.
				   gfx_backend, c_look,
				   c_game_state, c_team_id);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_display_game (SCM game_state, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1,
		  "c-lw6gfx-display-game");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, "c-lw6gfx-display-game");

      c_game_state = lw6_scm_to_game_state (game_state);
      c_look = lw6_scm_to_look (look);
      lw6gfx_display_game (lw6_global.gfx_backend, c_look, c_game_state);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_pick_map_coord (SCM game_state, SCM screen_x,
			   SCM screen_y, SCM look)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      int c_screen_x;
      int c_screen_y;
      lw6gui_look_t *c_look;
      int c_map_x;
      int c_map_y;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1,
		  "c-lw6gfx-pick-map-coord");
      SCM_ASSERT (scm_is_integer (screen_x), screen_x, SCM_ARG2,
		  "c-lw6gfx-pick-map-coord");
      SCM_ASSERT (scm_is_integer (screen_y), screen_y, SCM_ARG3,
		  "c-lw6gfx-pick-map-coord");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG4, "c-lw6gfx-pick-map-coord");

      c_game_state = lw6_scm_to_game_state (game_state);
      c_screen_x = scm_to_int (screen_x);
      c_screen_y = scm_to_int (screen_y);
      c_look = lw6_scm_to_look (look);

      lw6gfx_pick_map_coord (lw6_global.gfx_backend, c_look, &c_map_x,
			     &c_map_y, c_game_state, c_screen_x, c_screen_y);

      ret = scm_list_2 (scm_cons
			(scm_makfrom0str ("x"), scm_int2num (c_map_x)),
			scm_cons (scm_makfrom0str ("y"),
				  scm_int2num (c_map_y)));

    }

  return ret;
}

/*
 * In menu.c
 */
static SCM
scm_lw6gfx_display_menu (SCM menu, SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_menu_t *c_menu;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.menu,
		   menu), menu, SCM_ARG1, "c-lw6gfx-display-menu");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG2, "c-lw6gfx-display-menu");

      c_menu = lw6_scm_to_menu (menu);
      c_look = lw6_scm_to_look (look);

      lw6gfx_display_menu (lw6_global.gfx_backend, c_look, c_menu);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_pick_menuitem (SCM menu, SCM screen_x, SCM screen_y, SCM look)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      int c_position = 0;
      int c_scroll = 0;
      lw6gui_menu_t *c_menu;
      int c_screen_x;
      int c_screen_y;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.menu,
		   menu), menu, SCM_ARG1, "c-lw6gfx-pick-menuitem");
      SCM_ASSERT (scm_is_integer (screen_x), screen_x, SCM_ARG2,
		  "c-lw6gfx-pick-menuitem");
      SCM_ASSERT (scm_is_integer (screen_y), screen_y, SCM_ARG3,
		  "c-lw6gfx-pick-menuitem");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG4, "c-lw6gfx-pick-menuitem");

      c_menu = lw6_scm_to_menu (menu);
      c_screen_x = scm_to_int (screen_x);
      c_screen_y = scm_to_int (screen_y);
      c_look = lw6_scm_to_look (look);

      lw6gfx_pick_menuitem (lw6_global.gfx_backend, c_look, &c_position,
			    &c_scroll, c_menu, c_screen_x, c_screen_y);

      ret = scm_list_2
	(scm_cons
	 (scm_makfrom0str ("position"),
	  scm_int2num (c_position)),
	 scm_cons (scm_makfrom0str ("scroll"), scm_int2num (c_scroll)));
    }

  return ret;
}

static SCM
scm_lw6gfx_warp_mouse_on_menuitem (SCM menu, SCM position, SCM look)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_menu_t *c_menu;
      int c_position = 0;
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.menu,
		   menu), menu, SCM_ARG1, "c-lw6gfx-warp-mouse-on-menuitem");
      SCM_ASSERT (scm_is_integer (position), position, SCM_ARG2,
		  "c-lw6gfx-warp-mouse-on-menuitem");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG3, "c-lw6gfx-display-menu");

      c_menu = lw6_scm_to_menu (menu);
      c_position = scm_to_int (position);
      c_look = lw6_scm_to_look (look);

      ret =
	lw6gfx_warp_mouse_on_menuitem (lw6_global.gfx_backend, c_look, c_menu,
				       c_position) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

/*
 * In render.c
 */
static SCM
scm_lw6gfx_prepare_buffer (SCM look)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gui_look_t *c_look;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.look,
		   look), look, SCM_ARG1, "c-lw6gfx-prepare-buffer");

      c_look = lw6_scm_to_look (look);

      lw6gfx_prepare_buffer (lw6_global.gfx_backend, c_look);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_swap_buffers ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_swap_buffers (lw6_global.gfx_backend);
    }

  return SCM_UNDEFINED;
}

/*
 * In setup.c
 */
static SCM
scm_lw6gfx_init (SCM width, SCM height, SCM fullscreen, SCM ticks)
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;
      int c_fullscreen;
      int c_ticks;
      SCM_ASSERT (scm_is_integer (width), width, SCM_ARG1, "c-lw6gfx-init");
      SCM_ASSERT (scm_is_integer (height), height, SCM_ARG2, "c-lw6gfx-init");
      SCM_ASSERT (SCM_BOOLP (fullscreen), fullscreen, SCM_ARG3,
		  "c-lw6gfx-init");
      SCM_ASSERT (scm_is_integer (ticks), ticks, SCM_ARG4, "c-lw6gfx-init");
      c_width = scm_to_int (width);
      c_height = scm_to_int (height);
      c_fullscreen = SCM_NFALSEP (fullscreen);
      c_ticks = scm_to_int (ticks);
      if (lw6gfx_init (lw6_global.gfx_backend, c_width,
		       c_height, c_fullscreen, lw6_resize_callback, c_ticks))
	{
	  lw6_global.gfx_initialized = 1;
	  ret = SCM_BOOL_T;
	}
    }

  return ret;
}

static SCM
scm_lw6gfx_set_video_mode (SCM width, SCM height, SCM fullscreen)
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      int c_width;
      int c_height;
      int c_fullscreen;
      SCM_ASSERT (scm_is_integer (width),
		  width, SCM_ARG1, "c-lw6gfx-set-video-mode");
      SCM_ASSERT (scm_is_integer (height), height, SCM_ARG2,
		  "c-lw6gfx-set-video-mode");
      c_width = scm_to_int (width);
      c_height = scm_to_int (height);
      c_fullscreen = SCM_NFALSEP (fullscreen);
      ret =
	lw6gfx_set_video_mode (lw6_global.gfx_backend,
			       c_width, c_height,
			       c_fullscreen,
			       lw6_resize_callback) ? SCM_BOOL_T : SCM_BOOL_F;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6gfx_quit ()
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6gfx_quit (lw6_global.gfx_backend);
    }
  lw6_global.gfx_initialized = 0;

  return SCM_UNDEFINED;
}

/*
 * In timer.c
 */
static SCM
scm_lw6gfx_delay (SCM ms)
{
  int c_ms;
  if (lw6_global.gfx_backend != NULL)
    {
      SCM_ASSERT (scm_is_integer (ms), ms, SCM_ARG1, "c-lw6gfx-delay");
      c_ms = scm_to_int (ms);
      lw6gfx_delay (lw6_global.gfx_backend, c_ms);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6gfx_get_ticks ()
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      ret = scm_int2num (lw6gfx_get_ticks (lw6_global.gfx_backend));
    }

  return ret;
}




/*
 * In liquidwar6ldr
 */


static SCM
scm_lw6ldr_get_entries (SCM map_path, SCM relative_path)
{
  lw6sys_list_t *c_maps;
  lw6ldr_entry_t *c_entry;
  char *c_map_path;
  char *c_relative_path;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (map_path), relative_path, SCM_ARG1,
	      "c-lw6ldr-get-entries");
  SCM_ASSERT (scm_is_string (relative_path), relative_path, SCM_ARG2,
	      "c-lw6ldr-get-entries");

  ret = SCM_LIST0;
  c_map_path = to_0str (map_path);
  if (c_map_path)
    {
      c_relative_path = to_0str (relative_path);
      if (c_relative_path)
	{
	  c_maps = lw6ldr_get_entries (c_map_path, c_relative_path);
	  if (c_maps)
	    {
	      while (!lw6sys_list_is_empty (c_maps))
		{
		  c_entry = (lw6ldr_entry_t *) lw6sys_lifo_pop (&c_maps);
		  if (c_entry)
		    {
		      ret = scm_cons (scm_list_4
				      (scm_cons
				       (scm_makfrom0str ("title"),
					scm_makfrom0str (c_entry->
							 title)),
				       scm_cons (scm_makfrom0str
						 ("absolute-path"),
						 scm_makfrom0str
						 (c_entry->absolute_path)),
				       scm_cons (scm_makfrom0str
						 ("relative-path"),
						 scm_makfrom0str
						 (c_entry->relative_path)),
				       scm_cons (scm_makfrom0str
						 ("has-subdirs"),
						 c_entry->
						 has_subdirs ? SCM_BOOL_T :
						 SCM_BOOL_F)), ret);
		      lw6ldr_free_entry (c_entry);
		    }
		}
	      lw6sys_list_free (c_maps);
	    }
	  LW6SYS_FREE (c_relative_path);
	}
      LW6SYS_FREE (c_map_path);
    }

  ret = scm_reverse (ret);
  return ret;
}

static SCM
scm_lw6ldr_read (SCM dirname, SCM default_param, SCM forced_param,
		 SCM display_width, SCM display_height)
{
  char *c_dirname;
  int c_display_width;
  int c_display_height;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (dirname), dirname, SCM_ARG1, "c-lw6ldr-read");
  SCM_ASSERT (SCM_CONSP (default_param)
	      || default_param == SCM_EOL, default_param, SCM_ARG2,
	      "c-lw6ldr-read");
  SCM_ASSERT (SCM_CONSP (forced_param)
	      || forced_param == SCM_EOL, forced_param, SCM_ARG3,
	      "c-lw6ldr-read");
  SCM_ASSERT (scm_is_integer (display_width), display_width, SCM_ARG4,
	      "c-lw6ldr-read");
  SCM_ASSERT (scm_is_integer (display_height), display_height, SCM_ARG5,
	      "c-lw6ldr-read");

  c_dirname = to_0str (dirname);
  if (c_dirname)
    {
      lw6map_level_t *c_level;
      lw6sys_wh_t display_shape;

      c_display_width = scm_to_int (display_width);
      c_display_height = scm_to_int (display_height);

      display_shape.w = c_display_width;
      display_shape.h = c_display_height;

      // todo : get options from SCM obj
      c_level = lw6ldr_read (c_dirname, NULL, NULL, &display_shape);
      if (c_level)
	{
	  ret = lw6_make_scm_map (c_level);
	}
      LW6SYS_FREE (c_dirname);
    }

  return ret;
}

static SCM
scm_lw6ldr_read_relative (SCM map_path, SCM relative_path, SCM default_param,
			  SCM forced_param, SCM display_width,
			  SCM display_height)
{
  char *c_map_path;
  char *c_relative_path;
  int c_display_width;
  int c_display_height;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (map_path), map_path, SCM_ARG1,
	      "c-lw6ldr-read-relative");
  SCM_ASSERT (scm_is_string (relative_path), relative_path, SCM_ARG2,
	      "c-lw6ldr-read-relative");
  SCM_ASSERT (SCM_CONSP (default_param)
	      || default_param == SCM_EOL, default_param, SCM_ARG3,
	      "c-lw6ldr-read-relative");
  SCM_ASSERT (SCM_CONSP (forced_param)
	      || forced_param == SCM_EOL, forced_param, SCM_ARG4,
	      "c-lw6ldr-read-relative");
  SCM_ASSERT (scm_is_integer (display_width), display_width, SCM_ARG5,
	      "c-lw6ldr-read-relative");
  SCM_ASSERT (scm_is_integer (display_height), display_height, SCM_ARG6,
	      "c-lw6ldr-read-relative");

  c_map_path = to_0str (map_path);
  if (c_map_path)
    {
      c_relative_path = to_0str (relative_path);
      if (c_relative_path)
	{
	  lw6map_level_t *c_level;
	  lw6sys_wh_t display_shape;

	  c_display_width = scm_to_int (display_width);
	  c_display_height = scm_to_int (display_height);

	  display_shape.w = c_display_width;
	  display_shape.h = c_display_height;

	  // todo : get options from SCM obj
	  c_level =
	    lw6ldr_read_relative (c_map_path, c_relative_path, NULL, NULL,
				  &display_shape);
	  if (c_level)
	    {
	      ret = lw6_make_scm_map (c_level);
	    }
	  LW6SYS_FREE (c_relative_path);
	}
      LW6SYS_FREE (c_map_path);
    }

  return ret;
}

static SCM
scm_lw6ldr_print_examples ()
{
  char *user_dir = NULL;

  user_dir = lw6cfg_unified_get_user_dir (lw6_global.argc, lw6_global.argv);
  if (user_dir)
    {
      lw6ldr_print_examples (user_dir);
      LW6SYS_FREE (user_dir);
    }

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6map
 */

static SCM
scm_lw6map_get_background_style (SCM level)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6map_level_t *c_level;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, level), level,
		  SCM_ARG1, "c-lw6gfx-get-background-style");
      c_level = lw6_scm_to_map (level);
      if (c_level->param.style.background_style)
	{
	  ret = scm_makfrom0str (c_level->param.style.background_style);
	}
    }

  return ret;
}

static SCM
scm_lw6map_get_hud_style (SCM level)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6map_level_t *c_level;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, level), level,
		  SCM_ARG1, "c-lw6gfx-get-hud-style");
      c_level = lw6_scm_to_map (level);
      if (c_level->param.style.hud_style)
	{
	  ret = scm_makfrom0str (c_level->param.style.hud_style);
	}
    }

  return ret;
}

static SCM
scm_lw6map_get_menu_style (SCM level)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6map_level_t *c_level;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, level), level,
		  SCM_ARG1, "c-lw6gfx-get-menu-style");
      c_level = lw6_scm_to_map (level);
      if (c_level->param.style.menu_style)
	{
	  ret = scm_makfrom0str (c_level->param.style.menu_style);
	}
    }

  return ret;
}

static SCM
scm_lw6map_get_view_style (SCM level)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6map_level_t *c_level;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map, level), level,
		  SCM_ARG1, "c-lw6gfx-get-view-style");
      c_level = lw6_scm_to_map (level);
      if (c_level->param.style.view_style)
	{
	  ret = scm_makfrom0str (c_level->param.style.view_style);
	}
    }

  return ret;
}

/*
 * In liquidwar6ker
 */
static SCM
scm_lw6ker_build_game_struct (SCM level)
{
  lw6map_level_t *c_level;
  SCM ret = SCM_BOOL_F;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.map, level), level,
	      SCM_ARG1, "c-lw6ker-build-game-struct");
  c_level = lw6_scm_to_map (level);
  if (c_level)
    {
      lw6ker_game_struct_t *c_game_struct;

      c_game_struct = lw6ker_game_struct_new (c_level);
      if (c_game_struct)
	{
	  ret = lw6_make_scm_game_struct (c_game_struct, level);
	}
    }

  return ret;
}

static SCM
scm_lw6ker_build_game_state (SCM game_struct)
{
  lw6ker_game_struct_t *c_game_struct;
  SCM ret = SCM_BOOL_F;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_struct,
	       game_struct), game_struct, SCM_ARG1,
	      "c-lw6ker-build-game-state");
  c_game_struct = lw6_scm_to_game_struct (game_struct);
  if (c_game_struct)
    {
      lw6ker_game_state_t *c_game_state;
      c_game_state = lw6ker_game_state_new (c_game_struct);
      if (c_game_state)
	{
	  ret = lw6_make_scm_game_state (c_game_state, game_struct);
	}
    }

  return ret;
}

static SCM
scm_lw6ker_copy_game_state (SCM dst, SCM src)
{
  lw6ker_game_state_t *c_dst;
  lw6ker_game_state_t *c_src;
  SCM ret = SCM_BOOL_F;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       dst), dst, SCM_ARG1, "c-lw6ker-copy-game-state");
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       src), src, SCM_ARG2, "c-lw6ker-copy-game-state");
  c_dst = lw6_scm_to_game_state (dst);
  c_src = lw6_scm_to_game_state (src);
  if (c_dst && c_src)
    {
      ret = lw6ker_game_state_copy (c_dst, c_src) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6ker_game_struct_checksum (SCM game_struct)
{
  lw6ker_game_struct_t *c_game_struct;
  SCM ret = SCM_BOOL_F;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_struct,
	       game_struct), game_struct, SCM_ARG1,
	      "c-lw6ker-game-struct-checksum");
  c_game_struct = lw6_scm_to_game_struct (game_struct);
  if (c_game_struct)
    {
      ret = scm_int2num (lw6ker_game_struct_checksum (c_game_struct));
    }

  return ret;
}

static SCM
scm_lw6ker_game_state_checksum (SCM game_state)
{
  lw6ker_game_state_t *c_game_state;
  SCM ret = SCM_BOOL_F;
  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1,
	      "c-lw6ker-game-state-checksum");
  c_game_state = lw6_scm_to_game_state (game_state);
  if (c_game_state)
    {
      ret = scm_int2num (lw6ker_game_state_checksum (c_game_state));
    }

  return ret;
}

static SCM
scm_lw6ker_get_cursors (SCM game_state, SCM team_id)
{
  lw6ker_game_state_t *c_game_state;
  int c_team_id;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-get-cursors");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-get-cursors");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      int i;
      lw6ker_cursor_array_t *c_cursor_array =
	&(c_game_state->map_state.teams[c_team_id].cursor_array);
      ret = SCM_EOL;

      for (i = 0; i < c_cursor_array->nb_cursors; ++i)
	{
	  ret =
	    scm_cons (scm_list_3 (scm_cons (scm_makfrom0str ("x"),
					    scm_int2num (c_cursor_array->
							 cursors[i].pos.x)),
				  scm_cons (scm_makfrom0str ("y"),
					    scm_int2num (c_cursor_array->
							 cursors[i].pos.y)),
				  scm_cons (scm_makfrom0str ("pot-offset"),
					    scm_int2num (c_cursor_array->
							 cursors[i].
							 pot_offset))), ret);
	}
      ret = scm_reverse (ret);
    }

  return ret;
}

static SCM
scm_lw6ker_set_cursors (SCM game_state, SCM team_id, SCM cursor_array)
{
  lw6ker_game_state_t *c_game_state;
  int c_team_id;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-set-cursors");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-set-cursors");
  SCM_ASSERT (SCM_CONSP (cursor_array)
	      || cursor_array == SCM_EOL, cursor_array, SCM_ARG3,
	      "c-lw6ker-set-cursors");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      int i = 0;
      int n;
      lw6ker_cursor_array_t *c_cursor_array =
	&(c_game_state->map_state.teams[c_team_id].cursor_array);
      SCM cursor;

      n = scm_to_int (scm_length (cursor_array));
      for (i = 0; i < n; ++i)
	{
	  cursor = scm_list_ref (cursor_array, scm_int2num (i));
	  lw6ker_cursor_array_set (c_cursor_array,
				   i,
				   scm_to_int (scm_assoc_ref
					       (cursor,
						scm_makfrom0str ("x"))),
				   scm_to_int (scm_assoc_ref
					       (cursor,
						scm_makfrom0str ("y"))),
				   scm_to_int (scm_assoc_ref
					       (cursor,
						scm_makfrom0str
						("pot-offset"))),
				   &(c_game_state->game_struct->map_struct),
				   c_game_state->game_struct->options.
				   max_cursor_pot_offset);
	}
      lw6ker_cursor_array_force_nb_cursors (c_cursor_array, n);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6ker_enable_bot (SCM game_state, SCM team_id)
{
  SCM ret = SCM_BOOL_F;
  lw6ker_game_state_t *c_game_state;
  int c_team_id;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-enable-bot");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-enable-bot");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      ret =
	lw6ker_game_state_enable_bot (c_game_state,
				      c_team_id) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6ker_disable_bot (SCM game_state, SCM team_id)
{
  SCM ret = SCM_BOOL_F;
  lw6ker_game_state_t *c_game_state;
  int c_team_id;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.game_state,
	       game_state), game_state, SCM_ARG1, "c-lw6ker-disable-bot");
  SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
	      "c-lw6ker-disable-bot");

  c_game_state = lw6_scm_to_game_state (game_state);
  c_team_id = scm_to_int (team_id);
  if (c_game_state)
    {
      ret =
	lw6ker_game_state_disable_bot (c_game_state,
				       c_team_id) ? SCM_BOOL_T : SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6ker_do_round (SCM game_state)
{
  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-do-round");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  lw6ker_game_state_do_round (c_game_state);
	}
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6ker_get_moves (SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-get-moves");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  ret = scm_int2num (lw6ker_game_state_get_moves (c_game_state));
	}
    }

  return ret;
}

static SCM
scm_lw6ker_get_spreads (SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-get-spreads");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  ret = scm_int2num (lw6ker_game_state_get_spreads (c_game_state));
	}
    }

  return ret;
}

static SCM
scm_lw6ker_get_rounds (SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-get-rounds");
      c_game_state = lw6_scm_to_game_state (game_state);
      if (c_game_state)
	{
	  ret = scm_int2num (lw6ker_game_state_get_rounds (c_game_state));
	}
    }

  return ret;
}

static SCM
scm_lw6ker_add_team (SCM game_state, SCM team_id, SCM nb_cursors)
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      int c_team_id;
      int c_nb_cursors;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-add-team");
      SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
		  "c-lw6ker-add-team");
      SCM_ASSERT (scm_is_integer (nb_cursors), nb_cursors,
		  SCM_ARG3, "c-lw6ker-add-team");
      c_game_state = lw6_scm_to_game_state (game_state);
      c_team_id = scm_to_int (team_id);
      c_nb_cursors = scm_to_int (nb_cursors);
      if (c_game_state)
	{
	  ret =
	    lw6ker_game_state_add_team (c_game_state,
					c_team_id,
					c_nb_cursors) ?
	    SCM_BOOL_T : SCM_BOOL_F;
	}
    }

  return ret;
}

static SCM
scm_lw6ker_remove_team (SCM game_state, SCM team_id, SCM nb_cursors)
{
  SCM ret = SCM_BOOL_F;
  if (lw6_global.gfx_backend != NULL)
    {
      lw6ker_game_state_t *c_game_state;
      int c_team_id;
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG1, "c-lw6ker-remove-team");
      SCM_ASSERT (scm_is_integer (team_id), team_id, SCM_ARG2,
		  "c-lw6ker-remove-team");
      c_game_state = lw6_scm_to_game_state (game_state);
      c_team_id = scm_to_int (team_id);
      if (c_game_state)
	{
	  ret =
	    lw6ker_game_state_remove_team (c_game_state,
					   c_team_id) ?
	    SCM_BOOL_T : SCM_BOOL_F;
	}
    }

  return ret;
}

/*
 * In liquidwar6snd
 */

/*
 * In backend.c
 */
static SCM
scm_lw6snd_get_backends ()
{
  SCM ret = SCM_BOOL_F;
  lw6sys_assoc_t *backends;
  lw6sys_list_t *keys;
  lw6sys_list_t *key;
  char *module_id;
  char *module_name;

  backends = lw6snd_get_backends (lw6_global.argc, lw6_global.argv);
  if (backends)
    {
      keys = lw6sys_assoc_keys (backends);
      if (keys)
	{
	  ret = SCM_EOL;
	  key = keys;
	  while (key)
	    {
	      if (key->data)
		{
		  module_id = (char *) key->data;
		  module_name =
		    (char *) lw6sys_assoc_get (backends, module_id);
		  ret =
		    scm_cons (scm_cons
			      (scm_makfrom0str (module_id),
			       scm_makfrom0str (module_name)), ret);
		}
	      key = lw6sys_list_next (key);
	    }
	  lw6sys_list_free (keys);
	  ret = scm_reverse (ret);
	}
      lw6sys_assoc_free (backends);
    }

  return ret;
}

static SCM
scm_lw6snd_create_backend (SCM backend_name)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.snd_backend == NULL)
    {
      char *c_backend_name;

      SCM_ASSERT (scm_is_string (backend_name),
		  backend_name, SCM_ARG1, "c-lw6snd-create-backend");

      c_backend_name = to_0str (backend_name);
      if (c_backend_name)
	{
	  lw6_global.snd_backend =
	    lw6snd_create_backend (lw6_global.argc, lw6_global.argv,
				   c_backend_name);
	  LW6SYS_FREE (c_backend_name);
	}
    }

  if (lw6_global.snd_backend)
    {
      ret = SCM_BOOL_T;
    }
  else
    {
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6snd_destroy_backend ()
{
  if (lw6_global.snd_backend != NULL)
    {
      if (lw6_global.snd_initialized)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _
		      ("destroying snd backend, but backend has not been previously cleaned up"));
	  lw6snd_quit (lw6_global.snd_backend);
	}

      lw6snd_destroy_backend (lw6_global.snd_backend);
    }
  lw6_global.snd_backend = NULL;
  lw6_global.snd_initialized = 0;

  return SCM_UNDEFINED;
}

/*
 * In setup.c
 */
static SCM
scm_lw6snd_init (SCM sound_volume, SCM music_volume)
{
  SCM_ASSERT (SCM_REALP (sound_volume), sound_volume, SCM_ARG1,
	      "c-lw6snd-init");
  SCM_ASSERT (SCM_REALP (music_volume), music_volume, SCM_ARG2,
	      "c-lw6snd-init");

  if (lw6_global.snd_backend != NULL)
    {
      float c_sound_volume;
      float c_music_volume;

      c_sound_volume = scm_num2float (sound_volume, 0, NULL);
      c_music_volume = scm_num2float (music_volume, 0, NULL);

      lw6snd_init (lw6_global.snd_backend, c_sound_volume, c_music_volume);
      lw6_global.snd_initialized = 1;
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6snd_quit ()
{
  if (lw6_global.snd_backend != NULL)
    {
      lw6snd_quit (lw6_global.snd_backend);
    }
  lw6_global.snd_initialized = 0;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6snd_play_sound (SCM sound_id)
{
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_integer (sound_id), sound_id, SCM_ARG1,
	      "c-lw6snd-play-sound");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      int c_sound_id;

      c_sound_id = scm_to_int (sound_id);

      ret =
	scm_int2num (lw6snd_play_sound (lw6_global.snd_backend, c_sound_id));
    }

  return ret;
}

static SCM
scm_lw6snd_set_sound_volume (SCM sound_volume)
{
  SCM_ASSERT (SCM_REALP (sound_volume), sound_volume, SCM_ARG1,
	      "c-lw6snd-set-sound-volume");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      float c_sound_volume;

      c_sound_volume = scm_num2float (sound_volume, 0, NULL);
      lw6snd_set_sound_volume (lw6_global.snd_backend, c_sound_volume);
    }

  return SCM_UNDEFINED;
}

static SCM
scm_lw6snd_play_music (SCM music_name)
{
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_string (music_name), music_name, SCM_ARG1,
	      "c-lw6snd-play-music");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      char *c_music_name;

      c_music_name = to_0str (music_name);
      if (c_music_name)
	{
	  ret =
	    scm_int2num (lw6snd_play_music
			 (lw6_global.snd_backend, c_music_name));
	  LW6SYS_FREE (c_music_name);
	}
    }

  return ret;
}

static SCM
scm_lw6snd_set_music_volume (SCM music_volume)
{
  SCM_ASSERT (SCM_REALP (music_volume), music_volume, SCM_ARG1,
	      "c-lw6snd-set-music-volume");

  if (lw6_global.snd_backend != NULL && lw6_global.snd_initialized)
    {
      float c_music_volume;

      c_music_volume = scm_num2float (music_volume, 0, NULL);
      lw6snd_set_music_volume (lw6_global.snd_backend, c_music_volume);
    }

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6cns
 */

/*
 * In backend.c
 */
static SCM
scm_lw6cns_init ()
{
  lw6cns_handler_install (lw6_cns_handler);
  lw6_global.cns_initialized = 1;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6cns_quit ()
{
  lw6cns_handler_remove ();
  lw6_global.cns_initialized = 0;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6cns_poll ()
{
  lw6cns_handler_poll ();

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6tsk
 */
static SCM
scm_lw6tsk_loader_new (SCM sleep)
{
  float c_sleep;
  lw6tsk_loader_t *c_loader;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (scm_is_number (sleep), sleep, SCM_ARG1, "c-lw6tsk-loader-new");

  c_sleep = scm_to_double (sleep);
  c_loader = lw6tsk_loader_new (c_sleep);
  if (c_loader)
    {
      ret = lw6_make_scm_loader (c_loader);
    }

  return ret;
}

static SCM
scm_lw6tsk_loader_push (SCM loader, SCM map_path, SCM relative_path,
			SCM default_param, SCM forced_param,
			SCM display_width, SCM display_height)
{
  lw6tsk_loader_t *c_loader;
  char *c_map_path;
  char *c_relative_path;
  lw6sys_assoc_t *c_default_param;
  lw6sys_assoc_t *c_forced_param;
  int c_display_width;
  int c_display_height;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.loader,
	       loader), loader, SCM_ARG1, "c-lw6tsk-loader-push");
  SCM_ASSERT (scm_is_string (map_path), map_path, SCM_ARG2,
	      "c-lw6tsk-loader-push");
  SCM_ASSERT (scm_is_string (relative_path), relative_path, SCM_ARG3,
	      "c-lw6tsk-loader-push");
  SCM_ASSERT (SCM_CONSP (default_param)
	      || default_param == SCM_EOL, default_param, SCM_ARG4,
	      "c-lw6tsk-loader-push");
  SCM_ASSERT (SCM_CONSP (forced_param)
	      || forced_param == SCM_EOL, forced_param, SCM_ARG5,
	      "c-lw6tsk-loader-push");
  SCM_ASSERT (scm_is_integer (display_width), display_width, SCM_ARG6,
	      "c-lw6tsk-loader-push");
  SCM_ASSERT (scm_is_integer (display_height), display_height, SCM_ARG7,
	      "c-lw6tsk-loader-push");

  c_loader = lw6_scm_to_loader (loader);
  c_map_path = to_0str (map_path);
  if (c_map_path)
    {
      c_relative_path = to_0str (relative_path);
      if (c_relative_path)
	{
	  c_default_param = lw6sys_assoc_new (lw6sys_free_callback);
	  if (c_default_param)
	    {
	      c_forced_param = lw6sys_assoc_new (lw6sys_free_callback);
	      if (c_forced_param)
		{
		  lw6sys_wh_t display_shape;

		  c_display_width = scm_to_int (display_width);
		  c_display_height = scm_to_int (display_height);

		  display_shape.w = c_display_width;
		  display_shape.h = c_display_height;

		  lw6tsk_loader_push (c_loader, c_map_path, c_relative_path,
				      c_default_param, c_forced_param,
				      &display_shape);
		  ret = SCM_BOOL_T;

		  lw6sys_assoc_free (c_forced_param);
		}
	      lw6sys_assoc_free (c_default_param);
	    }
	  LW6SYS_FREE (c_relative_path);
	}
      LW6SYS_FREE (c_map_path);
    }

  return ret;
}

static SCM
scm_lw6tsk_loader_pop (SCM loader)
{
  lw6tsk_loader_t *c_loader;
  lw6map_level_t *c_level = NULL;
  lw6ker_game_struct_t *c_game_struct = NULL;
  lw6ker_game_state_t *c_game_state = NULL;
  SCM level = SCM_BOOL_F;
  SCM game_struct = SCM_BOOL_F;
  SCM game_state = SCM_BOOL_F;
  SCM ret = SCM_BOOL_F;

  SCM_ASSERT (SCM_SMOB_PREDICATE
	      (lw6_global.smob_types.loader,
	       loader), loader, SCM_ARG1, "c-lw6tsk-loader-pop");

  c_loader = lw6_scm_to_loader (loader);

  if (lw6tsk_loader_pop (&c_level, &c_game_struct, &c_game_state, c_loader))
    {
      if (c_level)
	{
	  level = lw6_make_scm_map (c_level);
	  if (c_game_struct)
	    {
	      game_struct = lw6_make_scm_game_struct (c_game_struct, level);
	      if (c_game_state)
		{
		  game_state =
		    lw6_make_scm_game_state (c_game_state, game_struct);
		}
	    }
	}
      ret =
	scm_list_3 (scm_cons
		    (scm_makfrom0str ("level"), level),
		    scm_cons
		    (scm_makfrom0str ("game-struct"), game_struct),
		    scm_cons (scm_makfrom0str ("game-state"), game_state));
    }

  return ret;
}



/*
 * In liquidwar6net
 */

/*
 * In setup.c
 */
static SCM
scm_lw6net_init ()
{
  SCM ret = SCM_BOOL_F;

  if (lw6net_init (lw6_global.argc, lw6_global.argv))
    {
      lw6_global.net_initialized = 1;
      ret = SCM_BOOL_F;
    }

  return ret;
}

static SCM
scm_lw6net_quit ()
{
  if (lw6_global.net_initialized)
    {
      lw6net_quit ();
    }
  lw6_global.net_initialized = 0;

  return SCM_UNDEFINED;
}

static SCM
scm_lw6net_server_start (SCM ip, SCM port)
{
  SCM ret = SCM_BOOL_F;
  char *c_ip;
  int c_port;

  SCM_ASSERT (scm_is_string (ip), ip, SCM_ARG1, "c-lw6net-server-start");
  SCM_ASSERT (scm_is_integer (port), port, SCM_ARG2, "c-lw6net-server-start");

  if (lw6_global.net_initialized)
    {
      c_ip = to_0str (ip);
      if (c_ip)
	{
	  c_port = scm_to_int (port);
	  /*
	     ret =
	     lw6net_server_start (lw6_global.net_context, c_ip,
	     c_port) ? SCM_BOOL_T : SCM_BOOL_F;
	   */
	  ret = SCM_BOOL_T;

	  LW6SYS_FREE (c_ip);
	}
    }

  return ret;
}

static SCM
scm_lw6net_server_poll_idle ()
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.net_initialized)
    {
      /*
         ret =
         lw6net_server_poll (lw6_global.net_context,
         NULL, NULL) ? SCM_BOOL_T : SCM_BOOL_F;
       */
      ret = SCM_BOOL_T;
    }

  return ret;
}

static SCM
scm_lw6net_server_poll_playing (SCM level, SCM game_state)
{
  SCM ret = SCM_BOOL_F;

  if (lw6_global.net_initialized)
    {
      lw6map_level_t *c_level;
      lw6ker_game_state_t *c_game_state;

      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.map,
		   level), level, SCM_ARG1, "c-lw6net-server-poll");
      SCM_ASSERT (SCM_SMOB_PREDICATE
		  (lw6_global.smob_types.game_state,
		   game_state), game_state, SCM_ARG2, "c-lw6net-server-poll");

      c_level = lw6_scm_to_map (level);
      c_game_state = lw6_scm_to_game_state (game_state);

      /*
         ret =
         lw6net_server_poll (lw6_global.net_context,
         c_level, c_game_state) ? SCM_BOOL_T : SCM_BOOL_F;
       */
      ret = SCM_BOOL_T;
    }

  return ret;
}

static SCM
scm_lw6net_server_stop ()
{
  if (lw6_global.net_initialized)
    {
      /*
         lw6net_server_stop (lw6_global.net_context);
       */
    }

  return SCM_UNDEFINED;
}

/*
 * Register all the functions, make them callable from Guile.
 */
int
lw6_register_funcs ()
{
  int ret = 1;
  scm_c_define_gsubr ("c-gettext", 1, 0, 0, (SCM (*)())scm_gettext);
  scm_c_define_gsubr ("_", 1, 0, 0, (SCM (*)())scm_gettext);

  /*
   * In liquidwar6sys
   */
  /*
   * In bazooka.c
   */
  scm_c_define_gsubr ("c-lw6sys-set-memory-bazooka-size", 1, 0, 0,
		      (SCM (*)())scm_lw6sys_set_memory_bazooka_size);
  scm_c_define_gsubr ("c-lw6sys-set-memory-bazooka-eraser", 1, 0, 0,
		      (SCM (*)())scm_lw6sys_set_memory_bazooka_eraser);
  /*
   * In build.c
   */
  scm_c_define_gsubr ("c-lw6sys-build-get-package-tarname", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_package_tarname);
  scm_c_define_gsubr ("c-lw6sys-build-get-package-name", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_package_name);
  scm_c_define_gsubr ("c-lw6sys-build-get-package-string", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_package_string);
  scm_c_define_gsubr ("c-lw6sys-build-get-version", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_version);
  scm_c_define_gsubr ("c-lw6sys-build-get-codename", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_codename);
  scm_c_define_gsubr ("c-lw6sys-build-get-stamp", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_stamp);
  scm_c_define_gsubr ("c-lw6sys-build-get-md5sum", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_md5sum);
  scm_c_define_gsubr ("c-lw6sys-build-get-copyright", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_copyright);
  scm_c_define_gsubr ("c-lw6sys-build-get-license", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_license);
  scm_c_define_gsubr ("c-lw6sys-build-get-configure-args", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_configure_args);
  scm_c_define_gsubr ("c-lw6sys-build-get-cflags", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_cflags);
  scm_c_define_gsubr ("c-lw6sys-build-get-ldflags", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_ldflags);
  scm_c_define_gsubr ("c-lw6sys-build-get-hostname", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_hostname);
  scm_c_define_gsubr ("c-lw6sys-build-get-date", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_date);
  scm_c_define_gsubr ("c-lw6sys-build-get-time", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_time);
  scm_c_define_gsubr ("c-lw6sys-build-get-target-cpu", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_target_cpu);
  scm_c_define_gsubr ("c-lw6sys-build-get-target-os", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_target_os);
  scm_c_define_gsubr ("c-lw6sys-build-get-top-srcdir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_top_srcdir);
  scm_c_define_gsubr ("c-lw6sys-build-get-prefix", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_prefix);
  scm_c_define_gsubr ("c-lw6sys-build-get-datadir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_datadir);
  scm_c_define_gsubr ("c-lw6sys-build-get-libdir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_libdir);
  scm_c_define_gsubr ("c-lw6sys-build-get-includedir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_includedir);
  scm_c_define_gsubr ("c-lw6sys-build-get-localedir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_build_get_localedir);
  /*
   * In dump.c
   */
  scm_c_define_gsubr ("c-lw6sys-dump", 1, 0, 0, (SCM (*)())scm_lw6sys_dump);
  scm_c_define_gsubr ("c-lw6sys-dump-clear", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_dump_clear);
  /*
   * In env.c
   */
  scm_c_define_gsubr ("c-lw6sys-get-username", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_username);
  scm_c_define_gsubr ("c-lw6sys-get-hostname", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_hostname);
  /*
   * In log.c
   */
  scm_c_define_gsubr ("c-lw6sys-log", 2, 0, 0, (SCM (*)())scm_lw6sys_log);
  /*
   * In options.c
   */
  scm_c_define_gsubr ("c-lw6sys-get-default-user-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_user_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-config-file", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_config_file);
  scm_c_define_gsubr ("c-lw6sys-get-default-log-file", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_log_file);
  scm_c_define_gsubr ("c-lw6sys-get-default-prefix", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_prefix);
  scm_c_define_gsubr ("c-lw6sys-get-default-mod-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_mod_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-data-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_data_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-map-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_map_dir);
  scm_c_define_gsubr ("c-lw6sys-get-default-map-path", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_map_path);
  scm_c_define_gsubr ("c-lw6sys-get-default-script-file", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_default_script_file);
  scm_c_define_gsubr ("c-lw6sys-get-user-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_user_dir);
  scm_c_define_gsubr ("c-lw6sys-get-config-file", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_config_file);
  scm_c_define_gsubr ("c-lw6sys-get-log-file", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_log_file);
  scm_c_define_gsubr ("c-lw6sys-get-prefix", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_prefix);
  scm_c_define_gsubr ("c-lw6sys-get-mod-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_mod_dir);
  scm_c_define_gsubr ("c-lw6sys-get-data-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_data_dir);
  scm_c_define_gsubr ("c-lw6sys-get-map-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_map_dir);
  scm_c_define_gsubr ("c-lw6sys-get-map-path", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_map_path);
  scm_c_define_gsubr ("c-lw6sys-get-script-file", 0, 0, 0,
		      (SCM (*)())scm_lw6sys_get_script_file);
  /*
   * In path.c
   */
  scm_c_define_gsubr ("c-lw6sys-path-concat", 2, 0, 0,
		      (SCM (*)())scm_lw6sys_path_concat);
  scm_c_define_gsubr ("c-lw6sys-path-parent", 1, 0, 0,
		      (SCM (*)())scm_lw6sys_path_parent);
  /*
   * In time.c
   */
  scm_c_define_gsubr ("c-lw6sys-timestamp",
		      0, 0, 0, (SCM (*)())scm_lw6sys_timestamp);
  scm_c_define_gsubr ("c-lw6sys-sleep", 1, 0, 0, (SCM (*)())scm_lw6sys_sleep);

  /*
   * In liquidwar6cfg
   */
  /*
   * In load.c
   */
  scm_c_define_gsubr ("c-lw6cfg-load", 1, 0, 0, (SCM (*)())scm_lw6cfg_load);
  /*
   * In option.c
   */
  scm_c_define_gsubr ("c-lw6cfg-option-exists",
		      1, 0, 0, (SCM (*)())scm_lw6cfg_option_exists);
  scm_c_define_gsubr ("c-lw6cfg-get-option", 1, 0, 0,
		      (SCM (*)())scm_lw6cfg_get_option);
  scm_c_define_gsubr ("c-lw6cfg-set-option", 2, 0, 0,
		      (SCM (*)())scm_lw6cfg_set_option);
  /*
   * In save.c
   */
  scm_c_define_gsubr ("c-lw6cfg-save", 1, 0, 0, (SCM (*)())scm_lw6cfg_save);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6cfg-init", 0, 0, 0, (SCM (*)())scm_lw6cfg_init);
  scm_c_define_gsubr ("c-lw6cfg-quit", 0, 0, 0, (SCM (*)())scm_lw6cfg_quit);
  /*
   * In unified.c
   */
  scm_c_define_gsubr ("c-lw6cfg-unified-get-user-dir", 0, 0, 0,
		      (SCM (*)())scm_lw6cfg_unified_get_user_dir);
  scm_c_define_gsubr ("c-lw6cfg-unified-get-log-file", 0, 0, 0,
		      (SCM (*)())scm_lw6cfg_unified_get_log_file);
  scm_c_define_gsubr ("c-lw6cfg-unified-get-map-path", 0, 0, 0,
		      (SCM (*)())scm_lw6cfg_unified_get_map_path);

  /*
   * In liquidwar6gui
   */
  scm_c_define_gsubr ("c-lw6gui-menu-new", 1, 0, 0,
		      (SCM (*)())scm_lw6gui_menu_new);
  scm_c_define_gsubr ("c-lw6gui-menu-append", 2, 0, 0,
		      (SCM (*)())scm_lw6gui_menu_append);
  scm_c_define_gsubr ("c-lw6gui-menu-select", 2, 0, 0,
		      (SCM (*)())scm_lw6gui_menu_select);
  scm_c_define_gsubr ("c-lw6gui-menu-scroll-up", 1, 0, 0,
		      (SCM (*)())scm_lw6gui_menu_scroll_up);
  scm_c_define_gsubr ("c-lw6gui-menu-scroll-down", 1, 0, 0,
		      (SCM (*)())scm_lw6gui_menu_scroll_down);
  scm_c_define_gsubr ("c-lw6gui-default-look",
		      0, 0, 0, (SCM (*)())scm_lw6gui_default_look);
  scm_c_define_gsubr ("c-lw6gui-look-set",
		      3, 0, 0, (SCM (*)())scm_lw6gui_look_set);

  /*
   * In liquidwar6gfx
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6gfx-get-backends", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_get_backends);
  scm_c_define_gsubr ("c-lw6gfx-create-backend",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_create_backend);
  scm_c_define_gsubr ("c-lw6gfx-destroy-backend", 0,
		      0, 0, (SCM (*)())scm_lw6gfx_destroy_backend);
  /*
   * In background.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-background",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_display_background);
  /*
   * In event.c
   */
  scm_c_define_gsubr ("c-lw6gfx-poll-key",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_key);
  scm_c_define_gsubr ("c-lw6gfx-is-key-esc", 1, 0, 0,
		      (SCM (*)())scm_lw6gfx_is_key_esc);
  scm_c_define_gsubr ("c-lw6gfx-is-key-enter", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_enter);
  scm_c_define_gsubr ("c-lw6gfx-is-key-up", 1, 0, 0,
		      (SCM (*)())scm_lw6gfx_is_key_up);
  scm_c_define_gsubr ("c-lw6gfx-is-key-down", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_down);
  scm_c_define_gsubr ("c-lw6gfx-is-key-left", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_left);
  scm_c_define_gsubr ("c-lw6gfx-is-key-right", 1, 0,
		      0, (SCM (*)())scm_lw6gfx_is_key_right);
  scm_c_define_gsubr ("c-lw6gfx-poll-quit", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_poll_quit);
  scm_c_define_gsubr ("c-lw6gfx-send-quit", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_send_quit);
  scm_c_define_gsubr ("c-lw6gfx-get-key-state", 1,
		      0, 0, (SCM (*)())scm_lw6gfx_get_key_state);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-move",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_mouse_move);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-button-left",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_mouse_button_left);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-button-right",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_mouse_button_right);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-wheel-up",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_mouse_wheel_up);
  scm_c_define_gsubr ("c-lw6gfx-poll-mouse-wheel-down",
		      0, 0, 0, (SCM (*)())scm_lw6gfx_poll_mouse_wheel_down);
  scm_c_define_gsubr ("c-lw6gfx-get-mouse-state", 0,
		      0, 0, (SCM (*)())scm_lw6gfx_get_mouse_state);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-up",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_poll_joystick_up);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-down",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_poll_joystick_down);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-left",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_poll_joystick_left);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-right",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_poll_joystick_right);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-button-a",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_poll_joystick_button_a);
  scm_c_define_gsubr ("c-lw6gfx-poll-joystick-button-b",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_poll_joystick_button_b);
  scm_c_define_gsubr ("c-lw6gfx-get-joystick-state", 1,
		      0, 0, (SCM (*)())scm_lw6gfx_get_joystick_state);

  /*
   * In hud.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-hud",
		      2, 0, 0, (SCM (*)())scm_lw6gfx_display_hud);

  /*
   * In info.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-log", 2, 0,
		      0, (SCM (*)())scm_lw6gfx_display_log);
  scm_c_define_gsubr ("c-lw6gfx-display-fps", 2, 0,
		      0, (SCM (*)())scm_lw6gfx_display_fps);
  /*
   * In map.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-map-preview",
		      2, 0, 0, (SCM (*)())scm_lw6gfx_display_map_preview);
  scm_c_define_gsubr ("c-lw6gfx-display-map-zones",
		      2, 0, 0, (SCM (*)())scm_lw6gfx_display_map_zones);
  scm_c_define_gsubr
    ("c-lw6gfx-display-map-gradient", 3, 0, 0,
     (SCM (*)())scm_lw6gfx_display_map_gradient);
  scm_c_define_gsubr ("c-lw6gfx-display-game", 2, 0,
		      0, (SCM (*)())scm_lw6gfx_display_game);
  scm_c_define_gsubr ("c-lw6gfx-pick-map-coord", 5, 0,
		      0, (SCM (*)())scm_lw6gfx_pick_map_coord);
  scm_c_define_gsubr ("c-lw6map-options-get-list", 0, 0, 0,
		      (SCM (*)())scm_lw6map_options_get_list);
  scm_c_define_gsubr ("c-lw6map-options-get-int", 2, 0, 0,
		      (SCM (*)())scm_lw6map_options_get_int);

  /*
   * In menu.c
   */
  scm_c_define_gsubr ("c-lw6gfx-display-menu",
		      2, 0, 0, (SCM (*)())scm_lw6gfx_display_menu);
  scm_c_define_gsubr ("c-lw6gfx-pick-menuitem",
		      4, 0, 0, (SCM (*)())scm_lw6gfx_pick_menuitem);
  scm_c_define_gsubr ("c-lw6gfx-warp-mouse-on-menuitem",
		      3, 0, 0, (SCM (*)())scm_lw6gfx_warp_mouse_on_menuitem);

  /* 
   * In render.c
   */
  scm_c_define_gsubr ("c-lw6gfx-prepare-buffer",
		      1, 0, 0, (SCM (*)())scm_lw6gfx_prepare_buffer);
  scm_c_define_gsubr ("c-lw6gfx-swap-buffers", 0, 0,
		      0, (SCM (*)())scm_lw6gfx_swap_buffers);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6gfx-init", 4, 0, 0, (SCM (*)())scm_lw6gfx_init);
  scm_c_define_gsubr ("c-lw6gfx-set-video-mode", 3,
		      0, 0, (SCM (*)())scm_lw6gfx_set_video_mode);
  scm_c_define_gsubr ("c-lw6gfx-quit", 0, 0, 0, (SCM (*)())scm_lw6gfx_quit);
  /*
   * In timer.c
   */
  scm_c_define_gsubr ("c-lw6gfx-delay", 1, 0, 0, (SCM (*)())scm_lw6gfx_delay);
  scm_c_define_gsubr ("c-lw6gfx-get-ticks", 0, 0, 0,
		      (SCM (*)())scm_lw6gfx_get_ticks);
  /*
   * In liquidwar6ldr
   */
  scm_c_define_gsubr ("c-lw6ldr-get-entries", 2, 0, 0,
		      (SCM (*)())scm_lw6ldr_get_entries);
  scm_c_define_gsubr ("c-lw6ldr-read", 5, 0, 0, (SCM (*)())scm_lw6ldr_read);
  scm_c_define_gsubr ("c-lw6ldr-read-relative", 6, 0, 0,
		      (SCM (*)())scm_lw6ldr_read_relative);
  scm_c_define_gsubr ("c-lw6ldr-print-examples", 0, 0, 0,
		      (SCM (*)())scm_lw6ldr_print_examples);
  /*
   * In liquidwar6map
   */
  scm_c_define_gsubr
    ("c-lw6map-get-background-style", 1, 0, 0,
     (SCM (*)())scm_lw6map_get_background_style);
  scm_c_define_gsubr ("c-lw6map-get-hud-style", 1,
		      0, 0, (SCM (*)())scm_lw6map_get_hud_style);
  scm_c_define_gsubr ("c-lw6map-get-menu-style", 1,
		      0, 0, (SCM (*)())scm_lw6map_get_menu_style);
  scm_c_define_gsubr ("c-lw6map-get-view-style", 1,
		      0, 0, (SCM (*)())scm_lw6map_get_view_style);
  /*
   * In liquidwar6ker
   */
  scm_c_define_gsubr ("c-lw6ker-build-game-struct",
		      1, 0, 0, (SCM (*)())scm_lw6ker_build_game_struct);
  scm_c_define_gsubr ("c-lw6ker-build-game-state", 1,
		      0, 0, (SCM (*)())scm_lw6ker_build_game_state);
  scm_c_define_gsubr ("c-lw6ker-copy-game-state", 2,
		      0, 0, (SCM (*)())scm_lw6ker_copy_game_state);
  scm_c_define_gsubr
    ("c-lw6ker-game-struct-checksum", 1, 0, 0,
     (SCM (*)())scm_lw6ker_game_struct_checksum);
  scm_c_define_gsubr ("c-lw6ker-game-state-checksum",
		      1, 0, 0, (SCM (*)())scm_lw6ker_game_state_checksum);
  scm_c_define_gsubr ("c-lw6ker-get-cursors",
		      2, 0, 0, (SCM (*)())scm_lw6ker_get_cursors);
  scm_c_define_gsubr ("c-lw6ker-set-cursors",
		      3, 0, 0, (SCM (*)())scm_lw6ker_set_cursors);
  scm_c_define_gsubr ("c-lw6ker-enable-bot",
		      2, 0, 0, (SCM (*)())scm_lw6ker_enable_bot);
  scm_c_define_gsubr ("c-lw6ker-disable-bot",
		      2, 0, 0, (SCM (*)())scm_lw6ker_disable_bot);
  scm_c_define_gsubr ("c-lw6ker-do-round",
		      1, 0, 0, (SCM (*)())scm_lw6ker_do_round);
  scm_c_define_gsubr ("c-lw6ker-get-moves", 1, 0, 0,
		      (SCM (*)())scm_lw6ker_get_moves);
  scm_c_define_gsubr ("c-lw6ker-get-spreads", 1, 0, 0,
		      (SCM (*)())scm_lw6ker_get_spreads);
  scm_c_define_gsubr ("c-lw6ker-get-rounds", 1, 0, 0,
		      (SCM (*)())scm_lw6ker_get_rounds);
  scm_c_define_gsubr ("c-lw6ker-add-team", 3, 0, 0,
		      (SCM (*)())scm_lw6ker_add_team);
  scm_c_define_gsubr ("c-lw6ker-remove-team", 2, 0,
		      0, (SCM (*)())scm_lw6ker_remove_team);
  /*
   * In liquidwar6snd
   */
  /*
   * In backend.c
   */
  scm_c_define_gsubr ("c-lw6snd-get-backends", 0, 0, 0,
		      (SCM (*)())scm_lw6snd_get_backends);
  scm_c_define_gsubr ("c-lw6snd-create-backend",
		      1, 0, 0, (SCM (*)())scm_lw6snd_create_backend);
  scm_c_define_gsubr ("c-lw6snd-destroy-backend", 0,
		      0, 0, (SCM (*)())scm_lw6snd_destroy_backend);
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6snd-init", 2, 0, 0, (SCM (*)())scm_lw6snd_init);
  scm_c_define_gsubr ("c-lw6snd-quit", 0, 0, 0, (SCM (*)())scm_lw6snd_quit);

  /*
   * In sound.c
   */
  scm_c_define_gsubr ("c-lw6snd-play-sound", 1,
		      0, 0, (SCM (*)())scm_lw6snd_play_sound);
  scm_c_define_gsubr ("c-lw6snd-set-sound-volume", 1,
		      0, 0, (SCM (*)())scm_lw6snd_set_sound_volume);

  /*
   * In music.c
   */
  scm_c_define_gsubr ("c-lw6snd-play-music", 1,
		      0, 0, (SCM (*)())scm_lw6snd_play_music);
  scm_c_define_gsubr ("c-lw6snd-set-music-volume", 1,
		      0, 0, (SCM (*)())scm_lw6snd_set_music_volume);

  /*
   * In liquidwar6con
   */
  /*
   * In handler.c
   */
  scm_c_define_gsubr ("c-lw6cns-init", 0, 0, 0, (SCM (*)())scm_lw6cns_init);
  scm_c_define_gsubr ("c-lw6cns-quit", 0, 0, 0, (SCM (*)())scm_lw6cns_quit);
  scm_c_define_gsubr ("c-lw6cns-poll", 0, 0, 0, (SCM (*)())scm_lw6cns_poll);

  /*
   * In liquidwar6tsk
   */
  scm_c_define_gsubr ("c-lw6tsk-loader-new", 1, 0, 0,
		      (SCM (*)())scm_lw6tsk_loader_new);
  scm_c_define_gsubr ("c-lw6tsk-loader-push", 7, 0, 0,
		      (SCM (*)())scm_lw6tsk_loader_push);
  scm_c_define_gsubr ("c-lw6tsk-loader-pop", 1, 0, 0,
		      (SCM (*)())scm_lw6tsk_loader_pop);

  /*
   * In liquidwar6net
   */
  /*
   * In setup.c
   */
  scm_c_define_gsubr ("c-lw6net-init", 0, 0, 0, (SCM (*)())scm_lw6net_init);
  scm_c_define_gsubr ("c-lw6net-quit", 0, 0, 0, (SCM (*)())scm_lw6net_quit);
  scm_c_define_gsubr ("c-lw6net-server-start", 2, 0, 0,
		      (SCM (*)())scm_lw6net_server_start);
  scm_c_define_gsubr ("c-lw6net-server-poll-idle", 0, 0, 0,
		      (SCM (*)())scm_lw6net_server_poll_idle);
  scm_c_define_gsubr ("c-lw6net-server-poll-playing", 2, 0, 0,
		      (SCM (*)())scm_lw6net_server_poll_playing);
  scm_c_define_gsubr ("c-lw6net-server-stop", 0, 0, 0,
		      (SCM (*)())scm_lw6net_server_stop);

  return ret;
}

extern void
lw6_cns_handler (char *c_line)
{
  SCM line;
  SCM func_symbol;
  SCM func;

  if (c_line)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("interpreting console input \"%s\""),
		  c_line);

      lw6cns_history_add_if_needed (c_line);
      line = scm_makfrom0str (c_line);

      func_symbol = scm_c_lookup ("lw6-console-try-catch");
      func = scm_variable_ref (func_symbol);
      scm_call_1 (func, line);
      free (c_line);		// exceptionnally, don't use LW6SYS_FREE
    }
  else
    {
      printf (_("(quit) or CTRL-C to quit"));
      printf ("\n");
    }
}
