/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6MAP_H
#define LIQUIDWAR6MAP_H

#include "../sys/sys.h"

#define LW6MAP_MAX_FIGHTER_HEALTH 10000
#define LW6MAP_MAX_NB_TEAMS 10
#define LW6MAP_MAX_NB_CURSORS 20
#define LW6MAP_MAX_CURSORS_PER_TEAM 3
#define LW6MAP_MAX_NB_DIR_TRIES 7

/*
 * All static options default/min/max
 */
#define LW6MAP_OPTIONS_DEFAULT_TOTAL_TIME 3600	// 1 hour
#define LW6MAP_OPTIONS_MIN_TOTAL_TIME 30	// 30 seconds
#define LW6MAP_OPTIONS_MAX_TOTAL_TIME 864000	// 10 days

#define LW6MAP_OPTIONS_DEFAULT_RESPAWN_TEAM 1
#define LW6MAP_OPTIONS_MIN_RESPAWN_TEAM 0
#define LW6MAP_OPTIONS_MAX_RESPAWN_TEAM 1

#define LW6MAP_OPTIONS_DEFAULT_MOVES_PER_ROUND 2
#define LW6MAP_OPTIONS_MIN_MOVES_PER_ROUND 1
#define LW6MAP_OPTIONS_MAX_MOVES_PER_ROUND 10

#define LW6MAP_OPTIONS_DEFAULT_SPREADS_PER_ROUND 3
#define LW6MAP_OPTIONS_MIN_SPREADS_PER_ROUND 1
#define LW6MAP_OPTIONS_MAX_SPREADS_PER_ROUND 12

#define LW6MAP_OPTIONS_DEFAULT_ROUNDS_PER_SEC 25
#define LW6MAP_OPTIONS_MIN_ROUNDS_PER_SEC 1
#define LW6MAP_OPTIONS_MAX_ROUNDS_PER_SEC 100

#define LW6MAP_OPTIONS_DEFAULT_FIGHTER_ATTACK 500
#define LW6MAP_OPTIONS_MIN_FIGHTER_ATTACK 1
#define LW6MAP_OPTIONS_MAX_FIGHTER_ATTACK LW6MAP_MAX_FIGHTER_HEALTH

#define LW6MAP_OPTIONS_DEFAULT_FIGHTER_DEFENSE 100
#define LW6MAP_OPTIONS_MIN_FIGHTER_DEFENSE 1
#define LW6MAP_OPTIONS_MAX_FIGHTER_DEFENSE LW6MAP_MAX_FIGHTER_HEALTH

#define LW6MAP_OPTIONS_DEFAULT_FIGHTER_NEW_HEALTH 5000
#define LW6MAP_OPTIONS_MIN_FIGHTER_NEW_HEALTH 1
#define LW6MAP_OPTIONS_MAX_FIGHTER_NEW_HEALTH LW6MAP_MAX_FIGHTER_HEALTH

#define LW6MAP_OPTIONS_DEFAULT_SIDE_ATTACK_FACTOR 20	// 20%
#define LW6MAP_OPTIONS_MIN_SIDE_ATTACK_FACTOR 0
#define LW6MAP_OPTIONS_MAX_SIDE_ATTACK_FACTOR 100

#define LW6MAP_OPTIONS_DEFAULT_SIDE_DEFENSE_FACTOR 20	// 20%
#define LW6MAP_OPTIONS_MIN_SIDE_DEFENSE_FACTOR 0
#define LW6MAP_OPTIONS_MAX_SIDE_DEFENSE_FACTOR 100

#define LW6MAP_OPTIONS_DEFAULT_NB_MOVE_TRIES 5
#define LW6MAP_OPTIONS_MIN_NB_MOVE_TRIES 3
#define LW6MAP_OPTIONS_MAX_NB_MOVE_TRIES LW6MAP_MAX_NB_DIR_TRIES

#define LW6MAP_OPTIONS_DEFAULT_NB_ATTACK_TRIES 3
#define LW6MAP_OPTIONS_MIN_NB_ATTACK_TRIES 1
#define LW6MAP_OPTIONS_MAX_NB_ATTACK_TRIES LW6MAP_MAX_NB_DIR_TRIES

#define LW6MAP_OPTIONS_DEFAULT_NB_DEFENSE_TRIES 1
#define LW6MAP_OPTIONS_MIN_NB_DEFENSE_TRIES 1
#define LW6MAP_OPTIONS_MAX_NB_DEFENSE_TRIES LW6MAP_MAX_NB_DIR_TRIES

#define LW6MAP_OPTIONS_DEFAULT_SINGLE_ARMY_SURFACE 75	// 7,5% (per 1000)
#define LW6MAP_OPTIONS_MIN_SINGLE_ARMY_SURFACE 1
#define LW6MAP_OPTIONS_MAX_SINGLE_ARMY_SURFACE 300	// << 1000

#define LW6MAP_OPTIONS_DEFAULT_TOTAL_ARMIES_SURFACE 300	// 30% (per 1000)
#define LW6MAP_OPTIONS_MIN_TOTAL_ARMIES_SURFACE 5
#define LW6MAP_OPTIONS_MAX_TOTAL_ARMIES_SURFACE 900	// < 1000

#define LW6MAP_OPTIONS_DEFAULT_MAX_NB_TEAMS 6
#define LW6MAP_OPTIONS_MIN_MAX_NB_TEAMS 2
#define LW6MAP_OPTIONS_MAX_MAX_NB_TEAMS LW6MAP_MAX_NB_TEAMS

#define LW6MAP_OPTIONS_DEFAULT_X_POLARITY 0
#define LW6MAP_OPTIONS_MIN_X_POLARITY -1
#define LW6MAP_OPTIONS_MAX_X_POLARITY 1

#define LW6MAP_OPTIONS_DEFAULT_Y_POLARITY 0
#define LW6MAP_OPTIONS_MIN_Y_POLARITY -1
#define LW6MAP_OPTIONS_MAX_Y_POLARITY 1

#define LW6MAP_OPTIONS_DEFAULT_MAX_ZONE_SIZE 8
#define LW6MAP_OPTIONS_MIN_MAX_ZONE_SIZE 1
#define LW6MAP_OPTIONS_MAX_MAX_ZONE_SIZE 64

#define LW6MAP_OPTIONS_DEFAULT_MIN_MAP_WIDTH 20
#define LW6MAP_OPTIONS_MIN_MIN_MAP_WIDTH 10
#define LW6MAP_OPTIONS_MAX_MIN_MAP_WIDTH 10000

#define LW6MAP_OPTIONS_DEFAULT_MAX_MAP_WIDTH 2000
#define LW6MAP_OPTIONS_MIN_MAX_MAP_WIDTH 10
#define LW6MAP_OPTIONS_MAX_MAX_MAP_WIDTH 10000

#define LW6MAP_OPTIONS_DEFAULT_MIN_MAP_HEIGHT 20
#define LW6MAP_OPTIONS_MIN_MIN_MAP_HEIGHT 10
#define LW6MAP_OPTIONS_MAX_MIN_MAP_HEIGHT 10000

#define LW6MAP_OPTIONS_DEFAULT_MAX_MAP_HEIGHT 2000
#define LW6MAP_OPTIONS_MIN_MAX_MAP_HEIGHT 10
#define LW6MAP_OPTIONS_MAX_MAX_MAP_HEIGHT 10000

#define LW6MAP_OPTIONS_DEFAULT_MIN_MAP_SURFACE 4800	// 80x60
#define LW6MAP_OPTIONS_MIN_MIN_MAP_SURFACE 100
#define LW6MAP_OPTIONS_MAX_MIN_MAP_SURFACE 100000000

#define LW6MAP_OPTIONS_DEFAULT_MAX_MAP_SURFACE 480000	// 800x600
#define LW6MAP_OPTIONS_MIN_MAX_MAP_SURFACE 100
#define LW6MAP_OPTIONS_MAX_MAX_MAP_SURFACE 100000000

#define LW6MAP_OPTIONS_DEFAULT_BOT_WAIT_BETWEEN_MOVES 100
#define LW6MAP_OPTIONS_MIN_BOT_WAIT_BETWEEN_MOVES 0
#define LW6MAP_OPTIONS_MAX_BOT_WAIT_BETWEEN_MOVES 10000

#define LW6MAP_OPTIONS_DEFAULT_BOT_NB_MOVE_STEPS 25
#define LW6MAP_OPTIONS_MIN_BOT_NB_MOVE_STEPS 1
#define LW6MAP_OPTIONS_MAX_BOT_NB_MOVE_STEPS 10000

#define LW6MAP_OPTIONS_DEFAULT_ROUND_DELTA 1
#define LW6MAP_OPTIONS_MIN_ROUND_DELTA 0
#define LW6MAP_OPTIONS_MAX_ROUND_DELTA 100

#define LW6MAP_OPTIONS_DEFAULT_MAX_ROUND_DELTA 1000
#define LW6MAP_OPTIONS_MIN_MAX_ROUND_DELTA 1
#define LW6MAP_OPTIONS_MAX_MAX_ROUND_DELTA 10000

#define LW6MAP_OPTIONS_DEFAULT_MAX_CURSOR_POT 1000000
#define LW6MAP_OPTIONS_MIN_MAX_CURSOR_POT 50000
#define LW6MAP_OPTIONS_MAX_MAX_CURSOR_POT 5000000

#define LW6MAP_OPTIONS_DEFAULT_CURSOR_POT_INIT 100000
#define LW6MAP_OPTIONS_MIN_CURSOR_POT_INIT 5000
#define LW6MAP_OPTIONS_MAX_CURSOR_POT_INIT 500000

#define LW6MAP_OPTIONS_DEFAULT_MAX_CURSOR_POT_OFFSET 100
#define LW6MAP_OPTIONS_MIN_MAX_CURSOR_POT_OFFSET 1
#define LW6MAP_OPTIONS_MAX_MAX_CURSOR_POT_OFFSET 10000

/*
 * Note: it's important that floats are never ever used in map options.
 * In fact, we need maps to be 100,00 (lots of zeroes) predictable,
 * given some identical user actions. Using floats could lead us
 * to very slight differences (floats are never accurate, especially
 * when you run calculus on different hardwares) which could, with
 * time, become very important. Sort of a "butterfly effect".
 * So well, we use int. Int32 to be exact.
 */
typedef struct lw6map_options_s
{
  int32_t total_time;
  int32_t respawn_team;
  int32_t moves_per_round;
  int32_t spreads_per_round;
  int32_t rounds_per_sec;
  int32_t fighter_attack;
  int32_t fighter_defense;
  int32_t fighter_new_health;
  int32_t side_attack_factor;
  int32_t side_defense_factor;
  int32_t nb_move_tries;
  int32_t nb_attack_tries;
  int32_t nb_defense_tries;
  int32_t single_army_surface;
  int32_t total_armies_surface;
  int32_t max_nb_teams;
  int32_t x_polarity;
  int32_t y_polarity;
  int32_t max_zone_size;
  int32_t min_map_width;
  int32_t max_map_width;
  int32_t min_map_height;
  int32_t max_map_height;
  int32_t min_map_surface;
  int32_t max_map_surface;
  int32_t bot_wait_between_moves;
  int32_t bot_nb_move_steps;
  int32_t round_delta;
  int32_t max_round_delta;
  int32_t max_cursor_pot;
  int32_t cursor_pot_init;
  int32_t max_cursor_pot_offset;
}
lw6map_options_t;

extern char *LW6MAP_OPTIONS_LIST[];

#define LW6MAP_MAX_DEPTH 4	// can't be too high for perf issues

#define LW6MAP_STYLE_DEFAULT_KEEP_RATIO 1
#define LW6MAP_STYLE_DEFAULT_ZOOM 1.0f
#define LW6MAP_STYLE_DEFAULT_FIGHTER_SCALE 4.0f
#define LW6MAP_STYLE_DEFAULT_USE_TEXTURE 1
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_STYLE "bubbles"
#define LW6MAP_STYLE_DEFAULT_HUD_STYLE "floating"
#define LW6MAP_STYLE_DEFAULT_MENU_STYLE "cylinder"
#define LW6MAP_STYLE_DEFAULT_VIEW_STYLE "flat"
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_DENSITY 1.0f
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_SPEED 1.0f
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_COLOR_AUTO 1
#define LW6MAP_STYLE_DEFAULT_HUD_COLOR_AUTO 1
#define LW6MAP_STYLE_DEFAULT_MENU_COLOR_AUTO 1
#define LW6MAP_STYLE_DEFAULT_VIEW_COLOR_AUTO 1
#define LW6MAP_STYLE_DEFAULT_SYSTEM_COLOR_AUTO 1
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_COLOR_BASE_BG "#000"
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_COLOR_BASE_FG "#ccc"
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_COLOR_ALTERNATE_BG "#333"
#define LW6MAP_STYLE_DEFAULT_BACKGROUND_COLOR_ALTERNATE_FG "#fff"
#define LW6MAP_STYLE_DEFAULT_HUD_COLOR_FRAME_BG "#000"
#define LW6MAP_STYLE_DEFAULT_HUD_COLOR_FRAME_FG "#ccc"
#define LW6MAP_STYLE_DEFAULT_HUD_COLOR_TEXT_BG "#333"
#define LW6MAP_STYLE_DEFAULT_HUD_COLOR_TEXT_FG "#fff"
#define LW6MAP_STYLE_DEFAULT_MENU_COLOR_DEFAULT_BG "#333"
#define LW6MAP_STYLE_DEFAULT_MENU_COLOR_DEFAULT_FG "#fff"
#define LW6MAP_STYLE_DEFAULT_MENU_COLOR_SELECTED_BG "#fff"
#define LW6MAP_STYLE_DEFAULT_MENU_COLOR_SELECTED_FG "#333"
#define LW6MAP_STYLE_DEFAULT_MENU_COLOR_DISABLED_BG "#000"
#define LW6MAP_STYLE_DEFAULT_MENU_COLOR_DISABLED_FG "#ccc"
#define LW6MAP_STYLE_DEFAULT_VIEW_COLOR_CURSOR_BG "#333"
#define LW6MAP_STYLE_DEFAULT_VIEW_COLOR_CURSOR_FG "#fff"
#define LW6MAP_STYLE_DEFAULT_VIEW_COLOR_MAP_BG "#000"
#define LW6MAP_STYLE_DEFAULT_VIEW_COLOR_MAP_FG "#ccc"
#define LW6MAP_STYLE_DEFAULT_SYSTEM_COLOR_BG "#333"
#define LW6MAP_STYLE_DEFAULT_SYSTEM_COLOR_FG "#fff"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_RED "#f00"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_GREEN "#0f0"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_BLUE "#00f"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_YELLOW "#ff0"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_CYAN "#0ff"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_MAGENTA "#f0f"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_ORANGE "#f80"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_LIGHTBLUE "#8bf"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_PURPLE "#b8f"
#define LW6MAP_STYLE_DEFAULT_TEAM_COLOR_PINK "#f8b"

#define LW6MAP_NB_TEAM_COLORS LW6MAP_MAX_NB_TEAMS
typedef enum LW6MAP_TEAM_COLOR_ENUM
{
  LW6MAP_TEAM_COLOR_RED = 0,
  LW6MAP_TEAM_COLOR_GREEN = 1,
  LW6MAP_TEAM_COLOR_BLUE = 2,
  LW6MAP_TEAM_COLOR_YELLOW = 3,
  LW6MAP_TEAM_COLOR_CYAN = 4,
  LW6MAP_TEAM_COLOR_MAGENTA = 5,
  LW6MAP_TEAM_COLOR_ORANGE = 6,
  LW6MAP_TEAM_COLOR_LIGHTBLUE = 7,
  LW6MAP_TEAM_COLOR_PURPLE = 8,
  LW6MAP_TEAM_COLOR_PINK = 9
}
LW6MAP_TEAM_COLOR;

typedef struct lw6map_metadata_s
{
  char *title;
  char *readme;
}
lw6map_metadata_t;

typedef struct lw6map_depth_s
{
  lw6sys_wh_t shape;
  /*
   * Depth is used for game logic. 8 fighters can fit on a slot/point
   * when depth is 8. The default depth (as of Liquid War 5) is 1, and
   * only one fighter can fit. 0 mean it's a wall. This 
   */
  unsigned char *data;
  int max;
}
lw6map_depth_t;

typedef struct lw6map_color_couple_s
{
  lw6sys_color_8_t fg;
  lw6sys_color_8_t bg;
}
lw6map_color_couple_t;

typedef struct lw6map_texture_s
{
  lw6sys_wh_t shape;
  lw6sys_color_8_t *data;
  lw6map_color_couple_t guessed_color_base;
  lw6map_color_couple_t guessed_color_alternate;
}
lw6map_texture_t;

typedef struct lw6map_style_s
{
  int keep_ratio;		// boolean
  float zoom;
  float fighter_scale;
  int use_texture;
  char *background_style;
  char *hud_style;
  char *menu_style;
  char *view_style;
  float background_density;
  float background_speed;
  int background_color_auto;
  int hud_color_auto;
  int menu_color_auto;
  int view_color_auto;
  int system_color_auto;
  lw6map_color_couple_t background_color_base;
  lw6map_color_couple_t background_color_alternate;
  lw6map_color_couple_t hud_color_frame;
  lw6map_color_couple_t hud_color_text;
  lw6map_color_couple_t menu_color_default;
  lw6map_color_couple_t menu_color_selected;
  lw6map_color_couple_t menu_color_disabled;
  lw6map_color_couple_t view_color_cursor;
  lw6map_color_couple_t view_color_map;
  lw6map_color_couple_t system_color;
  lw6sys_color_8_t team_colors[LW6MAP_NB_TEAM_COLORS];
}
lw6map_style_t;

typedef struct lw6map_param_s
{
  lw6map_options_t options;
  lw6map_style_t style;
}
lw6map_param_t;

typedef struct lw6map_level_s
{
  int id;
  lw6map_metadata_t metadata;
  lw6map_depth_t depth;
  lw6map_texture_t texture;
  lw6map_param_t param;
}
lw6map_level_t;

typedef struct lw6map_resampler_s
{
  lw6sys_wh_t target_shape;
  lw6sys_wh_t source_shape;
  float scale_x;
  float scale_y;
}
lw6map_resampler_t;

/*
 * Simple setters/getters. Not so usefull since the purpose of 
 * having plain pointers on a raw buffer is exactly to bypass 
 * these kind of cumbersome functions 8-). In fact having the 
 * setter is a good way to 1) test and 2) find out quickly how 
 * data is stored.
 */
static inline void
lw6map_depth_set (lw6map_depth_t * depth, int x, int y, unsigned char value)
{
  depth->data[y * depth->shape.w + x] = value;
}
static inline unsigned char
lw6map_depth_get (lw6map_depth_t * depth, int x, int y)
{
  return depth->data[y * depth->shape.w + x];
}
static inline void
lw6map_texture_set (lw6map_texture_t * texture, int x, int y,
		    lw6sys_color_8_t value)
{
  texture->data[y * texture->shape.w + x] = value;
}
static inline lw6sys_color_8_t
lw6map_texture_get (lw6map_texture_t * texture, int x, int y)
{
  return texture->data[y * texture->shape.w + x];
}

/*
 * In color.c
 */
extern void lw6map_color_invert (lw6map_color_couple_t * color);
extern void lw6map_auto_colors (lw6map_style_t * style);

/*
 * In depth.c
 */
extern void lw6map_depth_defaults (lw6map_depth_t * depth);
extern void lw6map_depth_clear (lw6map_depth_t * depth);
extern void lw6map_depth_fill (lw6map_depth_t * depth, unsigned char value);
extern int lw6map_depth_check_and_fix_holes (lw6map_depth_t * depth);
extern void lw6map_depth_block_up_down (lw6map_depth_t * depth);
extern void lw6map_depth_block_left_right (lw6map_depth_t * depth);
extern int lw6map_depth_coord_from_texture (lw6map_level_t * level,
					    int *depth_x, int *depth_y,
					    int texture_x, int texture_y);
extern unsigned char lw6map_depth_get_with_texture_coord (lw6map_level_t *
							  level,
							  int texture_x,
							  int texture_y);

/* map-hexa.c */
extern char *lw6map_to_hexa (lw6map_level_t * level);
extern lw6map_level_t *lw6map_from_hexa (char *hexa);

/* map-map.c */
extern lw6map_level_t *lw6map_new ();
extern lw6map_level_t *lw6map_defaults ();
extern void lw6map_free (lw6map_level_t * level);
extern int lw6map_memory_footprint (lw6map_level_t * level);
extern char *lw6map_repr (lw6map_level_t * level);

/*
 * In metadata.c
 */
extern void lw6map_metadata_defaults (lw6map_metadata_t * metadata);
extern void lw6map_metadata_clear (lw6map_metadata_t * metadata);

/*
 * In options.c
 */
extern void lw6map_options_defaults (lw6map_options_t * static_options);
extern void lw6map_options_copy (lw6map_options_t * dst,
				 lw6map_options_t * src);
extern void lw6map_options_update_checksum (lw6map_options_t * static_options,
					    u_int32_t * checksum);
extern int32_t lw6map_options_get_default (char *key);
extern int32_t lw6map_options_get_min (char *key);
extern int32_t lw6map_options_get_max (char *key);
extern int32_t lw6map_options_get_int (lw6map_options_t * static_options,
				       char *key);
extern int lw6map_options_set_int (lw6map_options_t * static_options,
				   char *key, int32_t value);
extern int lw6map_options_get_bool (lw6map_options_t * static_options,
				    char *key);
extern int lw6map_options_set_bool (lw6map_options_t * static_options,
				    char *key, int value);
extern void lw6map_options_clear (lw6map_options_t * param);

/*
 * In param.c
 */
extern void lw6map_param_defaults (lw6map_param_t * param);
extern void lw6map_param_clear (lw6map_param_t * param);
extern void lw6map_param_copy (lw6map_param_t * dst, lw6map_param_t * src);
extern void lw6map_param_set_int (lw6map_param_t * param, char *key,
				  int value);
extern void lw6map_param_set_bool (lw6map_param_t * param, char *key,
				   int value);
extern void lw6map_param_set_float (lw6map_param_t * param, char *key,
				    float value);
extern void lw6map_param_set_string (lw6map_param_t * param, char *key,
				     char *value);

/*
 * In resampler.c
 */
extern void lw6map_resampler_init (lw6map_resampler_t * resampler,
				   lw6map_options_t * static_options,
				   lw6sys_wh_t * source_shape,
				   lw6sys_wh_t * display_shape,
				   float target_ratio, float fighter_scale);
extern void lw6map_resampler_source2target (lw6map_resampler_t * resampler,
					    int *target_x, int *target_y,
					    int source_x, int source_y);
extern void lw6map_resampler_target2source (lw6map_resampler_t * resampler,
					    int *source_x, int *source_y,
					    int target_x, int target_y);

/*
 * In style.c
 */
extern void lw6map_style_defaults (lw6map_style_t * style);
extern void lw6map_style_clear (lw6map_style_t * style);
extern void lw6map_style_copy (lw6map_style_t * dst, lw6map_style_t * src);
extern int lw6map_style_set (lw6map_style_t * style, char *key, char *value);

/*
 * In test.c
 */
extern int lw6map_test ();

/*
 * In texture.c
 */
extern void lw6map_texture_defaults (lw6map_texture_t * texture);
extern void lw6map_texture_clear (lw6map_texture_t * texture);
extern int lw6map_texture_coord_from_depth (lw6map_level_t * level,
					    int *texture_x, int *texture_y,
					    int depth_x, int depth_y);
extern lw6sys_color_8_t lw6map_texture_get_with_depth_coord (lw6map_level_t *
							     level,
							     int depth_x,
							     int depth_y);

#endif
