/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>

#include "config.h"
#include "sys.h"
#include "sys-build.h"

#define COPYRIGHT "Copyright (C)  2008  Christian Mauduit <ufoot@ufoot.org>"
#define LICENSE "GPLv3+ (GNU GPL version 3 or later)"
#define TOP_SRCDIR_TEST_FILE "liquidwar6.spec.in"
#define TOP_SRCDIR_MAX_UNPARENT 4

/**
 * lw6sys_build_get_package_tarname:
 *
 * Returns the name of the package. This is the @PACKAGE_TARNAME constant
 * defined by the GNU Autoconf ./configure script. While it's always possible
 * to use the defined constant directly, using this function will return
 * the value defined when compiling the binary, not the one you're using
 * when compiling another program relying on Liquid War as a library.
 *
 * Return value: a non-NULL string "liquidwar6", must not be freed.
 */
char *
lw6sys_build_get_package_tarname ()
{
  return PACKAGE_TARNAME;
}

/**
 * lw6sys_build_get_package_name:
 *
 * Returns the name of the package, in a user friendly form, which can
 * include spaces, for instance. This is the @PACKAGE_NAME constant
 * defined by the GNU Autoconf ./configure script. While it's always possible
 * to use the defined constant directly, using this function will return
 * the value defined when compiling the binary, not the one you're using
 * when compiling another program relying on Liquid War as a library.
 *
 * Return value: a non-NULL string "Liquid War 6", must not be freed.
 */
char *
lw6sys_build_get_package_name ()
{
  return PACKAGE_NAME;
}

/**
 * lw6sys_build_get_package_string:
 *
 * Returns the description of the package. This is the @PACKAGE_STRING constant
 * defined by the GNU Autoconf ./configure script. It's the concatenation
 * of @PACKAGE_NAME and @VERSION. While it's always possible
 * to use the defined constant directly, using this function will return
 * the value defined when compiling the binary, not the one you're using
 * when compiling another program relying on Liquid War as a library.
 *
 * Return value: a non-NULL string "Liquid War 6 <version>", must not be freed.
 */
char *
lw6sys_build_get_package_string ()
{
  return PACKAGE_STRING;
}

/**
 * lw6sys_build_get_version:
 *
 * Returns the version of the program. This is the @VERSION constant
 * defined by the GNU Autoconf ./configure script. Same as @PACKAGE_VERSION.
 * Note that while using a function to get @PACKAGE_TARNAME might seem
 * useless, having both ways to get the version, that is, a function and
 * a constant, is very usefull. Think, for instance, that a dynamically
 * loaded shared library might need to check its own version against the
 * version of the core program.
 *
 * Return value: a non-NULL string, which must not be freed.
 */
char *
lw6sys_build_get_version ()
{
  return VERSION;
}

/**
 * lw6sys_build_get_codename:
 *
 * Returns the the program codename. This is the little name of the
 * version. It's been decided that all LW6 releases would take the
 * name of a famous general, warrior, whatever. For instance, it could
 * be "Napoleon".
 *
 * Return value: a non-NULL string, traditionnally the name of a famous
 *   general, someone which has been involved in war. Must not be freed
 *   (I mean, the string, not the general).
 */
char *
lw6sys_build_get_codename ()
{
  return LW6_CODENAME;
}

/**
 * lw6sys_build_get_stamp:
 *
 * Returns the program stamp. This is like a serial number. It's is not
 * the same as the version. The version is meant to be set to something
 * readable. This is just a cryptic thing, incremented at each ./configure
 * or each developper's "I feel like it needs to be incremented". The
 * idea is just to keep (one more...) track of which source code is build.
 * Ideally, this would be plugged to the source revision control system
 * but this has some drawbacks, including that it would require it to
 * modify files before commiting them, which is not safe, and almost
 * impossible if you sign archives. One more point: this is a string. It's
 * true the return value is actually a string containing the representation
 * of an integer, but because all other build parameters are strings, and
 * because we don't know what the future reserves, it's a string.
 *
 * Return value: a non-NULL string like "42", which must not be freed.
 */
char *
lw6sys_build_get_stamp ()
{
  return LW6_STAMP;
}

/**
 * lw6sys_build_get_md5sum:
 *
 * Returns an md5 checkum which is caculated from C (.c and .h) source
 * files. This is complementary with the build stamp. By default the stamp
 * will be enough to check what has been compiled, but one can always
 * imagine a case where Bob compiles something a little different than Alice,
 * with the same stamp, incremented by 1 from a common source tree. They
 * apply their own patches, for instance. This md5sum double-checks that
 * two binaries have been built from the same sources. Note that this
 * is not the md5 checksum of the generated binary. Nor does it include
 * any information about scheme scripts and data.
 *
 * Return value: a non-NULL string, which must not be freed.
 */
char *
lw6sys_build_get_md5sum ()
{
  return LW6_MD5SUM;
}

/**
 * lw6sys_build_get_copyright:
 *
 * Returns a (very) short copyright information about the program. 
 *
 * Return value: a non-NULL string, single line whithout '\n' at the end. 
 *   Must not be freed.
 */
char *
lw6sys_build_get_copyright ()
{
  return COPYRIGHT;
}

/**
 * lw6sys_build_get_license:
 *
 * Returns the license for the program (GNU GPL v3 or later).
 *
 * Return value: a non-NULL string, single line whithout '\n' at the end. 
 *   Must not be freed.
 */
char *
lw6sys_build_get_license ()
{
  return LICENSE;
}

/**
 * lw6sys_build_get_configure_args:
 *
 * Returns the arguments passed to the GNU Autoconf ./configure script when
 * buildling the game. Very usefull to know how the binary was generated,
 * that is, what kind of optimizations are peculiar settings it uses.
 *
 * Return value: a non-NULL string, which, passed to ./configure again, would
 *   hopefully generate the same binary. Must not be freed.
 */
char *
lw6sys_build_get_configure_args ()
{
  return LW6_CONFIGURE_ARGS;
}

/**
 * lw6sys_build_get_cflags:
 *
 * Returns the arguments which would allow another program to
 * use liquidwar6 as a library. Typically, pass this to gcc when
 * compiling your sources. Basically contains "-I" switches
 * which tell where the headers are.
 *
 * Return value: a non-NULL string, which must not be freed.
 */
char *
lw6sys_build_get_cflags ()
{
  return "-I" LW6_INCLUDEDIR " -I" LW6_INCLUDEDIR "/" PACKAGE_TARNAME;
}

/**
 * lw6sys_build_get_ldflags:
 *
 * Returns the arguments which would allow another program to
 * link against liquidwar6. Pass this to gcc or libtool when
 * compiling your program. Basically contains a "-L" option
 * which says where the library is. Note that this will only
 * allow you to link against the main libliquidwar6 library,
 * but not the dynamically loaded modules.
 *
 * Return value: a non-NULL string, which must not be freed.
 */
char *
lw6sys_build_get_ldflags ()
{
  return "-L" LW6_LIBDIR " -l" PACKAGE_TARNAME;
}

/**
 * lw6sys_build_get_hostname:
 *
 * Returns the value return by the standard shell @hostname command
 * on the machine where the game has been built. Usefull to track binaries
 * and know where do they come from.
 *
 * Return value: a non-NULL string, must not be freed.
 */
char *
lw6sys_build_get_hostname ()
{
  return LW6_HOSTNAME;
}

/**
 * lw6sys_build_get_date:
 *
 * Returns the compilation date. While this information can easily be
 * obtained with the C @__DATE__ macro, having this function is convenient
 * for it returns a value which is the same for the whole program, and does
 * not possibly change in every file.
 *
 * Return value: a non-NULL string, must not be freed.
 */
char *
lw6sys_build_get_date ()
{
  return __DATE__;
}

/**
 * lw6sys_build_get_date:
 *
 * Returns the compilation date. While this information can easily be
 * obtained with the C @__TIME__ macro, having this function is convenient
 * for it returns a value which is the same for the whole program, and does
 * not possibly change in every file.
 *
 * Return value: a non-NULL string, must not be freed.
 */
char *
lw6sys_build_get_time ()
{
  return __TIME__;
}

/**
 * lw6sys_build_get_target_cpu:
 *
 * Returns the CPU this program is designed for. Convenient on i386 compatible
 * CPUs to know which flavor (i386, i586...) the binary is made for.
 *
 * Return value: a non-NULL string, must not be freed.
 */
char *
lw6sys_build_get_target_cpu ()
{
  return LW6_TARGET_CPU;
}

/**
 * lw6sys_build_get_target_os:
 *
 * Returns the OS this program is designed for. Usefull for bug reports.
 *
 * Return value: a non-NULL string, must not be freed.
 */
char *
lw6sys_build_get_target_os ()
{
  return LW6_TARGET_OS;
}

/**
 * lw6sys_build_get_top_srcdir:
 *
 * Returns the top source directory, when the game was built. This can
 * seem useless and non relevant on the end-user's machine, but... it's
 * a must-have for developpers and packagers. Without this, binaries
 * would never find their associated data, especially when building
 * outside the source tree. Or, testing the game would be impossible
 * without installing it, given the fact that most of the code is in
 * scripts that are stored in /usr/local by default, this would be
 * painfull. So this function is here to help finding data within the
 * source tree when the game is not installed yet. Note that the function
 * is rather clever, since it will automatically try to remove useless
 * '../' sequences at the beginning of a possibly relative path.
 *
 * Return value: a non-NULL string, must not be freed.
 */
char *
lw6sys_build_get_top_srcdir ()
{
  char *top_srcdir = NULL;
  char *test_dir = NULL;
  char *test_file = NULL;
  int i = 0;

  test_dir = LW6_TOP_SRCDIR;
  while (i < TOP_SRCDIR_MAX_UNPARENT && !top_srcdir)
    {
      test_file = lw6sys_path_concat (test_dir, TOP_SRCDIR_TEST_FILE);
      if (test_file)
	{
	  if (lw6sys_file_exists (test_file))
	    {
	      top_srcdir = test_dir;
	    }
	  LW6SYS_FREE (test_file);
	}
      test_dir = lw6sys_path_unparent_no_malloc (test_dir);
      ++i;
    }

  if (!top_srcdir)
    {
      top_srcdir = LW6_TOP_SRCDIR;
    }

  return top_srcdir;
}

/**
 * lw6sys_build_get_prefix:
 *
 * Returns the @prefix value as given to the GNU Autoconf ./configure script.
 * Used to deduce the path to other directories and files.
 *
 * Return value: a non-NULL string, "/usr/local" by default. 
 *   Must not be freed.
 */
char *
lw6sys_build_get_prefix ()
{
  return LW6_PREFIX;
}

/**
 * lw6sys_build_get_datadir:
 *
 * Returns the @datadir value defined by the GNU Autoconf ./configure script.
 * This is not the value which can be overriden by the Liquid War 6 specific.
 * "--data-dir" option. @datadir is usually something like "/usr/local/share"
 * while the actual Liquid War 6 defined data dir is a more profound path
 * which includes the name of the package, its version, and so on.
 *
 * Return value: a non-NULL string, "/usr/local/share" by default.
 *   Must not be freed.
 */
char *
lw6sys_build_get_datadir ()
{
  return LW6_DATADIR;
}

/**
 * lw6sys_build_get_libdir:
 *
 * Returns the @libdir value defined by the GNU Autoconf ./configure script.
 * This is not the value which can be overriden by the Liquid War 6 specific.
 * "--mod-dir" option. @datadir is usually something like "/usr/local/lib"
 * while the actual Liquid War 6 defined module dir is a more profound path
 * which includes the name of the package, its version, and so on.
 *
 * Return value: a non-NULL string, "/usr/local/lib" by default.
 *   Must not be freed.
 */
char *
lw6sys_build_get_libdir ()
{
  return LW6_LIBDIR;
}

/**
 * lw6sys_build_get_includedir:
 *
 * Returns the @includedir value defined by the GNU Autoconf ./configure script.
 * As for other options, it's interesting to have this value, this enables
 * the program to inform people who want to hack the game of the place
 * headers are supposed to be installed.
 *
 * Return value: a non-NULL string, "/usr/local/include" by default.
 *   Must not be freed.
 */
char *
lw6sys_build_get_includedir ()
{
  return LW6_INCLUDEDIR;
}

/**
 * lw6sys_build_get_localedir
 *
 * Returns the @localedir value defined by the GNU Autoconf
 * ./configure script.
 * Used as an argument for gettext / libintl functions.
 *
 * Return value: a non-NULL string, "/usr/local/share/locale" by default.
 *   Must not be freed.
 */
char *
lw6sys_build_get_localedir ()
{
  return LW6_LOCALEDIR;
}

/**
 * lw6sys_build_get_docdir
 *
 * Returns the @docdir value defined by the GNU Autoconf
 * ./configure script.
 * Used to write consistent XML file headers.
 *
 * Return value: a non-NULL string, "/usr/local/share/doc/liquidwar6" by default.
 *   Must not be freed.
 */
char *
lw6sys_build_get_docdir ()
{
  return LW6_DOCDIR;
}

/**
 * lw6sys_build_log_pedigree
 *
 * Dumps in the log file the whole program pedigree, that is, what are
 * the values of all the build options. Usefull for bug reports.
 *
 * Return value: none.
 */
void
lw6sys_build_log_pedigree ()
{
  lw6sys_log (LW6SYS_LOG_INFO, _("build package tarname is \"%s\""),
	      lw6sys_build_get_package_tarname ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build package name is \"%s\""),
	      lw6sys_build_get_package_name ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build package string is \"%s\""),
	      lw6sys_build_get_package_string ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build version is \"%s\""),
	      lw6sys_build_get_version ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build codename is \"%s\""),
	      lw6sys_build_get_codename ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build stamp is \"%s\""),
	      lw6sys_build_get_stamp ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build md5sum is \"%s\""),
	      lw6sys_build_get_md5sum ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build copyright is \"%s\""),
	      lw6sys_build_get_copyright ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build license is \"%s\""),
	      lw6sys_build_get_license ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build configure args is \"%s\""),
	      lw6sys_build_get_configure_args ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build md5sum is \"%s\""),
	      lw6sys_build_get_md5sum ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build cflags is \"%s\""),
	      lw6sys_build_get_cflags ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build ldflags is \"%s\""),
	      lw6sys_build_get_ldflags ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build date is \"%s\""),
	      lw6sys_build_get_date ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build time is \"%s\""),
	      lw6sys_build_get_time ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build target cpu is \"%s\""),
	      lw6sys_build_get_target_cpu ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build target os is \"%s\""),
	      lw6sys_build_get_target_os ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build top_srcdir is \"%s\""),
	      lw6sys_build_get_top_srcdir ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build prefix is \"%s\""),
	      lw6sys_build_get_prefix ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build datadir is \"%s\""),
	      lw6sys_build_get_datadir ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build libdir is \"%s\""),
	      lw6sys_build_get_libdir ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build includedir is \"%s\""),
	      lw6sys_build_get_includedir ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build localedir is \"%s\""),
	      lw6sys_build_get_localedir ());
  lw6sys_log (LW6SYS_LOG_INFO, _("build docdir is \"%s\""),
	      lw6sys_build_get_docdir ());
}
