/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdarg.h>
#include <syslog.h>
#include <time.h>
#include <string.h>
#include <errno.h>

#include "config.h"
#include "sys.h"
#include "sys-internal.h"

#define STATIC_LOG_FILENAME_SIZE 65536
static char static_log_filename[STATIC_LOG_FILENAME_SIZE + 1] = { 0 };

static char *
get_log_file ()
{
  char *log_file;

  if (static_log_filename[0] != 0)
    {
      log_file = lw6sys_str_copy (static_log_filename);
    }
  else
    {
      log_file = lw6sys_get_default_log_file ();
    }

  if (!log_file)
    {
      lw6sys_log_sos (_("log_file is NULL"));
    }

  return log_file;
}

/*
 * Convenience fonction which returns the "macro" corresponding
 * to an errno code. I find it easier to use this than bothering
 * declaring a buffer for strerror_r... Besides LW6b has its own
 * error messages.
 */
static char *
errno_str (int errno_int)
{
  char *ret = "?";

  switch (errno_int)
    {
#ifdef E2BIG
    case E2BIG:
      ret = "E2BIG";
      break;
#endif
#ifdef EACCES
    case EACCES:
      ret = "EACCES";
      break;
#endif
#ifdef EADDRINUSE
    case EADDRINUSE:
      ret = "EADDRINUSE";
      break;
#endif
#ifdef EADDRNOTAVAIL
    case EADDRNOTAVAIL:
      ret = "EADDRNOTAVAIL";
      break;
#endif
#ifdef EAFNOSUPPORT
    case EAFNOSUPPORT:
      ret = "EAFNOSUPPORT";
      break;
#endif
#ifdef EAGAIN
    case EAGAIN:		// == EWOULDBLOCK
      ret = "EAGAIN";
      break;
#endif
#ifdef EALREADY
    case EALREADY:
      ret = "EALREADY";
      break;
#endif
#ifdef EBADF
    case EBADF:
      ret = "EBADF";
      break;
#endif
#ifdef EBADMSG
    case EBADMSG:
      ret = "EBADMSG";
      break;
#endif
#ifdef EBUSY
    case EBUSY:
      ret = "EBUSY";
      break;
#endif
#ifdef ECANCELED
    case ECANCELED:
      ret = "ECANCELED";
      break;
#endif
#ifdef ECHILD
    case ECHILD:
      ret = "ECHILD";
      break;
#endif
#ifdef ECONNABORTED
    case ECONNABORTED:
      ret = "ECONNABORTED";
      break;
#endif
#ifdef ECONNREFUSED
    case ECONNREFUSED:
      ret = "ECONNREFUSED";
      break;
#endif
#ifdef ECONNRESET
    case ECONNRESET:
      ret = "ECONNRESET";
      break;
#endif
#ifdef EDEADLK
    case EDEADLK:
      ret = "EDEADLK";
      break;
#endif
#ifdef EDESTADDRREQ
    case EDESTADDRREQ:
      ret = "EDESTADDRREQ";
      break;
#endif
#ifdef EDOM
    case EDOM:
      ret = "EDOM";
      break;
#endif
#ifdef EDQUOT
    case EDQUOT:
      ret = "EDQUOT";
      break;
#endif
#ifdef EEXIST
    case EEXIST:
      ret = "EEXIST";
      break;
#endif
#ifdef EFAULT
    case EFAULT:
      ret = "EFAULT";
      break;
#endif
#ifdef EFBIG
    case EFBIG:
      ret = "EFBIG";
      break;
#endif
#ifdef EHOSTUNREACH
    case EHOSTUNREACH:
      ret = "EHOSTUNREACH";
      break;
#endif
#ifdef EIDRM
    case EIDRM:
      ret = "EIDRM";
      break;
#endif
#ifdef EILSEQ
    case EILSEQ:
      ret = "EILSEQ";
      break;
#endif
#ifdef EINPROGRESS
    case EINPROGRESS:
      ret = "EINPROGRESS";
      break;
#endif
#ifdef EINTR
    case EINTR:
      ret = "EINTR";
      break;
#endif
#ifdef EINVAL
    case EINVAL:
      ret = "EINVAL";
      break;
#endif
#ifdef EIO
    case EIO:
      ret = "EIO";
      break;
#endif
#ifdef EISCONN
    case EISCONN:
      ret = "EISCONN";
      break;
#endif
#ifdef EISDIR
    case EISDIR:
      ret = "EISDIR";
      break;
#endif
#ifdef ELOOP
    case ELOOP:
      ret = "ELOOP";
      break;
#endif
#ifdef EMFILE
    case EMFILE:
      ret = "EMFILE";
      break;
#endif
#ifdef EMLINK
    case EMLINK:
      ret = "EMLINK";
      break;
#endif
#ifdef EMSGSIZE
    case EMSGSIZE:
      ret = "EMSGSIZE";
      break;
#endif
#ifdef EMULTIHOP
    case EMULTIHOP:
      ret = "EMULTIHOP";
      break;
#endif
#ifdef ENAMETOOLONG
    case ENAMETOOLONG:
      ret = "ENAMETOOLONG";
      break;
#endif
#ifdef ENETDOWN
    case ENETDOWN:
      ret = "ENETDOWN";
      break;
#endif
#ifdef ENETRESET
    case ENETRESET:
      ret = "ENETRESET";
      break;
#endif
#ifdef ENETUNREACH
    case ENETUNREACH:
      ret = "ENETUNREACH";
      break;
#endif
#ifdef ENFILE
    case ENFILE:
      ret = "ENFILE";
      break;
#endif
#ifdef ENOBUFS
    case ENOBUFS:
      ret = "ENOBUFS";
      break;
#endif
#ifdef ENODATA
    case ENODATA:
      ret = "ENODATA";
      break;
#endif
#ifdef ENODEV
    case ENODEV:
      ret = "ENODEV";
      break;
#endif
#ifdef ENOENT
    case ENOENT:
      ret = "ENOENT";
      break;
#endif
#ifdef ENOEXEC
    case ENOEXEC:
      ret = "ENOEXEC";
      break;
#endif
#ifdef ENOLCK
    case ENOLCK:
      ret = "ENOLCK";
      break;
#endif
#ifdef ENOLINK
    case ENOLINK:
      ret = "ENOLINK";
      break;
#endif
#ifdef ENOMEM
    case ENOMEM:
      ret = "ENOMEM";
      break;
#endif
#ifdef ENOMSG
    case ENOMSG:
      ret = "ENOMSG";
      break;
#endif
#ifdef ENOPROTOOPT
    case ENOPROTOOPT:
      ret = "ENOPROTOOPT";
      break;
#endif
#ifdef ENOSPC
    case ENOSPC:
      ret = "ENOSPC";
      break;
#endif
#ifdef ENOSR
    case ENOSR:
      ret = "ENOSR";
      break;
#endif
#ifdef ENOSTR
    case ENOSTR:
      ret = "ENOSTR";
      break;
#endif
#ifdef ENOSYS
    case ENOSYS:
      ret = "ENOSYS";
      break;
#endif
#ifdef ENOTCONN
    case ENOTCONN:
      ret = "ENOTCONN";
      break;
#endif
#ifdef ENOTDIR
    case ENOTDIR:
      ret = "ENOTDIR";
      break;
#endif
#ifdef ENOTEMPTY
    case ENOTEMPTY:
      ret = "ENOTEMPTY";
      break;
#endif
#ifdef ENOTSOCK
    case ENOTSOCK:
      ret = "ENOTSOCK";
      break;
#endif
#ifdef ENOTSUP
    case ENOTSUP:		// == EOPNOTSUPP
      ret = "ENOTSUP";
      break;
#endif
#ifdef ENOTTY
    case ENOTTY:
      ret = "ENOTTY";
      break;
#endif
#ifdef ENXIO
    case ENXIO:
      ret = "ENXIO";
      break;
#endif
#ifdef EOVERFLOW
    case EOVERFLOW:
      ret = "EOVERFLOW";
      break;
#endif
#ifdef EPERM
    case EPERM:
      ret = "EPERM";
      break;
#endif
#ifdef EPIPE
    case EPIPE:
      ret = "EPIPE";
      break;
#endif
#ifdef EPROTO
    case EPROTO:
      ret = "EPROTO";
      break;
#endif
#ifdef EPROTONOSUPPORT
    case EPROTONOSUPPORT:
      ret = "EPROTONOSUPPORT";
      break;
#endif
#ifdef EPROTOTYPE
    case EPROTOTYPE:
      ret = "EPROTOTYPE";
      break;
#endif
#ifdef ERANGE
    case ERANGE:
      ret = "ERANGE";
      break;
#endif
#ifdef EROFS
    case EROFS:
      ret = "EROFS";
      break;
#endif
#ifdef ESPIPE
    case ESPIPE:
      ret = "ESPIPE";
      break;
#endif
#ifdef ESRCH
    case ESRCH:
      ret = "ESRCH";
      break;
#endif
#ifdef ESTALE
    case ESTALE:
      ret = "ESTALE";
      break;
#endif
#ifdef ETIME
    case ETIME:
      ret = "ETIME";
      break;
#endif
#ifdef ETIMEDOUT
    case ETIMEDOUT:
      ret = "ETIMEDOUT";
      break;
#endif
#ifdef ETXTBSY
    case ETXTBSY:
      ret = "ETXTBSY";
      break;
#endif
#ifdef EXDEV
    case EXDEV:
      ret = "EXDEV";
      break;
#endif
    }

  return ret;
}

/**
 * lw6sys_log_set_file:
 *
 * @filename: the name of the log file.
 *
 * Sets up the log file. Until you call this function, messages
 * all logged to the default log file, as returned by
 * the @lw6sys_get_default_log_file function.
 *
 * Return value: void
 */
void
lw6sys_log_set_file (char *filename)
{
  memset (static_log_filename, 0, sizeof (static_log_filename));
  memcpy (static_log_filename, filename,
	  lw6sys_min (sizeof (static_log_filename) - 1, strlen (filename)));
}

/**
 * lw6sys_log_clear:
 *
 * @filename: the name of the log file.
 *
 * Clears the log file, this function would typically be called
 * at the beginning of the program.
 *
 * Return value: void
 */
void
lw6sys_log_clear (char *filename)
{
  lw6sys_log_set_file (filename);

  lw6sys_clear_file (filename);
  if (!lw6sys_file_exists (filename))
    {
      lw6sys_log_sos (_("can't open log file \"%s\""), filename);
    }
}

static FILE *
open_log_file ()
{
  FILE *ret = NULL;
  char *name;

  name = get_log_file ();
  if (name)
    {
      ret = fopen (name, "a");
      LW6SYS_FREE (name);
    }

  return ret;
}

static void
log_to_file (FILE * f, int level_id, char *level_str, char *file, int line,
	     char *fmt, va_list ap)
{
  char ctime_buf[_LW6SYS_LOG_CTIME_BUF_SIZE];
  time_t t;
  char *file_only;

  file_only = strrchr (file, '/');
  if (file_only && *file_only)
    {
      file_only++;
    }
  else
    {
      file_only = file;
    }

  t = time (NULL);
  ctime_r (&t, ctime_buf);
  ctime_buf[_LW6SYS_LOG_CTIME_ZERO] = '\0';

  fprintf (f, "%s\t%s:%d\t%d\t%s\t", ctime_buf,
	   file_only, line, level_id, level_str);
  vfprintf (f, fmt, ap);
  fprintf (f, "\n");
  fflush (f);
}

static void
log_to_console (FILE * f, char *level_str, char *fmt, va_list ap)
{
  fprintf (f, "%s: %s", lw6sys_build_get_package_tarname (), level_str);
  vfprintf (f, fmt, ap);
  fprintf (f, "\n");
  fflush (f);
}

/**
 * lw6sys_log:
 *
 * @level_id: the log level to use. Possible values are, by order, 
 *   LW6SYS_LOG_DEBUG_ID (-1), 
 *   LW6SYS_LOG_INFO_ID (0), 
 *   LW6SYS_LOG_NOTICE_ID (1), 
 *   LW6SYS_LOG_WARNING_ID (2) and 
 *   LW6SYS_LOG_ERROR_ID (3).
 * @file: the name of the source file where the function is called,
 *   one can use __FILE__
 * @line: the line in the source file where the function is called,
 *   one can use __LINE__
 * @fmt: a printf-like format string
 * @...: printf-like arguments, corresponding to @fmt.
 *
 * This function is usually called with the first three arguments
 * packed into a single macro. A typical use is 
 * "lw6sys_log(LW6SYS_LOG_WARNING,"sys","problem %s %d","foo",1)".
 * The @LW6SYS_LOG_WARNING macro expands and fills the first 3
 * args, so there's no need to type __FILE__ and __LINE__ again
 * and again. Note that this function will reset errno. The
 * idea is to call it whenever there's something to do with errno
 * (if you deal with errno, it's a good habit to log it) then
 * errno is cleared so that it won't interfere with next log messages.
 *
 * @Return value: void
 */
void
lw6sys_log (int level_id, char *file, int line, char *fmt, ...)
{
  char *level_str = NULL;
  int syslog_priority = 0;
  FILE *f;
  int errno_int;
  va_list ap;
  va_list ap2;

  errno_int = errno;

  va_start (ap, fmt);

  switch (level_id)
    {
    case LW6SYS_LOG_INFO_ID:
      level_str = lw6sys_str_copy ("");
      syslog_priority = LOG_INFO;
      break;
    case LW6SYS_LOG_NOTICE_ID:
      level_str = lw6sys_str_copy ("");
      syslog_priority = LOG_NOTICE;
      break;
    case LW6SYS_LOG_WARNING_ID:
      if (errno_int)
	{
	  level_str =
	    lw6sys_new_sprintf (_("WARNING! (errno=%d:%s) "), errno_int,
				errno_str (errno_int));
	}
      else
	{
	  level_str = lw6sys_str_copy (_("WARNING! "));
	}
      syslog_priority = LOG_WARNING;
      break;
    case LW6SYS_LOG_ERROR_ID:
      if (errno_int)
	{
	  level_str =
	    lw6sys_new_sprintf (_("ERROR! (errno=%d:%s) "), errno_int,
				errno_str (errno_int));
	}
      else
	{
	  level_str = lw6sys_str_copy (_("ERROR! "));
	}
      syslog_priority = LOG_ERR;
      break;
    default:			// LW6SYS_LOG_DEBUG_ID
      level_str = lw6sys_str_copy (_("[debug] "));
      syslog_priority = LOG_DEBUG;
      break;
    }

  if (fmt != NULL)
    {
      if (level_id != LW6SYS_LOG_INFO_ID)
	{
	  va_copy (ap2, ap);
	  log_to_console (stderr, level_str, fmt, ap2);
	  va_end (ap2);
	}
      if (level_id >= LW6SYS_LOG_NOTICE_ID)
	{
	  va_copy (ap2, ap);
	  vsyslog (syslog_priority, fmt, ap2);
	  va_end (ap2);
	}
      f = open_log_file ();
      if (f)
	{
	  va_copy (ap2, ap);
	  log_to_file (f, level_id, level_str, file, line, fmt, ap2);
	  va_end (ap2);
	  fclose (f);
	}
    }

  va_end (ap);

  if (level_str)
    {
      LW6SYS_FREE (level_str);
    }

  /*
   * We reset errno to avoid getting pointless errno in logs,
   * the idea is to print errno when needed...
   */
  errno = 0;
}

/**
 * lw6sys_log_sos:
 *
 * @fmt: a printf-like format string
 * @...: printf-like arguments, corresponding to @fmt.
 *
 * This function is a special log function which will dump informations
 * on the console only, without opening any log file whatsoever. The idea
 * is that it's a "never fail" function. In some cases, for instance reporting
 * that log file can't be opened, it's not wise to use the standard log
 * function.
 *
 * @Return value: void
 */
void
lw6sys_log_sos (char *fmt, ...)
{
  va_list ap;
  va_list ap2;

  va_start (ap, fmt);

  va_copy (ap2, ap);
  log_to_console (stderr, _("SOS! "), fmt, ap2);
  va_end (ap2);

  va_end (ap);
}
