/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>
#include <unistd.h>

#include "config.h"
#include "sys.h"
#include "sys-internal.h"

/**
 * lw6sys_malloc
 *
 * @size: number of bytes to allocate.
 * @file: name of the file calling the function, use @__FILE__
 * @line: line in the file calling the function, use @__LINE__
 *
 * This is a wrapper over the standard @malloc function. Additionnally
 * it will keep track of the call with an internal program-wide counter,
 * thus enabling memory leak checks. You should not use this function
 * directly but use the macro @LW6SYS_MALLOC which has the same syntax,
 * without the last two parameters, which are automatically provided
 * by macro expansion.
 *
 * Return value: the newly allocated pointer. Data is not initialized.
 */
void *
lw6sys_malloc (int size, char *file, int line)
{
  void *ptr;

  ptr = malloc (size);

  if (ptr != NULL)
    {
      _lw6sys_bazooka_register_malloc (ptr, size, file, line);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("malloc %d bytes failed in %s:%d"), size, file, size);
    }

  return ptr;
}

/**
 * lw6sys_calloc
 *
 * @size: number of bytes to allocate.
 * @file: name of the file calling the function, use @__FILE__
 * @line: line in the file calling the function, use @__LINE__
 *
 * This is a wrapper over the standard @calloc function. Additionnally
 * it will keep track of the call with an internal program-wide counter,
 * thus enabling memory leak checks. You should not use this function
 * directly but use the macro @LW6SYS_CALLOC which has the same syntax,
 * without the last two parameters, which are automatically provided
 * by macro expansion.
 *
 * Return value: the newly allocated pointer. Data is filled with zeros.
 */
void *
lw6sys_calloc (int size, char *file, int line)
{
  void *ptr;

  ptr = calloc (size, sizeof (char));

  if (ptr != NULL)
    {
      _lw6sys_bazooka_register_calloc (ptr, size, file, line);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("calloc %d bytes failed in %s:%d"), size, file, size);
    }

  return ptr;
}

/**
 * lw6sys_realloc
 *
 * @ptr: the pointer to reallocate.
 * @size: number of bytes to allocate.
 * @file: name of the file calling the function, use @__FILE__
 * @line: line in the file calling the function, use @__LINE__
 *
 * This is a wrapper over the standard @realloc function.
 * You should not use this function
 * directly but use the macro @LW6SYS_REALLOC which has the same syntax,
 * without the last two parameters, which are automatically provided
 * by macro expansion.
 *
 * Return value: the newly allocated pointer.
 */
void *
lw6sys_realloc (void *ptr, int size, char *file, int line)
{
  void *ptr2;

  _lw6sys_bazooka_register_realloc_1 (ptr, size, file, line);

  ptr2 = realloc (ptr, size);

  if (ptr2)
    {
      _lw6sys_bazooka_register_realloc_2 (ptr, ptr2, size, file, line);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("realloc %d bytes failed in %s:%d"), size, file, size);
    }

  return ptr2;
}

/**
 * lw6sys_free
 *
 * @ptr: the pointer to free.
 * @file: name of the file calling the function, use @__FILE__
 * @line: line in the file calling the function, use @__LINE__
 *
 * This is a wrapper over the standard @free function. Additionnally
 * it will keep track of the call with an internal program-wide counter,
 * thus enabling memory leak checks. You should not use this function
 * directly but use the macro @LW6SYS_FREE which has the same syntax,
 * without the last two parameters, which are automatically provided
 * by macro expansion.
 *
 * Return value: none.
 */
void
lw6sys_free (void *ptr, char *file, int line)
{
  if (ptr != NULL)
    {
      _lw6sys_bazooka_register_free (ptr);
      /*
       * It's important to free the pointer after registering, since
       * the later might overwrite memory in some cases.
       */
      free (ptr);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("trying to free null pointer in %s:%d"), file, line);
    }
}

/**
 * lw6sys_free_callback
 *
 * @ptr: the pointer to free.
 *
 * This is a callback to be used when the @lw6sys_free does not fit.
 * A good example is a list, which, to free its elements, requires
 * you to provide a callback that only takes 1 arg, the pointer
 * to free. Problem, @lw6sys_free takes 3 args. And the @LW6SYS_FREE
 * macro is not usable in such a context. And you can't use standard @free
 * either for it would mess up the @malloc / @free automatical count
 * which is so convenient to track memory leaks. So this callback is
 * here, it's only drawback is that in case of an error, the error
 * will not be reported with the real file and line parameters. It's
 * still better than nothing.
 *
 * Return value: none.
 */
void
lw6sys_free_callback (void *ptr)
{
  LW6SYS_FREE (ptr);
}

/**
 * lw6sys_megabytes_available
 *
 * Gives a raw approximation of available memory, in megabytes.
 * Value is to be taken with distance, but it can give good hints
 * when system is running short of ressources.
 *
 * Return value: number of megabytes (physical memory) available.
 */
int
lw6sys_megabytes_available ()
{
  int ret = 0;

  ret = (sysconf (_SC_PAGESIZE) * sysconf (_SC_AVPHYS_PAGES)) / (1024 * 1024);

  return ret;
}

/**
 * lw6sys_is_big_endian
 *
 * Checks the endianess of the machine. PPC is big endian, for instance.
 *
 * Return value: 1 if system is big endian, 0 if little endian.
 */
int
lw6sys_is_big_endian ()
{
  return !lw6sys_is_little_endian ();
}

/**
 * lw6sys_is_little_endian
 *
 * Checks the endianess of the machine. x86 is little endian, for instance.
 *
 * Return value: 1 if system is little endian, 0 if big endian.
 */
int
lw6sys_is_little_endian ()
{
  int int_test = 1;
  char *char_test = (char *) &int_test;
  int ret = ((*char_test) != 0);

  return ret;
}
