/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>

#include "config.h"
#include "sys.h"

#define CONFIG_FILE "config.xml"
#define LOG_FILE "log.csv"
#define MOD_LIBDIR_MASK "%s/%s-%s"
#define MOD_PREFIX_MASK "%s/lib/%s-%s"
#define SHARE_DATADIR_MASK "%s/%s-%s"
#define SHARE_PREFIX_MASK "%s/share/%s-%s"
#define DATA_DIR "data"
#define MAP_DIR "map"
#define SCRIPT_FILE "script/liquidwar6.scm"
#ifdef WIN32
#define README_FILE "README.txt"
#else
#define README_FILE "README"
#endif

char *
lw6sys_get_default_user_dir ()
{
  char *home_dir = NULL;
  char *user_dir = NULL;
  char *dot_tarname = NULL;

  home_dir = lw6sys_get_home ();
  if (home_dir)
    {
      dot_tarname =
	lw6sys_str_concat (".", lw6sys_build_get_package_tarname ());
      if (dot_tarname)
	{
	  user_dir = lw6sys_path_concat (home_dir, dot_tarname);
	  LW6SYS_FREE (dot_tarname);
	}
      LW6SYS_FREE (home_dir);
    }

  return user_dir;
}

static char *
get_config_file_with_user_dir (char *user_dir)
{
  char *config_file;

  config_file = lw6sys_path_concat (user_dir, CONFIG_FILE);

  return config_file;
}

char *
lw6sys_get_default_config_file ()
{
  char *config_file = NULL;
  char *user_dir = NULL;

  user_dir = lw6sys_get_default_user_dir ();
  if (user_dir)
    {
      config_file = get_config_file_with_user_dir (user_dir);
      LW6SYS_FREE (user_dir);
    }

  return config_file;
}

static char *
get_log_file_with_user_dir (char *user_dir)
{
  char *log_file;

  log_file = lw6sys_path_concat (user_dir, LOG_FILE);

  return log_file;
}

char *
lw6sys_get_default_log_file ()
{
  char *log_file = NULL;
  char *user_dir = NULL;

  user_dir = lw6sys_get_default_user_dir ();
  if (user_dir)
    {
      log_file = get_log_file_with_user_dir (user_dir);
      LW6SYS_FREE (user_dir);
    }

  return log_file;
}

char *
lw6sys_get_default_prefix ()
{
  char *prefix = NULL;

  prefix = lw6sys_path_strip_slash (lw6sys_build_get_prefix ());

  return prefix;
}

static char *
get_dir (char *mask1, char *prefix1, char *mask2, char *prefix2, char *sub)
{
  char *system_dir = NULL;
  char *top_srcdir = NULL;
  char *dir = NULL;

  if (dir == NULL)
    {
      system_dir =
	lw6sys_new_sprintf (mask1, prefix1,
			    lw6sys_build_get_package_tarname (),
			    lw6sys_build_get_version ());
      if (system_dir)
	{
	  dir = lw6sys_path_concat (system_dir, sub);
	  if (dir)
	    {
	      if (!lw6sys_dir_exists (dir))
		{
		  // directory doesn't exist, we ignore it
		  LW6SYS_FREE (dir);
		  dir = NULL;
		}
	    }
	  LW6SYS_FREE (system_dir);
	}
    }

  if (dir == NULL)
    {
      system_dir =
	lw6sys_new_sprintf (mask2, prefix2,
			    lw6sys_build_get_package_tarname (),
			    lw6sys_build_get_version ());
      if (system_dir)
	{
	  dir = lw6sys_path_concat (system_dir, sub);
	  if (dir)
	    {
	      if (!lw6sys_dir_exists (dir))
		{
		  // directory doesn't exist, we ignore it
		  LW6SYS_FREE (dir);
		  dir = NULL;
		}
	    }
	  LW6SYS_FREE (system_dir);
	}
    }

  if (dir == NULL && strlen (sub) > 0)
    {
      top_srcdir = lw6sys_build_get_top_srcdir ();
      dir = lw6sys_path_concat (top_srcdir, sub);
      if (dir)
	{
	  if (!lw6sys_dir_exists (dir))
	    {
	      // directory doesn't exist, we ignore it
	      LW6SYS_FREE (dir);
	      dir = NULL;
	    }
	}
    }

  if (dir == NULL)
    {
      // if not installed, if no source, then search in local directory
      dir = lw6sys_path_concat (".", sub);
    }

  return dir;
}

static char *
get_file (char *mask1, char *prefix1, char *mask2, char *prefix2, char *sub)
{
  char *system_dir = NULL;
  char *top_srcdir = NULL;
  char *file = NULL;

  if (file == NULL)
    {
      system_dir =
	lw6sys_new_sprintf (mask1, prefix1,
			    lw6sys_build_get_package_tarname (),
			    lw6sys_build_get_version ());
      if (system_dir)
	{
	  file = lw6sys_path_concat (system_dir, sub);
	  if (file)
	    {
	      if (!lw6sys_file_exists (file))
		{
		  // file doesn't exist, we ignore it
		  LW6SYS_FREE (file);
		  file = NULL;
		}
	    }
	  LW6SYS_FREE (system_dir);
	}
    }

  if (file == NULL)
    {
      system_dir =
	lw6sys_new_sprintf (mask2, prefix2,
			    lw6sys_build_get_package_tarname (),
			    lw6sys_build_get_version ());
      if (system_dir)
	{
	  file = lw6sys_path_concat (system_dir, sub);
	  if (file)
	    {
	      if (!lw6sys_file_exists (file))
		{
		  // file doesn't exist, we ignore it
		  LW6SYS_FREE (file);
		  file = NULL;
		}
	    }
	  LW6SYS_FREE (system_dir);
	}
    }

  if (file == NULL && strlen (sub) > 0)
    {
      top_srcdir = lw6sys_build_get_top_srcdir ();
      file = lw6sys_path_concat (top_srcdir, sub);
      if (file)
	{
	  if (!lw6sys_file_exists (file))
	    {
	      // file doesn't exist, we ignore it
	      LW6SYS_FREE (file);
	      file = NULL;
	    }
	}
    }

  if (file == NULL)
    {
      // if not installed, if no source, then search in local directory
      file = lw6sys_path_concat (".", sub);
    }

  return file;
}

static char *
get_mod_dir_with_prefix (char *prefix)
{
  char *mod_dir = NULL;
  char *libdir = NULL;

  libdir = lw6sys_build_get_libdir ();
  mod_dir = get_dir (MOD_PREFIX_MASK, prefix, MOD_LIBDIR_MASK, libdir, "");

  return mod_dir;
}

char *
lw6sys_get_default_mod_dir ()
{
  char *mod_dir = NULL;
  char *prefix = NULL;

  prefix = lw6sys_get_default_prefix ();
  if (prefix)
    {
      mod_dir = get_mod_dir_with_prefix (prefix);
      LW6SYS_FREE (prefix);
    }

  return mod_dir;
}

static char *
get_share_dir_with_prefix (char *prefix, char *sub)
{
  char *share_dir = NULL;
  char *datadir = NULL;

  datadir = lw6sys_build_get_datadir ();
  share_dir =
    get_dir (SHARE_PREFIX_MASK, prefix, SHARE_DATADIR_MASK, datadir, sub);

  return share_dir;
}

char *
lw6sys_get_default_data_dir ()
{
  char *data_dir = NULL;
  char *prefix = NULL;

  prefix = lw6sys_get_default_prefix ();
  if (prefix)
    {
      data_dir = get_share_dir_with_prefix (prefix, DATA_DIR);
      LW6SYS_FREE (prefix);
    }

  return data_dir;
}

char *
lw6sys_get_default_map_dir ()
{
  char *map_dir = NULL;
  char *prefix = NULL;

  prefix = lw6sys_get_default_prefix ();
  if (prefix)
    {
      map_dir = get_share_dir_with_prefix (prefix, MAP_DIR);
      LW6SYS_FREE (prefix);
    }

  return map_dir;
}

static char *
get_path_with_dir_and_user_dir (char *dir, char *user_dir, char *sub,
				int create_dir, char *readme_comment)
{
  char *path = NULL;
  char *user_sub_dir = NULL;
  char *user_sub_dir_readme = NULL;
  char *readme_content = NULL;

  user_sub_dir = lw6sys_path_concat (user_dir, sub);
  if (user_sub_dir)
    {
      if (create_dir)
	{
	  if (!lw6sys_dir_exists (user_sub_dir))
	    {
	      lw6sys_create_dir (user_sub_dir);
	    }
	  user_sub_dir_readme =
	    lw6sys_path_concat (user_sub_dir, README_FILE);
	  if (user_sub_dir_readme)
	    {
	      if (readme_comment)
		{
		  readme_content =
		    lw6sys_str_reformat (readme_comment,
					 LW6SYS_REFORMAT_DEFAULT_PREFIX,
					 LW6SYS_REFORMAT_DEFAULT_NB_COLUMNS);
		  if (readme_content)
		    {
		      lw6sys_write_file_content (user_sub_dir_readme,
						 readme_content);
		      LW6SYS_FREE (readme_content);
		    }
		}
	      LW6SYS_FREE (user_sub_dir_readme);
	    }
	}
      path = lw6sys_env_concat (user_sub_dir, dir);
      LW6SYS_FREE (user_sub_dir);
    }

  return path;
}

char *
lw6sys_get_default_map_path ()
{
  char *map_path = NULL;
  char *map_dir = NULL;
  char *user_dir = NULL;

  map_dir = lw6sys_get_default_map_dir ();
  if (map_dir)
    {
      user_dir = lw6sys_get_default_user_dir ();
      if (user_dir)
	{
	  map_path =
	    get_path_with_dir_and_user_dir (map_dir, user_dir, MAP_DIR, 0,
					    NULL);
	  LW6SYS_FREE (user_dir);
	}
      LW6SYS_FREE (map_dir);
    }

  return map_path;
}

static char *
get_share_file_with_prefix (char *prefix, char *sub)
{
  char *share_file = NULL;
  char *datadir;

  datadir = lw6sys_build_get_datadir ();
  share_file =
    get_file (SHARE_PREFIX_MASK, prefix, SHARE_DATADIR_MASK, datadir, sub);

  return share_file;
}

char *
lw6sys_get_default_script_file ()
{
  char *script_file = NULL;
  char *prefix = NULL;

  prefix = lw6sys_get_default_prefix ();
  if (prefix)
    {
      script_file = get_share_file_with_prefix (prefix, SCRIPT_FILE);
      LW6SYS_FREE (prefix);
    }

  return script_file;
}

void
lw6sys_options_log_defaults ()
{
  char *path = NULL;

  path = lw6sys_get_default_user_dir ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default user dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_config_file ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default config file is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_log_file ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default log file is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_prefix ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default prefix is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_mod_dir ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default mod dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_data_dir ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default data dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_map_dir ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default map dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_map_path ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default map path is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_default_script_file ();
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("default script file is \"%s\""), path);
      LW6SYS_FREE (path);
    }
}

static void *
get_command_arg_with_env (int argc, char *argv[], char *keyword)
{
  char *arg_value = NULL;
  char *ret = NULL;

  arg_value = lw6sys_arg_get_value_with_env (argc, argv, keyword);
  if (arg_value)
    {
      ret = lw6sys_path_strip_slash (arg_value);
      LW6SYS_FREE (arg_value);
    }

  return ret;
}

char *
lw6sys_get_user_dir (int argc, char *argv[])
{
  char *user_dir = NULL;

  user_dir = get_command_arg_with_env (argc, argv, "user-dir");
  if (user_dir == NULL)
    {
      user_dir = lw6sys_get_default_user_dir ();
    }

  if (user_dir)
    {
      if (!lw6sys_dir_exists (user_dir))
	{
	  lw6sys_create_dir_silent (user_dir);
	  if (!lw6sys_dir_exists (user_dir))
	    {
	      lw6sys_log_sos (_("can't open user dir \"%s\""), user_dir);
	    }
	}
    }

  return user_dir;
}

char *
lw6sys_get_config_file (int argc, char *argv[])
{
  char *config_file = NULL;
  char *user_dir = NULL;

  config_file = get_command_arg_with_env (argc, argv, "config-file");
  if (config_file == NULL)
    {
      user_dir = lw6sys_get_user_dir (argc, argv);
      if (user_dir)
	{
	  config_file = get_config_file_with_user_dir (user_dir);
	  LW6SYS_FREE (user_dir);
	}
    }

  return config_file;
}

char *
lw6sys_get_log_file (int argc, char *argv[])
{
  char *log_file = NULL;
  char *user_dir = NULL;

  log_file = get_command_arg_with_env (argc, argv, "log-file");
  if (log_file == NULL)
    {
      user_dir = lw6sys_get_user_dir (argc, argv);
      if (user_dir)
	{
	  log_file = get_log_file_with_user_dir (user_dir);
	  LW6SYS_FREE (user_dir);
	}
    }

  return log_file;
}

char *
lw6sys_get_prefix (int argc, char *argv[])
{
  char *prefix = NULL;

  prefix = get_command_arg_with_env (argc, argv, "prefix");
  if (prefix == NULL)
    {
      prefix = lw6sys_get_default_prefix ();
    }

  return prefix;
}

char *
lw6sys_get_mod_dir (int argc, char *argv[])
{
  char *mod_dir = NULL;
  char *prefix = NULL;

  mod_dir = get_command_arg_with_env (argc, argv, "mod-dir");
  if (mod_dir == NULL)
    {
      prefix = lw6sys_get_prefix (argc, argv);
      if (prefix)
	{
	  mod_dir = get_mod_dir_with_prefix (prefix);
	  LW6SYS_FREE (prefix);
	}
    }

  return mod_dir;
}

char *
lw6sys_get_data_dir (int argc, char *argv[])
{
  char *data_dir = NULL;
  char *prefix = NULL;

  data_dir = get_command_arg_with_env (argc, argv, "data-dir");
  if (data_dir == NULL)
    {
      prefix = lw6sys_get_prefix (argc, argv);
      if (prefix)
	{
	  data_dir = get_share_dir_with_prefix (prefix, DATA_DIR);
	  LW6SYS_FREE (prefix);
	}
    }

  return data_dir;
}

char *
lw6sys_get_map_dir (int argc, char *argv[])
{
  char *map_dir = NULL;
  char *prefix = NULL;

  map_dir = get_command_arg_with_env (argc, argv, "map-dir");
  if (map_dir == NULL)
    {
      prefix = lw6sys_get_prefix (argc, argv);
      if (prefix)
	{
	  map_dir = get_share_dir_with_prefix (prefix, MAP_DIR);
	  LW6SYS_FREE (prefix);
	}
    }

  return map_dir;
}

char *
lw6sys_get_map_path (int argc, char *argv[])
{
  char *map_path = NULL;
  char *map_path_cmd = NULL;
  char *map_path_env = NULL;
  char *map_dir = NULL;
  char *user_dir = NULL;
  char *tmp = NULL;

  map_dir = lw6sys_get_map_dir (argc, argv);
  if (map_dir)
    {
      user_dir = lw6sys_get_user_dir (argc, argv);
      if (user_dir)
	{
	  map_path =
	    get_path_with_dir_and_user_dir (map_dir, user_dir, MAP_DIR, 1,
					    _
					    ("This is your Liquid War 6 user map directory. You can put your own maps here, they will be automatically loaded by the game, and appear in the menus. To create maps, you need to 1) read the documentation 2) study existing map to see how things work in practice and 3) be creative. FYI the Liquid War 6 map format is rather simple and hopefully well documented, the general idea is that a map is a directory containing files, the most important of them being 'map.png'. Again, read the documentation. I repeat: 'Read the docs!'. http://www.gnu.org/software/liquidwar6/manual/"));
	  if (map_path)
	    {
	      map_path_env = lw6sys_getenv ("map-path");
	      if (map_path && map_path_env)
		{
		  tmp = map_path;
		  map_path = lw6sys_env_concat (map_path_env, map_path);
		  LW6SYS_FREE (tmp);
		  LW6SYS_FREE (map_path_env);
		}
	      map_path_cmd = lw6sys_arg_get_value (argc, argv, "map-path");
	      if (map_path && map_path_cmd)
		{
		  tmp = map_path;
		  map_path = lw6sys_env_concat (map_path_cmd, map_path);
		  LW6SYS_FREE (tmp);
		  LW6SYS_FREE (map_path_cmd);
		}
	    }
	  LW6SYS_FREE (user_dir);
	}
      LW6SYS_FREE (map_dir);
    }

  return map_path;
}

char *
lw6sys_get_script_file (int argc, char *argv[])
{
  char *script_file = NULL;
  char *prefix = NULL;

  script_file = get_command_arg_with_env (argc, argv, "script-file");
  if (script_file == NULL)
    {
      prefix = lw6sys_get_prefix (argc, argv);
      if (prefix)
	{
	  script_file = get_share_file_with_prefix (prefix, SCRIPT_FILE);
	  LW6SYS_FREE (prefix);
	}
    }

  return script_file;
}

void
lw6sys_options_log (int argc, char *argv[])
{
  char *path = NULL;

  path = lw6sys_get_user_dir (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("user dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_config_file (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("config file is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_log_file (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("log file is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_prefix (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("prefix is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_mod_dir (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("mod dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_data_dir (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("data dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_map_dir (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("map dir is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_map_path (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("map path is \"%s\""), path);
      LW6SYS_FREE (path);
    }
  path = lw6sys_get_script_file (argc, argv);
  if (path)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("script file is \"%s\""), path);
      LW6SYS_FREE (path);
    }
}
