/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>

#include "config.h"
#include "sys.h"

#define DIR_SEP_CHAR_UNIX '/'
#define DIR_SEP_STR_UNIX "/"
#define DIR_SEP_CHAR_WIN32 '\\'
#define DIR_SEP_STR_WIN32 "\\"

#ifdef SHOULD_CHECK_IF_CYGWIN_SUPPORTS_SLASH_IF_YES_NEVER_DEFINE_THIS
#define DIR_SEP_CHAR DIR_SEP_CHAR_WIN32
#define DIR_SEP_STR DIR_SEP_STR_WIN32
#else
#define DIR_SEP_CHAR DIR_SEP_CHAR_UNIX
#define DIR_SEP_C DIR_SEP_CHAR_UNIX
#endif

static int
is_dir_sep (char c)
{
  return ((c == DIR_SEP_CHAR_UNIX) || (c == DIR_SEP_CHAR_WIN32));
}

/*
 * Tests if a file exists and is readable.
 */
int
lw6sys_file_exists (char *filename)
{
  int ret = 0;
  FILE *f;

  f = fopen (filename, "r");
  if (f)
    {
      ret = 1;
      fclose (f);
    }

  return ret;
}

/*
 * Tests if a directory exists.
 */
int
lw6sys_dir_exists (char *dirname)
{
  struct dirent **namelist;
  int n;
  int ret = 0;

  n = scandir (dirname, &namelist, NULL, NULL);
  if (n >= 0)
    {
      ret = 1;
      free (namelist);
    }

  return ret;
}

static int
create_dir (char *dirname, int verbose)
{
  int ret = 0;

  if (mkdir (dirname, S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP) == 0)
    {
      if (verbose)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating dir \"%s\""), dirname);
	}
      ret = 1;
    }
  else
    {
      if (verbose)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, _("unable to create dir \"%s\""),
		      dirname);
	}
    }

  return ret;
}

/*
 * Wrapper on mkdir with all the options we like, + logs.
 */
int
lw6sys_create_dir (char *dirname)
{
  return create_dir (dirname, 1);
}

int
lw6sys_create_dir_silent (char *dirname)
{
  return create_dir (dirname, 0);
}

char *
lw6sys_path_add_slash (char *path)
{
  char *ret;
  int len;

  len = strlen (path);
  if (len == 0 || (len > 0 && !is_dir_sep (path[len - 1])))
    {
      ret = lw6sys_str_concat (path, "/");
    }
  else
    {
      ret = lw6sys_str_copy (path);
    }

  return ret;
}

char *
lw6sys_path_strip_slash (char *path)
{
  char *ret;
  int len;

  ret = lw6sys_str_copy (path);
  if (ret)
    {
      len = strlen (ret);
      if (len > 0 && is_dir_sep (ret[len - 1]))
	{
	  ret[len - 1] = '\0';
	}
    }

  return ret;
}

char *
lw6sys_path_concat (char *path1, char *path2)
{
  char *path1_stripped = NULL;
  char *path2_stripped = NULL;
  char *path2_start = NULL;
  char *ret = NULL;

  path1_stripped = lw6sys_path_strip_slash (path1);
  if (path1_stripped)
    {
      path2_stripped = lw6sys_path_strip_slash (path2);
      if (path2_stripped)
	{
	  path2_start = path2_stripped;
	  if (is_dir_sep (path2_start[0]))
	    {
	      path2_start++;
	    }
	  if (strlen (path2_start) > 0 && strcmp (path2_start, ".") != 0)
	    {
	      ret = lw6sys_new_sprintf ("%s/%s", path1_stripped, path2_start);
	    }
	  else
	    {
	      ret = lw6sys_str_copy (path1_stripped);
	    }
	  LW6SYS_FREE (path2_stripped);
	}
      LW6SYS_FREE (path1_stripped);
    }

  return ret;
}

int
lw6sys_path_is_relative (char *path)
{
  int ret = 0;

  if (strcmp (".", path) == 0 ||
      strncmp ("." DIR_SEP_STR_UNIX, path, 2) == 0 ||
      strncmp ("." DIR_SEP_STR_WIN32, path, 2) == 0 ||
      strncmp (".." DIR_SEP_STR_UNIX, path, 3) == 0 ||
      strncmp (".." DIR_SEP_STR_WIN32, path, 3) == 0)
    {
      ret = 1;
    }

  return ret;
}

char *
lw6sys_path_parent (char *path)
{
  char *parent = NULL;
  char *stripped_path = NULL;
  char *pos = NULL;

  stripped_path = lw6sys_path_strip_slash (path);
  if (stripped_path)
    {
      pos = strrchr (stripped_path, DIR_SEP_CHAR);
      if (pos)
	{
	  if (strcmp (pos + 1, "..") == 0)
	    {
	      parent = lw6sys_path_concat (stripped_path, "..");
	    }
	  else
	    {
	      (*pos) = '\0';
	      parent = lw6sys_str_copy (stripped_path);
	    }
	}
      else
	{
	  if (strcmp (stripped_path, "") == 0
	      || strcmp (stripped_path, ".") == 0)
	    {
	      parent = lw6sys_str_copy ("..");
	    }
	  else
	    {
	      parent = lw6sys_str_copy (".");
	    }
	}
      LW6SYS_FREE (stripped_path);
    }

  return parent;
}

char *
lw6sys_path_unparent (char *path)
{
  char *unparent = NULL;
  char *slashed_path = NULL;

  slashed_path = lw6sys_path_add_slash (path);
  if (slashed_path)
    {
      if (strncmp (".." DIR_SEP_STR_UNIX, slashed_path, 3) == 0 ||
	  strncmp (".." DIR_SEP_STR_WIN32, slashed_path, 3) == 0)
	{
	  unparent = lw6sys_str_copy (path + 3);
	}
      else
	{
	  unparent = lw6sys_str_copy (path);
	}
      LW6SYS_FREE (slashed_path);
    }

  return unparent;
}

char *
lw6sys_path_unparent_no_malloc (char *path)
{
  char *unparent = NULL;

  if (strncmp (".." DIR_SEP_STR_UNIX, path, 3) == 0 ||
      strncmp (".." DIR_SEP_STR_WIN32, path, 3) == 0)
    {
      unparent = path + 3;
    }
  else
    {
      unparent = path;
    }

  return unparent;
}
