
/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <string.h>

#include "config.h"
#include "tsk.h"
#include "tsk-internal.h"

#define LOADER_LOCK loader_lock (__FILE__,__LINE__,__FUNCTION__,loader_data)
#define LOADER_UNLOCK loader_unlock (__FILE__,__LINE__,__FUNCTION__,loader_data)

static void
loader_lock (char *file, int line, const char *func,
	     _lw6tsk_loader_data_t * loader_data)
{
  /*
   * Could put some log command here to debug
   */
  lw6sys_mutex_lock (loader_data->mutex);
  /*
   * Could put some log command here to debug
   */
}

static void
loader_unlock (char *file, int line, const char *func,
	       _lw6tsk_loader_data_t * loader_data)
{
  /*
   * Could put some log command here to debug
   */
  lw6sys_mutex_unlock (loader_data->mutex);
}

static void
stage1_clear_request (_lw6tsk_loader_stage1_t * stage1)
{
  if (stage1->map_path)
    {
      LW6SYS_FREE (stage1->map_path);
      stage1->map_path = NULL;
    }
  if (stage1->relative_path)
    {
      LW6SYS_FREE (stage1->relative_path);
      stage1->relative_path = NULL;
    }
  if (stage1->default_param)
    {
      lw6sys_assoc_free (stage1->default_param);
      stage1->default_param = NULL;
    }
  if (stage1->forced_param)
    {
      lw6sys_assoc_free (stage1->forced_param);
      stage1->forced_param = NULL;
    }
  stage1->display_shape.w = 0;
  stage1->display_shape.h = 0;
}

static void
stage1_clear_response (_lw6tsk_loader_stage1_t * stage1)
{
  if (stage1->level)
    {
      lw6map_free (stage1->level);
      stage1->level = NULL;
    }
}

static void
stage1_clear (_lw6tsk_loader_stage1_t * stage1)
{
  stage1_clear_request (stage1);
  stage1_clear_response (stage1);
  memset (stage1, 0, sizeof (_lw6tsk_loader_stage1_t));
}

static void
stage2_clear (_lw6tsk_loader_stage2_t * stage2)
{
  if (stage2->game_struct)
    {
      lw6ker_game_struct_free (stage2->game_struct);
    }
  if (stage2->level)
    {
      lw6map_free (stage2->level);
    }
  memset (stage2, 0, sizeof (_lw6tsk_loader_stage2_t));
}

static void
stage3_clear (_lw6tsk_loader_stage3_t * stage3)
{
  if (stage3->game_state)
    {
      lw6ker_game_state_free (stage3->game_state);
    }
  if (stage3->game_struct)
    {
      lw6ker_game_struct_free (stage3->game_struct);
    }
  if (stage3->level)
    {
      lw6map_free (stage3->level);
    }
  memset (stage3, 0, sizeof (_lw6tsk_loader_stage3_t));
}

static void
clear (_lw6tsk_loader_data_t * loader_data)
{
  stage3_clear (&(loader_data->stage3));
  stage2_clear (&(loader_data->stage2));
  stage1_clear (&(loader_data->stage1));
}

static void
stage1 (_lw6tsk_loader_data_t * loader_data)
{
  int request_number = 0;
  char *map_path = NULL;
  char *relative_path = NULL;
  lw6sys_assoc_t *default_param = NULL;
  lw6sys_assoc_t *forced_param = NULL;
  lw6map_level_t *level = NULL;
  lw6sys_wh_t display_shape = { 0, 0 };

  LOADER_LOCK;
  request_number = loader_data->request_number;

  if (loader_data->stage1.map_path && loader_data->stage1.relative_path)
    {
      map_path = lw6sys_str_copy (loader_data->stage1.map_path);
      relative_path = lw6sys_str_copy (loader_data->stage1.relative_path);
      default_param =
	lw6sys_assoc_dup (loader_data->stage1.default_param,
			  (LW6SYS_DUP_FUNC) lw6sys_str_copy);
      forced_param =
	lw6sys_assoc_dup (loader_data->stage1.forced_param,
			  (LW6SYS_DUP_FUNC) lw6sys_str_copy);
      display_shape = loader_data->stage1.display_shape;
      stage1_clear_request (&(loader_data->stage1));
    }
  LOADER_UNLOCK;

  if (map_path && relative_path)
    {
      level =
	lw6ldr_read_relative (map_path, relative_path, default_param,
			      forced_param, &display_shape);
    }

  if (map_path)
    {
      LW6SYS_FREE (map_path);
    }
  if (relative_path)
    {
      LW6SYS_FREE (relative_path);
    }
  if (default_param)
    {
      lw6sys_assoc_free (default_param);
    }
  if (forced_param)
    {
      lw6sys_assoc_free (forced_param);
    }

  if (level)
    {
      char *repr = NULL;

      repr = lw6map_repr (level);
      if (repr)
	{
	  LOADER_LOCK;
	  if (request_number == loader_data->request_number)
	    {
	      stage1_clear_response (&(loader_data->stage1));
	      loader_data->stage1.level = level;
	      lw6sys_log (LW6SYS_LOG_INFO,
			  _
			  ("async load stage1 done, request_number %d, level \"%s\""),
			  request_number, repr);
	      // todo: push request to stage2...
	    }
	  else if (request_number < loader_data->request_number)
	    {
	      lw6sys_log (LW6SYS_LOG_INFO,
			  _
			  ("async load stage1 abort, request number %d < %d, level \"%s\""),
			  request_number, loader_data->request_number, repr);
	      lw6map_free (level);
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _
			  ("async load stage1, inconsistent request number %d > %d, level \"%s\""),
			  request_number, loader_data->request_number, repr);
	    }
	  LOADER_UNLOCK;
	  LW6SYS_FREE (repr);
	}
    }
}

static void
stage2 (_lw6tsk_loader_data_t * loader_data)
{
}

static void
stage3 (_lw6tsk_loader_data_t * loader_data)
{
}

void
_lw6tsk_loader_poll (_lw6tsk_loader_data_t * loader_data)
{
  loader_data->stage = 1;
  stage1 (loader_data);
  loader_data->stage = 2;
  stage2 (loader_data);
  loader_data->stage = 3;
  stage3 (loader_data);
  loader_data->stage = 0;
}

void
lw6tsk_loader_push (lw6tsk_loader_t * loader, char *map_path,
		    char *relative_path, lw6sys_assoc_t * default_param,
		    lw6sys_assoc_t * forced_param,
		    lw6sys_wh_t * display_shape)
{
  _lw6tsk_loader_data_t *loader_data;

  loader_data = (_lw6tsk_loader_data_t *) loader->data;

  LOADER_LOCK;

  loader_data->request_number++;
  clear (loader_data);
  loader_data->stage1.map_path = lw6sys_str_copy (map_path);
  loader_data->stage1.relative_path = lw6sys_str_copy (relative_path);
  loader_data->stage1.default_param =
    lw6sys_assoc_dup (default_param, (LW6SYS_DUP_FUNC) lw6sys_str_copy);
  loader_data->stage1.forced_param =
    lw6sys_assoc_dup (forced_param, (LW6SYS_DUP_FUNC) lw6sys_str_copy);
  loader_data->stage1.display_shape = *display_shape;

  LOADER_UNLOCK;
}

int
lw6tsk_loader_pop (lw6map_level_t ** level,
		   lw6ker_game_struct_t ** game_struct,
		   lw6ker_game_state_t ** game_state,
		   lw6tsk_loader_t * loader)
{
  int ret = 0;
  _lw6tsk_loader_data_t *loader_data;

  loader_data = (_lw6tsk_loader_data_t *) loader->data;

  LOADER_LOCK;

  if (level)
    {
      (*level) = NULL;
      if (loader_data->stage1.level)
	{
	  (*level) = loader_data->stage1.level;
	  loader_data->stage1.level = NULL;
	  ret = 1;
	}
    }
  if (game_struct)
    {
      (*game_struct) = NULL;
    }
  if (game_state)
    {
      (*game_state) = NULL;
    }

  LOADER_UNLOCK;

  return ret;
}

/*
 * This is used to stamp task loaders as they are created.
 * Note that there's no race condition due to the fact that this
 * is global here, in fact even when 2 processes would share
 * this sequence id, it would not matter for they would then
 * try and identify the objects in their on per-process lists,
 * structures, Guile object, whatever they use.
 */
static int seq_id = 0;

static void
loader_data_free (_lw6tsk_loader_data_t * loader_data)
{
  if (loader_data->mutex)
    {
      LOADER_LOCK;

      clear (loader_data);

      LOADER_UNLOCK;

      lw6sys_mutex_destroy (loader_data->mutex);
    }
  LW6SYS_FREE (loader_data);
}

static void
loader_callback (void *data)
{
  _lw6tsk_loader_data_t *loader_data;

  loader_data = (_lw6tsk_loader_data_t *) data;

  while (!loader_data->stop)
    {
      _lw6tsk_loader_poll (loader_data);
      lw6sys_sleep (loader_data->sleep);
    }
}

static void
loader_join (void *data)
{
  _lw6tsk_loader_data_t *loader_data;

  loader_data = (_lw6tsk_loader_data_t *) data;

  loader_data_free (loader_data);
}

/**
 * lw6tsk_loader_new
 *
 * @sleep: how many seconds to wait between every poll
 *
 * Creates a new loader. This object is used to do some reputed
 * slow calculus in the background, in a separated thread. Typical example
 * is map loading. This is a high-level objects which encapsulates threads
 * and other wizardry.
 *
 * Return value: a pointer to the loader, NULL if failed.
 */
lw6tsk_loader_t *
lw6tsk_loader_new (float sleep)
{
  lw6tsk_loader_t *loader;
  _lw6tsk_loader_data_t *loader_data;

  loader = (lw6tsk_loader_t *) LW6SYS_CALLOC (sizeof (lw6tsk_loader_t));
  if (loader)
    {
      loader->id = ++seq_id;
      loader_data =
	(_lw6tsk_loader_data_t *)
	LW6SYS_CALLOC (sizeof (_lw6tsk_loader_data_t));
      loader->data = loader_data;
      if (loader->data)
	{
	  loader_data->sleep = sleep;
	  loader_data->mutex = lw6sys_mutex_create ();
	  if (loader_data->mutex)
	    {
	      loader->thread =
		lw6sys_thread_create (loader_callback, loader_join,
				      loader->data, 0);
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _("unable to create mutex for loader"));
	    }
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("unable to allocate memory for loader data"));
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to allocate memory for loader"));
    }

  if (loader && !loader->thread)
    {
      if (loader->thread)
	{
	  lw6sys_thread_join (loader->thread);
	}
      if (loader->data)
	{
	  loader_data_free (loader->data);
	}
      LW6SYS_FREE (loader);
      loader = NULL;
    }

  return loader;
}

/**
 * lw6tsk_loader_free
 *
 * @loader: the loader to free.
 *
 * Deletes a loader. Will automatically stop the child thread,
 * free data, and so on.
 *
 * Return value: none.
 */
void
lw6tsk_loader_free (lw6tsk_loader_t * loader)
{
  _lw6tsk_loader_data_t *loader_data;

  loader_data = (_lw6tsk_loader_data_t *) loader->data;
  loader_data->stop = 1;
  lw6sys_thread_join (loader->thread);
  // no need to free loader_data, done by join
  LW6SYS_FREE (loader);
}

/**
 * lw6tsk_loader_repr
 *
 * @loader: the loader to represent.
 *
 * Creates a string which briefly describes the loader.
 *
 * Return value: a dynamically allocated pointer, must be freed.
 */
char *
lw6tsk_loader_repr (lw6tsk_loader_t * loader)
{
  char *ret = NULL;
  _lw6tsk_loader_data_t *loader_data;

  if (loader)
    {
      loader_data = (_lw6tsk_loader_data_t *) loader->data;
      ret =
	lw6sys_new_sprintf (_("%d request_number=%d stage=%d"), loader->id,
			    loader_data->request_number, loader_data->stage);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("can't generate string id for NULL loader"));
    }

  return ret;
}
