/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <math.h>

#include "../../mod-gl.h"
#include "gl-cylinder-internal.h"

#define TEST_MENU_SIZE 8

/*
 * OpenGL wizardry, to prepare view parameters.
 */
static void
prepare_view (mod_gl_utils_context_t * utils_context,
	      _mod_gl_menu_cylinder_context_t * cylinder_context)
{
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
}

static void
draw_cylinder (mod_gl_utils_context_t * utils_context,
	       _mod_gl_menu_cylinder_context_t * cylinder_context,
	       int i, int n, float ratio)
{
  _mod_gl_menu_cylinder_draw_cylinder (utils_context, cylinder_context,
				       GL_RENDER, i, n, ratio);
}

static void
draw_sphere (mod_gl_utils_context_t * utils_context,
	     _mod_gl_menu_cylinder_context_t * cylinder_context, int i, int n,
	     int sphere_i, int nb_spheres)
{
  _mod_gl_menu_cylinder_draw_sphere (utils_context, cylinder_context,
				     GL_RENDER, i, n, sphere_i, nb_spheres);
}

static SDL_Color
get_fg_color (lw6gui_look_t * look, lw6gui_menuitem_t * menuitem)
{
  SDL_Color color;

  if (!menuitem->enabled)
    {
      color =
	mod_gl_utils_color_8_to_sdl (look->style.menu_color_disabled.fg);
    }
  else if (menuitem->selected)
    {
      color =
	mod_gl_utils_color_8_to_sdl (look->style.menu_color_selected.fg);
    }
  else
    {
      color = mod_gl_utils_color_8_to_sdl (look->style.menu_color_default.fg);
    }

  return color;
}

static SDL_Color
get_bg_color (lw6gui_look_t * look, lw6gui_menuitem_t * menuitem)
{
  SDL_Color color;

  if (!menuitem->enabled)
    {
      color =
	mod_gl_utils_color_8_to_sdl (look->style.menu_color_disabled.bg);
    }
  else if (menuitem->colored)
    {
      if (menuitem->value >= 0 && menuitem->value < LW6MAP_NB_TEAM_COLORS)
	{
	  // todo, get it from config...
	  color =
	    mod_gl_utils_color_8_to_sdl (look->style.
					 team_colors[menuitem->value]);
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("invalid value %d for colored menu item"),
		      menuitem->value);
	  color = mod_gl_utils_color_8_to_sdl (LW6SYS_COLOR_8_WHITE);
	}
    }
  else if (menuitem->selected)
    {
      color =
	mod_gl_utils_color_8_to_sdl (look->style.menu_color_selected.bg);
    }
  else
    {
      color = mod_gl_utils_color_8_to_sdl (look->style.menu_color_default.bg);
    }

  return color;
}

/*
 * Draw a basic text menu item (button).
 */
static void
draw_button (mod_gl_utils_context_t * utils_context,
	     _mod_gl_menu_cylinder_context_t * cylinder_context,
	     lw6gui_look_t * look, lw6gui_menuitem_t * menuitem, int i, int n)
{
  SDL_Surface *text_surface = NULL;
  SDL_Surface *outline_surface = NULL;
  GLuint text_texture;
  char *utf8;
  SDL_Color bg_color;
  SDL_Color fg_color;
  int ret = 0;
  SDL_Rect text_rect;
  SDL_Rect outline_rect;

  outline_surface =
    mod_gl_utils_get_button_from_menucache (utils_context, look, menuitem);

  if (outline_surface == NULL)
    {
      utf8 = lw6sys_locale_to_utf8 (menuitem->label);
      if (utf8 != NULL)
	{
	  fg_color = get_fg_color (look, menuitem);
	  bg_color = get_bg_color (look, menuitem);
	  text_surface =
	    TTF_RenderUTF8_Shaded (utils_context->font_data.menu, utf8,
				   fg_color, bg_color);
	  if (text_surface != NULL)
	    {
	      utils_context->surface_counter.new_counter++;
	      outline_surface =
		mod_gl_utils_create_surface (utils_context,
					     text_surface->w +
					     2 *
					     utils_context->const_data.
					     menu_font_size,
					     MOD_GL_UTILS_MENU_TEXTURE_H);
	      if (outline_surface != NULL)
		{
		  text_rect.x = 0;
		  text_rect.y = 0;
		  text_rect.w = text_surface->w;
		  text_rect.h = text_surface->h;
		  outline_rect.x = (outline_surface->w - text_surface->w) / 2;
		  outline_rect.y = (outline_surface->h - text_surface->h) / 2;
		  outline_rect.w = outline_surface->w;
		  outline_rect.h = outline_surface->h;

		  mod_gl_utils_clear_surface_with_color (outline_surface,
							 bg_color);
		  SDL_BlitSurface (text_surface, &text_rect, outline_surface,
				   &outline_rect);

		  if (!mod_gl_utils_store_button_in_menucache
		      (utils_context, look, menuitem, outline_surface))
		    {
		      /*
		       * Storing stuff in menucache should always work, but in
		       * case it wouldn't work, we just cancel everything,
		       * it's better not to display something just one time
		       * than risk a memory leak. Besides doing this will
		       * show the user that "something" is going wrong, which
		       * is an order of magnitude better than an obscure invisible
		       * memory leak...
		       */
		      mod_gl_utils_delete_surface (utils_context,
						   outline_surface);
		      outline_surface = NULL;
		    }
		}
	      mod_gl_utils_delete_surface (utils_context, text_surface);
	    }
	  LW6SYS_FREE (utf8);
	}
    }

  if (outline_surface != NULL)
    {
      text_texture =
	mod_gl_utils_surface2texture_xywh_using_cache (utils_context,
						       outline_surface,
						       (MOD_GL_UTILS_MENU_TEXTURE_W
							-
							outline_surface->w) /
						       2,
						       (MOD_GL_UTILS_MENU_TEXTURE_H
							-
							outline_surface->h) /
						       2,
						       MOD_GL_UTILS_MENU_TEXTURE_W,
						       MOD_GL_UTILS_MENU_TEXTURE_H,
						       menuitem->label,
						       lw6gui_menuitem_checksum
						       (menuitem, look));
      if (text_texture)
	{
	  glEnable (GL_TEXTURE_2D);
	  glColor3f (1.0f, 1.0f, 1.0f);
	  glBindTexture (GL_TEXTURE_2D, text_texture);

	  glMatrixMode (GL_TEXTURE);
	  glPushMatrix ();
	  glLoadIdentity ();
	  glScalef (1.0f, -1.0f, 1.0f);
	  glTranslatef (0.0f,
			-0.25f + cylinder_context->const_data.rotate_offset,
			0.0f);
	  glRotatef (-90.0f, 0.0f, 0.0f, 1.0f);

	  draw_cylinder (utils_context, cylinder_context, i, n,
			 ((float) MOD_GL_UTILS_MENU_TEXTURE_W) /
			 ((float) outline_surface->w));

	  glMatrixMode (GL_TEXTURE);
	  glPopMatrix ();

	  ret = 1;

	  // _mod_gl_schedule_delete_texture(context,text_texture);
	}
    }

  /*
   * Note: it's important *NOT* to delete outline_surface.
   * Indeed it's kept in the menucache structure. It will
   * be free when needed, wizardry inside menucache.c 8-)
   */

  if (!ret)
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to draw button \"%s\""), menuitem->label);
    }
}

static void
draw_spheres (mod_gl_utils_context_t * utils_context,
	      _mod_gl_menu_cylinder_context_t * cylinder_context, int i,
	      int n, int blink_state, int nb_spheres)
{
  int sphere_i;

  glEnable (GL_TEXTURE_2D);
  glColor3f (1.0f, 1.0f, 1.0f);

  if (blink_state)
    {
      glBindTexture (GL_TEXTURE_2D, utils_context->textures_1x1.menu_fg);
    }
  else
    {
      glBindTexture (GL_TEXTURE_2D, utils_context->textures_1x1.menu_bg);
    }

  for (sphere_i = 0; sphere_i < nb_spheres; ++sphere_i)
    {
      draw_sphere (utils_context, cylinder_context, i, n, sphere_i,
		   nb_spheres);
    }
}

/*
 * Display a menu.
 */
void
_mod_gl_menu_cylinder_display (mod_gl_utils_context_t * utils_context,
			       _mod_gl_menu_cylinder_context_t *
			       cylinder_context, lw6gui_look_t * look,
			       lw6gui_menu_t * menu)
{
  int i, j, n;
  int blink_state;
  lw6gui_menuitem_t *menuitem;

  mod_gl_utils_set_render_mode_3d_menu (utils_context);
  mod_gl_utils_texture_1x1_update (utils_context, look);

  prepare_view (utils_context, cylinder_context);

  lw6gui_menu_update_display_range (menu,
				    cylinder_context->const_data.
				    max_displayed_items);

  blink_state =
    mod_gl_utils_get_ticks (utils_context) /
    (cylinder_context->const_data.sphere_blink_period / 2) % 2;
  n = menu->nb_items_displayed + 2;
  if (menu->first_item_displayed > 0)
    {
      draw_spheres (utils_context, cylinder_context, 0, n,
		    blink_state, cylinder_context->const_data.nb_spheres);
    }
  for (i = 0; i < menu->nb_items_displayed; ++i)
    {
      j = i + menu->first_item_displayed;
      menuitem = menu->items[j];
      draw_button (utils_context, cylinder_context, look, menuitem, i + 1, n);
    }
  if (menu->first_item_displayed + menu->nb_items_displayed < menu->nb_items)
    {
      draw_spheres (utils_context, cylinder_context, n - 1, n,
		    blink_state ? 0 : 1,
		    cylinder_context->const_data.nb_spheres);
    }
}

void
mod_gl_menu_cylinder_display (mod_gl_utils_context_t * utils_context,
			      void *cylinder_context, lw6gui_look_t * look,
			      lw6gui_menu_t * menu)
{
  _mod_gl_menu_cylinder_display (utils_context,
				 (_mod_gl_menu_cylinder_context_t *)
				 cylinder_context, look, menu);
}
