/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libpng12/png.h>

#include "ldr.h"
#include "ldr-internal.h"

static int
read_png (lw6map_depth_t * depth, char *map_dot_png,
	  lw6map_options_t * options, lw6sys_wh_t * display_shape,
	  float target_ratio, float fighter_scale)
{
  int ret = 0;
  _lw6ldr_image_bw_t image;

  memset (&image, 0, sizeof (_lw6ldr_image_bw_t));

  if (_lw6ldr_bw_read (&image, map_dot_png))
    {
      lw6map_resampler_t resampler;
      lw6map_resampler_init (&resampler, options, &(image.shape),
			     display_shape, target_ratio, fighter_scale);

      depth->shape = resampler.target_shape;
      depth->data =
	(unsigned char *) LW6SYS_MALLOC (depth->shape.w * depth->shape.h *
					 sizeof (unsigned char *));

      if (depth->data)
	{
	  int row, col, row2, col2;
	  unsigned int value;

	  for (row = 0; row < depth->shape.h; ++row)
	    {
	      for (col = 0; col < depth->shape.w; ++col)
		{
		  lw6map_resampler_target2source (&resampler, &col2, &row2,
						  col, row);

		  value = image.data[row2][col2 * image.step];
		  /*
		   * For now we do not handle depth the full way. There's
		   * a big TODO here, one could read in another file (say
		   * an XML file) a parameter that would state "maximum depth
		   * is 5" and maybe even the various thresholds for the
		   * different levels. For now using depth 0 or 1 is just like
		   * in Liquid War 5, we have flat levels, will be fine for
		   * a some time anyway.
		   */
		  value = (value > 127) ? 1 : 0;
		  lw6map_depth_set (depth, col, row, value);
		}
	    }

	  ret = 1;
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_ERROR,
		      _("unable to allocate memory for depth"));
	}

      _lw6ldr_bw_clear (&image);
    }

  if (!ret)
    {
      lw6map_depth_clear (depth);
    }

  return ret;
}

/*
 * Read a depth map. Pointer to depth must be valid,
 * it's modified in-place.
 */
int
lw6ldr_depth_read (lw6map_depth_t * depth, char *dirname,
		   lw6map_options_t * options, lw6sys_wh_t * display_shape,
		   float ratio, float fighter_scale)
{
  int ret = 0;
  char *map_dot_png;

  lw6map_depth_clear (depth);

  map_dot_png = lw6sys_path_concat (dirname, _LW6LDR_FILE_MAP_PNG);
  if (map_dot_png)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("reading \"%s\""), map_dot_png);

      ret =
	read_png (depth, map_dot_png, options, display_shape, ratio,
		  fighter_scale);
      if (ret)
	{
	  if (options->x_polarity == 0)
	    {
	      lw6map_depth_block_left_right (depth);
	    }
	  if (options->y_polarity == 0)
	    {
	      lw6map_depth_block_up_down (depth);
	    }
	  ret = lw6map_depth_check_and_fix_holes (depth);
	}

      LW6SYS_FREE (map_dot_png);
    }

  return ret;
}
