/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "ldr.h"
#include "../cfg/cfg.h"
#include "ldr-internal.h"

typedef struct options_update_data_s
{
  lw6map_options_t *options;
  lw6sys_assoc_t *values;
}
options_update_data_t;

/*
 * One could wonder why this is not in the "opt" module. Basically,
 * doing so would tie the "opt" module to some XML library, and since
 * the "ker" module requires it, we would end up with "ker" requiring
 * an XML library, which is certainly not want we want. Actually
 * "opt" is only here to avoid "ker" depending directly on "map" and
 * its dependencies.
 */

static void
read_callback (void *callback_data, char *element, char *key, char *value)
{
  lw6map_options_t *options_data;
  int int_value;

  options_data = (lw6map_options_t *) callback_data;

  if (!strcmp (element, LW6CFG_XML_INT))
    {
      lw6cfg_read_xml_int (key, value, key, &int_value);
      lw6map_options_set_int (options_data, key, int_value);
    }

  if (!strcmp (element, LW6CFG_XML_BOOL))
    {
      lw6cfg_read_xml_bool (key, value, key, &int_value);
      lw6map_options_set_bool (options_data, key, int_value);
    }

  if (!strcmp (element, LW6CFG_XML_FLOAT))
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("map options should not use floats, XML entry %s=%s is not correct"),
		  key, value);
    }

  if (!strcmp (element, LW6CFG_XML_STRING))
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("map options should not use strings, XML entry %s=%s is not correct"),
		  key, value);
    }
}

/**
 * lw6ldr_options_read
 *
 * @param: the options struct to fill with values (read/write parameter)
 * @dirname: the directory of the map
 *
 * Read the options (options.xml) of a map. Pointer to options must be valid,
 * and values already initialized, either zeroed or filled in with defaults
 * or custom values.
 *
 * Return value: 1 if success, 0 if failed.
 */
int
lw6ldr_options_read (lw6map_options_t * options, char *dirname)
{
  int ret = 0;
  char *buf = NULL;

  buf = lw6sys_path_concat (dirname, _LW6LDR_FILE_OPTIONS_XML);

  if (buf)
    {
      if (lw6sys_file_exists (buf))
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("reading options \"%s\""), buf);
	  ret =
	    lw6cfg_read_key_value_xml_file (buf, read_callback,
					    (void *) options);
	}
      else
	{
	  // using defaults
	  ret = 1;
	}

      LW6SYS_FREE (buf);
    }

  if (!ret)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("unable to read map options"));
    }

  return ret;
}

static void
options_update_callback (void *func_data, void *data)
{
  options_update_data_t *update_data;
  char *key;
  char *value;
  char *about;
  char *element;
  LW6HLP_TYPE type = LW6HLP_TYPE_VOID;

  key = (char *) data;
  update_data = (options_update_data_t *) func_data;

  if (lw6sys_assoc_has_key (update_data->values, key))
    {
      about = lw6hlp_about (&type, key);
      value = lw6sys_assoc_get (update_data->values, key);
      element = lw6cfg_xml_element (type);
      read_callback (update_data->options, element, key, value);
    }
}

/**
 * lw6ldr_options_update
 *
 * @options: the options struct to fill with values (read/write parameter)
 * @values: an assoc containing strings with the new values
 *
 * Overrides options with values. Pointer to options must be valid,
 * and values already initialized, either zeroed or filled in with defaults
 * or custom values. Not all parameters need be defined in values. It can
 * even be NULL. The idea is just that if something is defined in values,
 * it will override the existing options.
 *
 * Return value: 1 if success, 0 if failed.
 */
int
lw6ldr_options_update (lw6map_options_t * options, lw6sys_assoc_t * values)
{
  int ret = 0;
  lw6sys_list_t *list;
  options_update_data_t data;

  if (values)
    {
      list = lw6hlp_list_map_options ();
      if (list)
	{
	  data.options = options;
	  data.values = values;
	  lw6sys_list_map (list, options_update_callback, &data);
	  lw6sys_list_free (list);
	  ret = 1;
	}
    }
  else
    {
      // NULL values is allowed
      ret = 1;
    }

  return ret;
}
