/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "liquidwar6.h"

/*
 * Map smob
 */
static SCM
mark_map (SCM map)
{
  return SCM_BOOL_F;
}

static size_t
free_map (SCM map)
{
  char *repr = NULL;
  lw6_map_smob_t *map_smob = (lw6_map_smob_t *) SCM_SMOB_DATA (map);

  repr = lw6map_repr (map_smob->c_map);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.map_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated map is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("freeing map smob which isn't in global assoc"));
      lw6_free_map_smob (map_smob);
    }

  return 0;
}

static int
print_map (SCM map, SCM port, scm_print_state * pstate)
{
  lw6map_level_t *c_map = lw6_scm_to_map (map);
  char *repr = NULL;

  repr = lw6map_repr (c_map);

  scm_puts ("#<lw6map-map ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

SCM
lw6_make_scm_map (lw6map_level_t * c_map)
{
  char *repr = NULL;
  lw6_map_smob_t *map_smob = NULL;

  /*
   * Calling the garbage collector, the idea is that when we
   * create a map object, we're doing something rare, and
   * we're not playing, so there's no harm in garbage collecting.
   * Additionnally, this will free memory for new map processing.
   * Ideally we would garbage collect just after affecting the
   * new map, or just before allocating memory for it. Putting
   * a garbage collection here does not harm anyway.
   */
  scm_gc ();

  map_smob = (lw6_map_smob_t *) LW6SYS_CALLOC (sizeof (lw6_map_smob_t));
  if (map_smob)
    {
      map_smob->c_map = c_map;
      repr = lw6map_repr (c_map);
      if (repr)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating map smob \"%s\""), repr);
	  LW6_MUTEX_LOCK;
	  lw6sys_assoc_set (&lw6_global.map_smobs, repr, (void *) map_smob);
	  LW6_MUTEX_UNLOCK;
	  LW6SYS_FREE (repr);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to create map smob, expect trouble"));
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.map, map_smob);
}

lw6map_level_t *
lw6_scm_to_map (SCM map)
{
  lw6map_level_t *c_map;

  c_map = ((lw6_map_smob_t *) SCM_SMOB_DATA (map))->c_map;

  return c_map;
}

void
lw6_free_map_smob (lw6_map_smob_t * map_smob)
{
  char *repr = NULL;

  repr = lw6map_repr (map_smob->c_map);
  if (repr)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("freeing map smob \"%s\""), repr);
      LW6SYS_FREE (repr);
    }

  lw6map_free (map_smob->c_map);
  LW6SYS_FREE (map_smob);
}

/*
 * Menu smob
 */
static SCM
mark_menu (SCM menu)
{
  return SCM_BOOL_F;
}

static size_t
free_menu (SCM menu)
{
  char *repr = NULL;
  lw6_menu_smob_t *menu_smob = (lw6_menu_smob_t *) SCM_SMOB_DATA (menu);

  repr = lw6gui_menu_repr (menu_smob->c_menu);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.menu_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated menu is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("freeing menu smob which isn't in global assoc"));
      lw6_free_menu_smob (menu_smob);
    }

  return 0;
}

static int
print_menu (SCM menu, SCM port, scm_print_state * pstate)
{
  lw6gui_menu_t *c_menu = lw6_scm_to_menu (menu);
  char *repr = NULL;

  repr = lw6gui_menu_repr (c_menu);

  scm_puts ("#<lw6gui-menu ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

SCM
lw6_make_scm_menu (lw6gui_menu_t * c_menu)
{
  char *repr = NULL;
  lw6_menu_smob_t *menu_smob = NULL;

  scm_gc ();

  menu_smob = (lw6_menu_smob_t *) LW6SYS_CALLOC (sizeof (lw6_menu_smob_t));
  if (menu_smob)
    {
      menu_smob->c_menu = c_menu;
      repr = lw6gui_menu_repr (c_menu);
      if (repr)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating menu smob \"%s\""), repr);
	  LW6_MUTEX_LOCK;
	  lw6sys_assoc_set (&lw6_global.menu_smobs, repr, (void *) menu_smob);
	  LW6_MUTEX_UNLOCK;
	  LW6SYS_FREE (repr);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to create menu smob, expect trouble"));
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.menu, menu_smob);
}

lw6gui_menu_t *
lw6_scm_to_menu (SCM menu)
{
  lw6gui_menu_t *c_menu;

  c_menu = ((lw6_menu_smob_t *) SCM_SMOB_DATA (menu))->c_menu;

  return c_menu;
}

void
lw6_free_menu_smob (lw6_menu_smob_t * menu_smob)
{
  char *repr = NULL;

  repr = lw6gui_menu_repr (menu_smob->c_menu);
  if (repr)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("freeing menu smob \"%s\""), repr);
      LW6SYS_FREE (repr);
    }

  lw6gui_menu_free (menu_smob->c_menu);
  LW6SYS_FREE (menu_smob);
}

/*
 * Game struct smob
 */
static SCM
mark_game_struct (SCM game_struct)
{
  lw6_game_struct_smob_t *game_struct_smob =
    (lw6_game_struct_smob_t *) SCM_SMOB_DATA (game_struct);

  scm_gc_mark (game_struct_smob->map);

  return SCM_BOOL_F;
}

static size_t
free_game_struct (SCM game_struct)
{
  char *repr = NULL;
  lw6_game_struct_smob_t *game_struct_smob =
    (lw6_game_struct_smob_t *) SCM_SMOB_DATA (game_struct);

  repr = lw6ker_game_struct_repr (game_struct_smob->c_game_struct);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.game_struct_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated game_struct is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("freeing game_struct smob which isn't in global assoc"));
      lw6_free_game_struct_smob (game_struct_smob);
    }

  return 0;
}

static int
print_game_struct (SCM game_struct, SCM port, scm_print_state * pstate)
{
  lw6ker_game_struct_t *c_game_struct = lw6_scm_to_game_struct (game_struct);
  char *repr = NULL;

  repr = lw6ker_game_struct_repr (c_game_struct);

  scm_puts ("#<lw6ker-game-struct ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

SCM
lw6_make_scm_game_struct (lw6ker_game_struct_t * c_game_struct, SCM map)
{
  char *repr = NULL;
  lw6_game_struct_smob_t *game_struct_smob = NULL;

  scm_gc ();

  game_struct_smob =
    (lw6_game_struct_smob_t *)
    LW6SYS_CALLOC (sizeof (lw6_game_struct_smob_t));
  if (game_struct_smob)
    {
      game_struct_smob->c_game_struct = c_game_struct;
      game_struct_smob->map = map;
      repr = lw6ker_game_struct_repr (c_game_struct);
      if (repr)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating game struct smob \"%s\""),
		      repr);
	  LW6_MUTEX_LOCK;
	  lw6sys_assoc_set (&lw6_global.game_struct_smobs, repr,
			    (void *) game_struct_smob);
	  LW6_MUTEX_UNLOCK;
	  LW6SYS_FREE (repr);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to create game struct smob, expect trouble"));
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.game_struct, game_struct_smob);
}

lw6ker_game_struct_t *
lw6_scm_to_game_struct (SCM game_struct)
{
  lw6ker_game_struct_t *c_game_struct;

  c_game_struct =
    ((lw6_game_struct_smob_t *) SCM_SMOB_DATA (game_struct))->c_game_struct;

  return c_game_struct;
}

void
lw6_free_game_struct_smob (lw6_game_struct_smob_t * game_struct_smob)
{
  char *repr = NULL;

  repr = lw6ker_game_struct_repr (game_struct_smob->c_game_struct);
  if (repr)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("freeing game struct smob \"%s\""),
		  repr);
      LW6SYS_FREE (repr);
    }

  lw6ker_game_struct_free (game_struct_smob->c_game_struct);
  LW6SYS_FREE (game_struct_smob);
}

/*
 * Game state smob
 */
static SCM
mark_game_state (SCM game_state)
{
  lw6_game_state_smob_t *game_state_smob =
    (lw6_game_state_smob_t *) SCM_SMOB_DATA (game_state);

  scm_gc_mark (game_state_smob->game_struct);

  return SCM_BOOL_F;
}

static size_t
free_game_state (SCM game_state)
{
  char *repr = NULL;
  lw6_game_state_smob_t *game_state_smob =
    (lw6_game_state_smob_t *) SCM_SMOB_DATA (game_state);

  repr = lw6ker_game_state_repr (game_state_smob->c_game_state);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.game_state_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated game_state is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("freeing game_state smob which isn't in global assoc"));
      lw6_free_game_state_smob (game_state_smob);
    }

  return 0;
}

static int
print_game_state (SCM game_state, SCM port, scm_print_state * pstate)
{
  lw6ker_game_state_t *c_game_state = lw6_scm_to_game_state (game_state);
  char *repr = NULL;

  repr = lw6ker_game_state_repr (c_game_state);

  scm_puts ("#<lw6ker-game-state ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

SCM
lw6_make_scm_game_state (lw6ker_game_state_t * c_game_state, SCM game_struct)
{
  char *repr = NULL;
  lw6_game_state_smob_t *game_state_smob = NULL;

  scm_gc ();

  game_state_smob =
    (lw6_game_state_smob_t *) LW6SYS_CALLOC (sizeof (lw6_game_state_smob_t));
  if (game_state_smob)
    {
      game_state_smob->c_game_state = c_game_state;
      game_state_smob->game_struct = game_struct;
      repr = lw6ker_game_state_repr (c_game_state);
      if (repr)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating game state smob \"%s\""),
		      repr);
	  LW6_MUTEX_LOCK;
	  lw6sys_assoc_set (&lw6_global.game_state_smobs, repr,
			    (void *) game_state_smob);
	  LW6_MUTEX_UNLOCK;
	  LW6SYS_FREE (repr);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to create game state smob, expect trouble"));
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.game_state, game_state_smob);
}

lw6ker_game_state_t *
lw6_scm_to_game_state (SCM game_state)
{
  lw6ker_game_state_t *c_game_state;

  c_game_state =
    ((lw6_game_state_smob_t *) SCM_SMOB_DATA (game_state))->c_game_state;

  return c_game_state;
}

void
lw6_free_game_state_smob (lw6_game_state_smob_t * game_state_smob)
{
  char *repr = NULL;

  repr = lw6ker_game_state_repr (game_state_smob->c_game_state);
  if (repr)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("freeing game state smob \"%s\""), repr);
      LW6SYS_FREE (repr);
    }

  lw6ker_game_state_free (game_state_smob->c_game_state);
  LW6SYS_FREE (game_state_smob);
}

/*
 * Game look smob
 */
static SCM
mark_look (SCM look)
{
  return SCM_BOOL_F;
}

static size_t
free_look (SCM look)
{
  char *repr = NULL;
  lw6_look_smob_t *look_smob = (lw6_look_smob_t *) SCM_SMOB_DATA (look);

  repr = lw6gui_look_repr (look_smob->c_look);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.look_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated look is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("freeing look smob which isn't in global assoc"));
      lw6_free_look_smob (look_smob);
    }

  return 0;
}

static int
print_look (SCM look, SCM port, scm_print_state * pstate)
{
  lw6gui_look_t *c_look = lw6_scm_to_look (look);
  char *repr = NULL;

  repr = lw6gui_look_repr (c_look);

  scm_puts ("#<lw6gui-look ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

SCM
lw6_make_scm_look (lw6gui_look_t * c_look)
{
  char *repr = NULL;
  lw6_look_smob_t *look_smob = NULL;

  scm_gc ();

  look_smob = (lw6_look_smob_t *) LW6SYS_CALLOC (sizeof (lw6_look_smob_t));
  if (look_smob)
    {
      look_smob->c_look = c_look;
      repr = lw6gui_look_repr (c_look);
      if (repr)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating look smob \"%s\""), repr);
	  LW6_MUTEX_LOCK;
	  lw6sys_assoc_set (&lw6_global.look_smobs, repr, (void *) look_smob);
	  LW6_MUTEX_UNLOCK;
	  LW6SYS_FREE (repr);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to create look smob, expect trouble"));
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.look, look_smob);
}

lw6gui_look_t *
lw6_scm_to_look (SCM look)
{
  lw6gui_look_t *c_look;

  c_look = ((lw6_look_smob_t *) SCM_SMOB_DATA (look))->c_look;

  return c_look;
}

void
lw6_free_look_smob (lw6_look_smob_t * look_smob)
{
  char *repr = NULL;

  repr = lw6gui_look_repr (look_smob->c_look);
  if (repr)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("freeing look smob \"%s\""), repr);
      LW6SYS_FREE (repr);
    }

  lw6gui_look_free (look_smob->c_look);
  LW6SYS_FREE (look_smob);
}

/*
 * Loader smob
 */
static SCM
mark_loader (SCM loader)
{
  return SCM_BOOL_F;
}

static size_t
free_loader (SCM loader)
{
  char *repr = NULL;
  lw6_loader_smob_t *loader_smob =
    (lw6_loader_smob_t *) SCM_SMOB_DATA (loader);

  repr = lw6tsk_loader_repr (loader_smob->c_loader);
  if (repr)
    {
      LW6_MUTEX_LOCK;
      lw6sys_assoc_unset (lw6_global.loader_smobs, repr);
      LW6_MUTEX_UNLOCK;
      LW6SYS_FREE (repr);
    }
  else
    {
      /*
       * Normally we should never get there, for any smob
       * allocated loader is in the global assoc. But well,
       * who knows...
       */
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("freeing loader smob which isn't in global assoc"));
      lw6_free_loader_smob (loader_smob);
    }

  return 0;
}

static int
print_loader (SCM loader, SCM port, scm_print_state * pstate)
{
  lw6tsk_loader_t *c_loader = lw6_scm_to_loader (loader);
  char *repr = NULL;

  repr = lw6tsk_loader_repr (c_loader);

  scm_puts ("#<lw6tsk-loader ", port);
  if (repr)
    {
      scm_display (scm_makfrom0str (repr), port);
      LW6SYS_FREE (repr);
    }
  scm_puts (">", port);

  return 1;
}

SCM
lw6_make_scm_loader (lw6tsk_loader_t * c_loader)
{
  char *repr = NULL;
  lw6_loader_smob_t *loader_smob = NULL;

  scm_gc ();

  loader_smob =
    (lw6_loader_smob_t *) LW6SYS_CALLOC (sizeof (lw6_loader_smob_t));
  if (loader_smob)
    {
      loader_smob->c_loader = c_loader;
      repr = lw6tsk_loader_repr (c_loader);
      if (repr)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating loader smob \"%s\""),
		      repr);
	  LW6_MUTEX_LOCK;
	  lw6sys_assoc_set (&lw6_global.loader_smobs, repr,
			    (void *) loader_smob);
	  LW6_MUTEX_UNLOCK;
	  LW6SYS_FREE (repr);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to create loader smob, expect trouble"));
    }

  SCM_RETURN_NEWSMOB (lw6_global.smob_types.loader, loader_smob);
}

lw6tsk_loader_t *
lw6_scm_to_loader (SCM loader)
{
  lw6tsk_loader_t *c_loader;

  c_loader = ((lw6_loader_smob_t *) SCM_SMOB_DATA (loader))->c_loader;

  return c_loader;
}

void
lw6_free_loader_smob (lw6_loader_smob_t * loader_smob)
{
  char *repr = NULL;

  repr = lw6tsk_loader_repr (loader_smob->c_loader);
  if (repr)
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("freeing loader smob \"%s\""), repr);
      LW6SYS_FREE (repr);
    }

  lw6tsk_loader_free (loader_smob->c_loader);
  LW6SYS_FREE (loader_smob);
}

/*
 * Register all the smobs, that is new types.
 */
int
lw6_register_smobs ()
{
  int ret = 1;

  lw6_global.smob_types.map =
    scm_make_smob_type ("lw6map-map", sizeof (lw6map_level_t));
  scm_set_smob_mark (lw6_global.smob_types.map, mark_map);
  scm_set_smob_free (lw6_global.smob_types.map, free_map);
  scm_set_smob_print (lw6_global.smob_types.map, print_map);

  lw6_global.smob_types.menu =
    scm_make_smob_type ("lw6gui-menu", sizeof (lw6gui_menu_t));
  scm_set_smob_mark (lw6_global.smob_types.menu, mark_menu);
  scm_set_smob_free (lw6_global.smob_types.menu, free_menu);
  scm_set_smob_print (lw6_global.smob_types.menu, print_menu);

  lw6_global.smob_types.game_struct =
    scm_make_smob_type ("lw6ker-game-struct", sizeof (lw6ker_game_struct_t));
  scm_set_smob_mark (lw6_global.smob_types.game_struct, mark_game_struct);
  scm_set_smob_free (lw6_global.smob_types.game_struct, free_game_struct);
  scm_set_smob_print (lw6_global.smob_types.game_struct, print_game_struct);

  lw6_global.smob_types.game_state =
    scm_make_smob_type ("lw6ker-game-state", sizeof (lw6ker_game_state_t));
  scm_set_smob_mark (lw6_global.smob_types.game_state, mark_game_state);
  scm_set_smob_free (lw6_global.smob_types.game_state, free_game_state);
  scm_set_smob_print (lw6_global.smob_types.game_state, print_game_state);

  lw6_global.smob_types.look =
    scm_make_smob_type ("lw6gui-look", sizeof (lw6gui_look_t));
  scm_set_smob_mark (lw6_global.smob_types.look, mark_look);
  scm_set_smob_free (lw6_global.smob_types.look, free_look);
  scm_set_smob_print (lw6_global.smob_types.look, print_look);

  lw6_global.smob_types.loader =
    scm_make_smob_type ("lw6tsk-loader", sizeof (lw6tsk_loader_t));
  scm_set_smob_mark (lw6_global.smob_types.loader, mark_loader);
  scm_set_smob_free (lw6_global.smob_types.loader, free_loader);
  scm_set_smob_print (lw6_global.smob_types.loader, print_loader);

  return ret;
}
