/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <unistd.h>

#include "sys.h"
#include "sys-internal.h"

#define BAZOOKA_ERASER_MALLOC 'M'
#define BAZOOKA_ERASER_REALLOC_1 'E'
#define BAZOOKA_ERASER_REALLOC_2 'R'
#define BAZOOKA_ERASER_FREE 'F'

static int bazooka_size = 0;
static _lw6sys_bazooka_t *bazooka_data = NULL;
static void *bazooka_mutex = NULL;
static int bazooka_malloc_counter = 0;
static int bazooka_free_counter = 0;
static int bazooka_eraser = 1;

static void
bazooka_lock ()
{
  if (bazooka_mutex)
    {
      lw6sys_mutex_lock (bazooka_mutex);
    }
}

static void
bazooka_unlock ()
{
  if (bazooka_mutex)
    {
      lw6sys_mutex_unlock (bazooka_mutex);
    }
}

static void
bazooka_destroy_mutex ()
{
  void *tmp_mutex;

  tmp_mutex = bazooka_mutex;
  bazooka_mutex = NULL;
  if (tmp_mutex)
    {
      lw6sys_mutex_destroy (tmp_mutex);
    }
}

static void
bazooka_free_data ()
{
  bazooka_size = 0;
  if (bazooka_data)
    {
      free (bazooka_data);
    }
  bazooka_data = NULL;
}

/**
 * lw6sys_default_memory_bazooka
 *
 * Will set up a default memory bazooka, a slow yet convenient
 * tool to track down and hopefully kill memory leaks.
 * Named bazooka after a night wasted to track down an unfoundable
 * leak... BAZOOOOOOKA!!!
 *
 * Return value: 1 if success, 0 if failed.
 */
int
lw6sys_default_memory_bazooka ()
{
  int ret = 0;

  ret = lw6sys_set_memory_bazooka_size (_LW6SYS_BAZOOKA_DEFAULT_SIZE);

  return ret;
}

/**
 * lw6sys_clear_memory_bazooka
 *
 * Clears the memory bazooka.
 *
 * Return value: none.
 */
void
lw6sys_clear_memory_bazooka ()
{
  lw6sys_set_memory_bazooka_size (0);
}

/**
 * lw6sys_set_memory_bazooka_size
 *
 * @size: number of items (calls to malloc) to keep
 *
 * Resizes, the memory bazooka. What's this? It's an inelegant yet 
 * efficient tool to track down memory leak. Memory bazooka will keep
 * track of every call to malloc, keeping a trace of what has been
 * malloced, where it has been called (from which file, which line), how
 * much memory was allocated, it will even show you what's at the
 * address in a 0-terminated string-friendly fashion. Of course this
 * slows down the program, so in production, you might set this to 0,
 * but for debugging, a million bazooka is worth the megabytes and
 * CPU cycles it wastes.
 *
 * Return value: 1 if success, 0 if failure.
 */
int
lw6sys_set_memory_bazooka_size (int size)
{
  int ret = 0;
  int i;

  if (size == bazooka_size)
    {
      // nothing to do
      ret = 1;
    }
  else
    {
      if (size > 0)
	{
	  bazooka_lock ();

	  if (bazooka_data)
	    {
	      bazooka_data =
		(_lw6sys_bazooka_t *) realloc (bazooka_data,
					       size *
					       sizeof (_lw6sys_bazooka_t));
	      if (bazooka_data)
		{
		  // Performance killer, but works...
		  for (i = bazooka_size; i < size; ++i)
		    {
		      memset (&(bazooka_data[i]), 0,
			      sizeof (_lw6sys_bazooka_t));
		    }
		  bazooka_size = size;
		  ret = 1;
		}
	      else
		{
		  bazooka_size = 0;
		}
	    }
	  else
	    {
	      bazooka_data =
		(_lw6sys_bazooka_t *) calloc (size,
					      sizeof (_lw6sys_bazooka_t));
	      if (bazooka_data)
		{
		  bazooka_size = size;
		  ret = 1;
		}
	      else
		{
		  bazooka_size = 0;
		}
	    }

	  if (bazooka_mutex)
	    {
	      bazooka_unlock ();
	    }
	  else
	    {
	      /*
	       * It's important to create the mutex now only, for mutex
	       * creation calls memory allocation routines, and could
	       * therefore wreck tracking.
	       */
	      bazooka_mutex = lw6sys_mutex_create ();
	    }
	}
      else
	{
	  bazooka_destroy_mutex ();
	  bazooka_free_data ();

	  ret = 1;
	}
    }

  return ret;
}

/**
 * lw6sys_get_memory_bazooka_size
 *
 * The companion of @lw6sys_set_memory_bazooka_size. This function
 * will return how many calls to malloc can be traced. A return
 * value of 0 indicates that feature is disabled.
 *
 * Return value: size of the bazooka array.
 */
int
lw6sys_get_memory_bazooka_size ()
{
  int ret = 0;

  if (bazooka_data)
    {
      ret = bazooka_size;
    }

  return ret;
}

/**
 * lw6sys_set_memory_bazooka_eraser
 *
 * @state: the state of the eraser
 *
 * Sets the memory bazooka eraser state. Note that to really work,
 * it requires the memory bazooka to be "big enough".
 *
 * Return value: 1 if activated, 0 if not. Note that the main reason
 *   for it not to be activated is if the memory bazooka has zero size.
 */
int
lw6sys_set_memory_bazooka_eraser (int state)
{
  bazooka_eraser = state ? 1 : 0;

  return lw6sys_get_memory_bazooka_eraser ();
}

int
lw6sys_get_memory_bazooka_eraser ()
{
  return (bazooka_eraser && bazooka_data && bazooka_size > 0);
}

static void
bazooka_register_malloc (char *ptr, int size, char *file, int line, int erase)
{
  if (bazooka_data)
    {
      bazooka_lock ();

      {
	int i;
	char *file_only;

	bazooka_malloc_counter++;

	if (erase)
	  {
	    memset (ptr, BAZOOKA_ERASER_MALLOC, size);
	  }

	for (i = 0; i < bazooka_size; ++i)
	  {
	    if (bazooka_data[i].ptr == NULL)
	      {
		memset (&(bazooka_data[i]), 0, sizeof (_lw6sys_bazooka_t));
		bazooka_data[i].ptr = ptr;
		bazooka_data[i].size = size;
		file_only = strrchr (file, '/');
		if (file_only && *file_only)
		  {
		    file_only++;
		  }
		else
		  {
		    file_only = file;
		  }
		strncpy (bazooka_data[i].file, file_only,
			 _LW6SYS_BAZOOKA_FILE_SIZE - 1);
		bazooka_data[i].line = line;
		bazooka_data[i].timestamp = lw6sys_timestamp ();
		break;		// important to leave loop, else serious perfomance problem
	      }
	  }
	if (i == bazooka_size)
	  {
	    // OK, we're full, big cleanup...
	    memset (bazooka_data, 0,
		    bazooka_size * sizeof (_lw6sys_bazooka_t));
	  }
      }

      bazooka_unlock ();
    }
}

void
_lw6sys_bazooka_register_malloc (char *ptr, int size, char *file, int line)
{
  bazooka_register_malloc (ptr, size, file, line, bazooka_eraser);
}

void
_lw6sys_bazooka_register_calloc (char *ptr, int size, char *file, int line)
{
  bazooka_register_malloc (ptr, size, file, line, 0);
}

void
_lw6sys_bazooka_register_realloc_1 (char *ptr, int size, char *file, int line)
{
  if (bazooka_data)
    {
      bazooka_lock ();

      {
	int i;

	for (i = 0; i < bazooka_size; ++i)
	  {
	    if (bazooka_data[i].ptr == ptr)
	      {
		if (bazooka_eraser && bazooka_data[i].size > size)
		  {
		    memset (bazooka_data[i].ptr + size,
			    BAZOOKA_ERASER_REALLOC_1,
			    bazooka_data[i].size - size);
		  }
		break;		// important to leave loop, else serious perfomance problem
	      }
	  }
      }

      bazooka_unlock ();
    }
}

void
_lw6sys_bazooka_register_realloc_2 (char *ptr, char *ptr2, int size,
				    char *file, int line)
{
  if (bazooka_data)
    {
      bazooka_lock ();

      {
	int i;
	char *file_only;

	for (i = 0; i < bazooka_size; ++i)
	  {
	    if (bazooka_data[i].ptr == ptr)
	      {
		if (bazooka_eraser && bazooka_data[i].size < size)
		  {
		    memset (ptr2 + bazooka_data[i].size,
			    BAZOOKA_ERASER_REALLOC_2,
			    size - bazooka_data[i].size);
		  }
		memset (&(bazooka_data[i]), 0, sizeof (_lw6sys_bazooka_t));
		bazooka_data[i].ptr = ptr2;
		bazooka_data[i].size = size;
		file_only = strrchr (file, '/');
		if (file_only && *file_only)
		  {
		    file_only++;
		  }
		else
		  {
		    file_only = file;
		  }
		strncpy (bazooka_data[i].file, file_only,
			 _LW6SYS_BAZOOKA_FILE_SIZE - 1);
		bazooka_data[i].line = line;
		bazooka_data[i].timestamp = lw6sys_timestamp ();
		break;		// important to leave loop, else serious perfomance problem
	      }
	  }
      }

      bazooka_unlock ();
    }
}

void
_lw6sys_bazooka_register_free (char *ptr)
{
  if (bazooka_data)
    {
      bazooka_lock ();

      {
	int i;

	bazooka_free_counter++;

	for (i = 0; i < bazooka_size; ++i)
	  {
	    if (bazooka_data[i].ptr == ptr)
	      {
		if (bazooka_eraser)
		  {
		    memset (bazooka_data[i].ptr, BAZOOKA_ERASER_FREE,
			    bazooka_data[i].size);
		  }
		memset (&(bazooka_data[i]), 0, sizeof (_lw6sys_bazooka_t));
		break;		// important to leave loop, else serious perfomance problem
	      }
	  }
      }

      bazooka_unlock ();
    }
}

/**
 * lw6sys_get_memory_bazooka_malloc_count
 *
 * Provided you have always called the @LW6SYS_MALLOC an @LW6SYS_CALLOC to 
 * allocate memory, this function will tell you how many times @malloc 
 * has been called.
 *
 * Return value: the number of calls to @lw6sys_malloc or @lw6sys_calloc since
 *   program was started.
 */
int
lw6sys_get_memory_bazooka_malloc_count ()
{
  return bazooka_malloc_counter;
}

/**
 * lw6sys_get_memory_bazooka_free_count
 *
 * Provided you have always called the @LW6SYS_FREE macro to free
 * memory, this function will tell you how many times @free has been called.
 *
 * Return value: the number of calls to @lw6sys_free since
 *   program was started.
 */
int
lw6sys_get_memory_bazooka_free_count ()
{
  return bazooka_free_counter;
}

static void
report_line (_lw6sys_bazooka_t * bazooka)
{
  _lw6sys_bazooka_t local_bazooka;
  int sample_int = 0;
  char sample_str[_LW6SYS_BAZOOKA_SAMPLE_SIZE];
  char *pos;
  char ctime_buf[_LW6SYS_LOG_CTIME_BUF_SIZE];
  time_t t;

  // make a local copy, log functions can modify our data...
  memcpy (&local_bazooka, bazooka, sizeof (_lw6sys_bazooka_t));

  memset (sample_str, 0, _LW6SYS_BAZOOKA_SAMPLE_SIZE);
  if (local_bazooka.size > 0)
    {
      memcpy (sample_str, local_bazooka.ptr,
	      lw6sys_min (local_bazooka.size - 1,
			  _LW6SYS_BAZOOKA_SAMPLE_SIZE - 1));
    }
  sample_int = *((int *) sample_str);
  for (pos = sample_str; *pos; pos++)
    {
      // get rid of stange system characters
      if ((*pos) < 32)
	{
	  (*pos) = ' ';
	}
    }

  t = time (NULL);
  ctime_r (&t, ctime_buf);
  ctime_buf[_LW6SYS_LOG_CTIME_ZERO] = '\0';

  lw6sys_log (LW6SYS_LOG_NOTICE,
	      _
	      ("memory bazooka found unfreed data ptr=%p size=%d file:line=\"%s:%d\" time=\"%s\" sample_int=%d sample_str=\"%s\""),
	      local_bazooka.ptr, local_bazooka.size, local_bazooka.file,
	      local_bazooka.line, ctime_buf, sample_int, sample_str);
}

/**
 * lw6sys_memory_bazooka_report
 *
 * Reports memory bazooka diagnostics on the console. Carefull, this
 * one is not reentrant, call at the end of your program when all threads
 * are joined.
 *
 * Return value: 1 if no allocated stuff left, 0 if there are still malloc'ed stuff
 */
int
lw6sys_memory_bazooka_report ()
{
  int ret = 1;
  int recreate_mutex = 0;
  int i;
  int malloc_counter;
  int free_counter;

  if (bazooka_mutex)
    {
      bazooka_destroy_mutex ();
      recreate_mutex = 1;
    }

  malloc_counter = lw6sys_get_memory_bazooka_malloc_count ();
  free_counter = lw6sys_get_memory_bazooka_free_count ();

  if (malloc_counter == free_counter)
    {
      if (bazooka_data)
	{
	  lw6sys_log (LW6SYS_LOG_NOTICE, _("%d malloc calls"),
		      malloc_counter);
	}
    }
  else
    {
      ret = 0;
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("possible memory leak, %d calls to malloc and %d calls to free, note that if the program exited abnormally because of an unexpected error, this difference might be \"normal\""),
		  malloc_counter, free_counter);
    }

  if (bazooka_data)
    {
      for (i = 0; i < bazooka_size; ++i)
	{
	  if (bazooka_data[i].ptr)
	    {
	      ret = 0;
	      report_line (&(bazooka_data[i]));
	    }
	}
    }

  if (recreate_mutex)
    {
      bazooka_mutex = lw6sys_mutex_create ();
    }

  return ret;
}
