/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "sys.h"

#define RANDOM_FLOAT_PRECISION 1000000

/**
 * lw6sys_random:
 *
 * @range: the high limit for random generated numbers. If you want 
 *         random numbers between 0 and 5, set this to 6.
 * 
 * Wrapper over standard random function. This one is thread safe.
 *
 * @Return value: a pseudo-random number 
 */
u_int32_t
lw6sys_random (u_int32_t range)
{
  static unsigned int dirty_seed = 0;
  u_int32_t ret = 0;

  dirty_seed += lw6sys_timestamp ();

#ifdef LW6_MS_WINDOWS
  /*
   * rand_r not available on Microsoft proprietary platform. Rather
   * than calling rand() which is not said to be thread-safe, we rely
   * on a dummy checksum, since dirty_seed is incremented at each
   * call, it should still give a (poor...) illusion of random numbers.
   */
  ret = lw6sys_checksum_int32 (dirty_seed);
#else
  ret =
    (((u_int64_t) rand_r (&dirty_seed)) * ((u_int64_t) range)) /
    (((u_int64_t) RAND_MAX) + 1);
#endif

  return ret;
}

/**
 * lw6sys_random_float:
 *
 * @min: the min value, as a float
 * @max: the max value, as a float
 * 
 * Returns a random float number between min & max. Can be equal to
 * min or max.
 *
 * @Return value: a pseudo-random number 
 */
float
lw6sys_random_float (float min, float max)
{
  float ret = 0.0f;

  ret =
    (((lw6sys_random (RANDOM_FLOAT_PRECISION) * (max - min))) /
     (float) RANDOM_FLOAT_PRECISION) + min;

  return ret;
}
