/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "../../gfx.h"
#include "gl-utils.h"
#include "gl-utils-internal.h"

/*
 * Initialize display.
 */
int
mod_gl_utils_set_video_mode (mod_gl_utils_context_t * utils_context,
			     int width, int height, int fullscreen,
			     void (*resize_callback) (int width, int height,
						      int fullscreen))
{
  /* Information about the current video settings. */
  const SDL_VideoInfo *info = NULL;
  /* Color depth in bits of our window. */
  int bpp = 0;
  /* Flags we will pass into SDL_SetVideoMode. */
  int flags = 0;

  lw6sys_sleep (utils_context->const_data.mode_sleep);

  utils_context->resize_callback = resize_callback;

  /* Let's get some video information. */
  info = SDL_GetVideoInfo ();

  if (info)
    {
      lw6sys_log (LW6SYS_LOG_INFO,
		  _
		  ("SDL VideoInfo hw_available=%d wm_available=%d video_mem=%dkb"),
		  info->hw_available, info->wm_available, info->video_mem);
    }
  else
    {
      /* This should probably never happen. */
      lw6sys_log (LW6SYS_LOG_ERROR,
		  _("SDL GetVideoInfo failed: \"%s\""), SDL_GetError ());
      fflush (stderr);
      return 0;
    }

  bpp = info->vfmt->BitsPerPixel;

  /*
   * Now, we want to setup our requested
   * window attributes for our OpenGL window.
   * We want *at least* 5 bits of red, green
   * and blue. We also want at least a 16-bit
   * depth buffer.
   *
   * The last thing we do is request a double
   * buffered window. '1' turns on double
   * buffering, '0' turns it off.
   *
   * Note that we do not use SDL_DOUBLEBUF in
   * the flags to SDL_SetVideoMode. That does
   * not affect the GL attribute state, only
   * the standard 2D blitting setup.
   */
  SDL_GL_SetAttribute (SDL_GL_RED_SIZE, 5);
  SDL_GL_SetAttribute (SDL_GL_GREEN_SIZE, 5);
  SDL_GL_SetAttribute (SDL_GL_BLUE_SIZE, 5);
  SDL_GL_SetAttribute (SDL_GL_DEPTH_SIZE, 16);
  SDL_GL_SetAttribute (SDL_GL_DOUBLEBUFFER, 1);

  flags = SDL_OPENGL | (fullscreen ? SDL_FULLSCREEN : SDL_RESIZABLE);

  if (width <= 0 || height <= 0)
    {
      mod_gl_utils_get_standard_resolution (utils_context, &width, &height);
    }

  if (fullscreen)
    {
      mod_gl_utils_find_closest_resolution (utils_context, &width, &height,
					    width, height);
    }

  /*
   * Set the video mode
   */
  if (SDL_SetVideoMode (width, height, bpp, flags) == 0)
    {
      /* 
       * This could happen for a variety of reasons,
       * including DISPLAY not being set, the specified
       * resolution not being available, etc.
       */
      lw6sys_log (LW6SYS_LOG_ERROR,
		  _("SDL SetVideoMode %dx%d fullscreen=%d failed: \"%s\""),
		  width, height, fullscreen, SDL_GetError ());
      fflush (stderr);
      return 0;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_INFO, _("SDL SetVideoMode %dx%d fullscreen=%d"),
		  width, height, fullscreen);
      utils_context->screen.width = width;
      utils_context->screen.height = height;
      utils_context->screen.bpp = bpp;
      utils_context->screen.fullscreen = fullscreen;
      mod_gl_utils_sync_viewport (utils_context);

      glGetIntegerv (GL_MAX_TEXTURE_SIZE,
		     &(utils_context->caps.max_texture_size));
      if (!utils_context->caps.max_texture_size)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("unable to get MAX_TEXTURE_SIZE"));
	  utils_context->caps.max_texture_size = 512;
	}
    }

  SDL_WM_SetCaption (lw6sys_build_get_package_string (),
		     lw6sys_build_get_package_tarname ());

  lw6sys_log (LW6SYS_LOG_INFO, _("%d BPP"),
	      SDL_GetVideoSurface ()->format->BitsPerPixel);
  lw6sys_log (LW6SYS_LOG_INFO, _("OpenGL vendor \"%s\""),
	      glGetString (GL_VENDOR));
  lw6sys_log (LW6SYS_LOG_INFO, _("OpenGL renderer \"%s\""),
	      glGetString (GL_RENDERER));
  lw6sys_log (LW6SYS_LOG_INFO, _("OpenGL version \"%s\""),
	      glGetString (GL_VERSION));

  lw6sys_log (LW6SYS_LOG_INFO,
	      _("OpenGL MAX_TEXTURE_SIZE = %d"),
	      utils_context->caps.max_texture_size);

  if (utils_context->caps.max_texture_size <
      MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE)
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("OpenGL driver only handles %dx%d textures when %dx%d textures are needed"),
		  utils_context->caps.max_texture_size,
		  utils_context->caps.max_texture_size,
		  MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE,
		  MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE);
    }

  /*
   * Call this function to be sure we're truely in some known mode
   */
  mod_gl_utils_set_render_mode_2d (utils_context);

  resize_callback (width, height, fullscreen);

  lw6sys_sleep (utils_context->const_data.mode_sleep);

  return 1;
}


/*
 * Called whenever window resize is asked for.
 */
int
mod_gl_utils_resize_video_mode (mod_gl_utils_context_t * utils_context,
				int width, int height)
{
  int ret = 0;
  int fullscreen = 0;

  fullscreen = utils_context->screen.fullscreen;
  if (width != utils_context->screen.width
      || height != utils_context->screen.height)
    {
      int flags = 0;

      flags = SDL_OPENGL | (fullscreen ? SDL_FULLSCREEN : SDL_RESIZABLE);

      if (SDL_SetVideoMode (width, height, utils_context->screen.bpp, flags)
	  == 0)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("unable to resize screen %dx%d fullscreen=%d"), width,
		      height, fullscreen);
	}
      else
	{
	  utils_context->screen.width = width;
	  utils_context->screen.height = height;
	  mod_gl_utils_sync_viewport (utils_context);

	  if (utils_context->resize_callback)
	    {
	      utils_context->resize_callback (width, height, fullscreen);
	    }

	  lw6sys_log (LW6SYS_LOG_INFO,
		      _("resizing screen %dx%d"), width, height);

	  ret = 1;
	}
    }
  else
    {
      /*
       * Nothing to do...
       */
      ret = 1;
    }

  return ret;
}

int
mod_gl_utils_get_video_mode (mod_gl_utils_context_t * utils_context,
			     int *width, int *height, int *fullscreen)
{
  int ret = 0;

  (*width) = utils_context->screen.width;
  (*height) = utils_context->screen.height;
  (*fullscreen) = utils_context->screen.fullscreen;

  ret = 1;

  return ret;
}

/*
 * Update viewport
 */
int
mod_gl_utils_sync_viewport (mod_gl_utils_context_t * utils_context)
{
  int ret = 1;

  glViewport (0, 0, utils_context->screen.width,
	      utils_context->screen.height);

  return ret;
}

/*
 * Force mode.
 */
int
mod_gl_utils_sync_mode (mod_gl_utils_context_t * utils_context, int force)
{
  int ret = 0;
  int width = 0;
  int height = 0;
  int fullscreen = 0;
  int flags = 0;

  mod_gl_utils_get_video_mode (utils_context, &width, &height, &fullscreen);
  flags = SDL_OPENGL | (fullscreen ? SDL_FULLSCREEN : SDL_RESIZABLE);

  if (force)
    {
      if (SDL_SetVideoMode (width, height, utils_context->screen.bpp, flags)
	  == 0)
	{
	  mod_gl_utils_sync_viewport (utils_context);
	  ret = 1;
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING,
		      _("unable to sync screen %dx%d fullscreen=%d"), width,
		      height, fullscreen);
	}
    }
  else
    {
      mod_gl_utils_sync_viewport (utils_context);
      ret = 1;
    }

  return ret;
}
