/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <math.h>

#include "ldr.h"

void
lw6ldr_resampler_init (lw6ldr_resampler_t * resampler,
		       lw6map_param_t * param, lw6ldr_hints_t * hints,
		       int source_w,
		       int source_h,
		       int display_w, int display_h, float target_ratio)
{
  int target_w = 0;
  int target_h = 0;
  float f;
  float fighter_scale;

  memset (resampler, 0, sizeof (lw6ldr_resampler_t));

  target_w = source_w;
  target_h = source_h;

  if (source_w > 0 && source_h > 0)
    {
      if (hints->resample)
	{
	  // 0*0=0 so size must be non-zero
	  if (hints->fighter_scale > 0.0f)
	    {
	      fighter_scale = hints->fighter_scale;
	    }
	  else
	    {
	      fighter_scale = LW6LDR_HINTS_DEFAULT_FIGHTER_SCALE;
	    }
	  target_w =
	    lw6sys_max (display_w * param->style.zoom / fighter_scale,
			LW6MAP_MIN_SIDE);
	  target_h =
	    lw6sys_max (display_h * param->style.zoom / fighter_scale,
			LW6MAP_MIN_SIDE);

	  if (target_w > 0 && target_h > 0)
	    {
	      f = target_ratio / (((float) target_w) / ((float) target_h));
	      if (f > 1.0f)
		{
		  target_w *= f;
		}
	      else if (f < 1.0f)
		{
		  target_h /= f;
		}
	    }

	  if (target_w * target_h > hints->max_map_surface)
	    {
	      f =
		sqrt ((((float) target_w) * ((float) target_h)) /
		      ((float) hints->max_map_surface));
	      target_w /= f;
	      target_h /= f;
	    }

	  if (target_w * target_h < hints->min_map_surface)
	    {
	      f =
		sqrt (((float) hints->min_map_surface) /
		      (((float) target_w) * ((float) target_h)));
	      target_w *= f;
	      target_h *= f;
	    }

	  target_w = lw6sys_min (target_w, hints->max_map_width);
	  target_h = lw6sys_min (target_h, hints->max_map_height);

	  target_w = lw6sys_max (target_w, hints->min_map_width);
	  target_h = lw6sys_max (target_h, hints->min_map_height);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to handle map with shape %dx%d"), source_w,
		  source_h);
    }

  resampler->target_w = target_w;
  resampler->target_h = target_h;
  resampler->source_w = source_w;
  resampler->source_h = source_h;
  resampler->scale_x =
    ((float) (resampler->target_w)) / ((float) (resampler->source_w));
  resampler->scale_y =
    ((float) (resampler->target_h)) / ((float) (resampler->source_h));
}

static void
check_bounds (int *x, int *y, int w, int h)
{
  (*x) = lw6sys_min ((*x), w - 1);
  (*y) = lw6sys_min ((*y), h - 1);
  (*x) = lw6sys_max ((*x), 0);
  (*y) = lw6sys_max ((*y), 0);
}

void
lw6ldr_resampler_force (lw6ldr_resampler_t * resampler,
			int source_w,
			int source_h, int target_w, int target_h)
{
  resampler->target_w = target_w;
  resampler->target_h = target_h;
  resampler->source_w = source_w;
  resampler->source_h = source_h;
  resampler->scale_x =
    ((float) (resampler->target_w)) / ((float) (resampler->source_w));
  resampler->scale_y =
    ((float) (resampler->target_h)) / ((float) (resampler->source_h));
}

void
lw6ldr_resampler_source2target (lw6ldr_resampler_t * resampler, int *target_x,
				int *target_y, int source_x, int source_y)
{
  (*target_x) = (int) (source_x * resampler->scale_x);
  (*target_y) = (int) (source_y * resampler->scale_y);
  check_bounds (target_x, target_y, resampler->target_w, resampler->target_h);
}

void
lw6ldr_resampler_target2source (lw6ldr_resampler_t * resampler, int *source_x,
				int *source_y, int target_x, int target_y)
{
  (*source_x) = (int) (target_x / resampler->scale_x);
  (*source_y) = (int) (target_y / resampler->scale_y);
  check_bounds (source_x, source_y, resampler->source_w, resampler->source_h);
}
