//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <gtkmm/builder.h>
#include "listerola-window.h"
#include "boxcompose.h"
#include "timing.h"
#include "file.h"
#include "event-box.h"
#include "host-box.h"
#include "host-list.h"
#include "host.h"
#include "event.h"
#include "event-date-region.h"
#include "hosted-event-list.h"
#include "listerola.h"

std::string get_glade_path ()
{
  return File::get_data_path () + "/../glade";
}

ListerolaWindow::ListerolaWindow (std::string listing_file,
				  double num_row_channels, 
                                  double num_column_hours, double row_speed,
				  double row_pause, double col_interval,
                                  double flipahead,
				  std::string advertisement_file,
				  double vertical_step,
				  double row_height_in_pixels,
				  double hour_width_in_pixels,
				  double channel_width_in_pixels)
  :hadj(0,0,0,0,0,0), vadj(0,0,0,0,0,0)
{
  listing_filename = listing_file;
  row_channels = num_row_channels;
  column_hours = num_column_hours;
  scroll_speed = row_speed;
  scroll_row_pause = row_pause;
  vertical_scroll_step = vertical_step;
  row_height = row_height_in_pixels;
  flipahead_interval = flipahead;
  column_interval = col_interval;
  advertisement_image_filename = advertisement_file;
  hour_width = hour_width_in_pixels;
  channel_width = channel_width_in_pixels;

  Glib::RefPtr < Gtk::Builder > xml =
    Gtk::Builder::create_from_file (get_glade_path () + "/listerola.gtk");

  xml->get_widget ("window", window);

  xml->get_widget ("advertisement_image", advertisement_image);
  Gtk::HBox *listerola_viewport_container;
  xml->get_widget ("listerola_viewport_container", 
                   listerola_viewport_container);
  xml->get_widget ("listing_hbox", listing_hbox);
  xml->get_widget ("listing_vbox", listing_vbox);
  listerola_viewport = new Gtk::Viewport(hadj, vadj);
  scroll_hbox = new Gtk::HBox();
  listerola_viewport->add(*manage(scroll_hbox));
  scroll_vbox = new Gtk::VBox();
  scroll_hbox->add(*manage(scroll_vbox));
  listerola_viewport_container->add(*manage(listerola_viewport));

  xml->get_widget ("header_vbox", header_vbox);
  xml->get_widget ("header_hbox", header_hbox);

  xml->get_widget ("ad_hbox", ad_hbox);
  xml->get_widget ("ad_vbox", ad_vbox);
  EventDateRegion now(num_column_hours);
  now.decrement_to_nearest_interval(column_interval);
  Listerola::getInstance ()->set_current_date (now);
  update_listing_file (listing_file);
  if (advertisement_file != "")
    update_advertisement_image (advertisement_file, "none");
  update_scroll_box_size (row_channels, column_hours);
  update_scrolling_speed (row_speed);
    
  second_handler = Timing::instance ().register_timer
      (sigc::mem_fun (*this, &ListerolaWindow::on_second_passed), 1000);
  fill_headers();
}

void ListerolaWindow::fill_headers()
{
  header_hbox->children ().erase (header_hbox->children ().begin (),
                                  header_hbox->children ().end ());
  EventDate nowdate;
  timebox =
    new FrameBox (Box::ucompose("%1", nowdate.to_string(false)), 
                  Gdk::Color("white"), double (row_height), channel_width);
  timebox->set_background_colour(Gdk::Color("light blue"));
  header_hbox->pack_start(*timebox, Gtk::PACK_SHRINK, 0);
  EventDate then = Listerola::getInstance()->get_current_date();
  for (double i = 0; i < column_hours; i+=column_interval)
    {
      double width = hour_width * column_interval;
      if (i + column_interval > column_hours)
        width = column_hours + column_interval - i;
      FrameBox * interval =
        new FrameBox (Box::ucompose("%1", then.to_string()), 
                      Gdk::Color("white"), double (row_height), width);
      interval->set_background_colour(Gdk::Color("light blue"));
      header_hbox->pack_start(*interval, Gtk::PACK_SHRINK, 0);
      then.add_hours(column_interval);
    }
  header_hbox->show_all();
}

bool
ListerolaWindow::on_second_passed()
{
  Listerola *l = Listerola::getInstance();
  double then = l->get_current_date().get_time().as_double();

  EventDate nowdate;
  double now = nowdate.get_time().as_double();

  double diff = now - then;
  if (diff >= (column_interval * 60 * 60) - (flipahead_interval * 60))
    {
      EventDateRegion newnow = l->get_current_date();
      newnow.add_hours(column_interval);
      l->set_current_date(newnow);
      on_column_scrolled();
    }

  Gtk::HBox *contents = Box::ucompose("%1", nowdate.to_string(false));
  timebox->set_box_contents(contents);
  timebox->show_all();
  return Timing::CONTINUE;
}

      
void ListerolaWindow::on_column_scrolled()
{
  fill_headers();
  update_scroll_box_size (row_channels, column_hours);
}

void
ListerolaWindow::hide ()
{
  window->hide ();
}

bool
ListerolaWindow::run ()
{
  window->show_all ();
  return true;
}

double
get_random_hours ()
{
  return double (((rand () % 4) + 1) * 0.5);
}

void
ListerolaWindow::update_listing_file (std::string listing_file)
{
  HostList *channels = HostList::getInstance ();
  Host *channel;
  Event *show;
  char buf[32];
  std::string genre;
  for (int i = 0; i < 57; i++)	//57 channels with nothing on
    {
      snprintf (buf, sizeof (buf), "channel %d", i + 1);
      channel = new Host (buf, i + 1);
      channels->add (channel);
      EventDateRegion now = Listerola::getInstance()->get_current_date();
      now.set_duration(get_random_hours());
      for (int j = 0; j < 100; j++)
	{
	  snprintf (buf, sizeof (buf), "ch%d show%d", i + 1, j + 1);
	  switch (rand () % 6)
	    {
	    case 0:
	      genre = "pink";
	      break;
	    case 1:
	      genre = "green";
	      break;
	    case 2:
	      genre = "red";
	      break;
	    case 3:
	      genre = "brown";
	      break;
	    case 4:
	      genre = "grey";
	      break;
	    case 5:
	      genre = "yellow";
	      break;
	    }
	  show = new Event (buf, genre, now);
	  channel->add (show);
	  now.add_hours (now.get_duration ());
	  now.set_duration (get_random_hours ());
	}
    }
}

void
ListerolaWindow::set_listing_file (std::string listing_file)
{
  listing_filename = listing_file;
  update_listing_file (listing_file);
}

void
ListerolaWindow::set_number_of_channels (double num_row_channels)
{
  row_channels = num_row_channels;
}

void
ListerolaWindow::set_number_of_hours (double num_column_hours)
{
  column_hours = num_column_hours;
}

void
ListerolaWindow::update_scrolling_speed (double tick_speed)
{
  scroll_tick_handler.disconnect ();
  if (tick_speed > 0.0)
    scroll_tick_handler = Timing::instance ().register_timer
      (sigc::mem_fun (*this, &ListerolaWindow::on_vertical_scroll_tick),
       int (tick_speed));
}

Gtk::HBox * ListerolaWindow::get_next_row ()
{
  static guint32 channel_id = 1;
  Gdk::Color pink;
  pink.set_rgb_p ((rand () % 100) / 100.0, (rand () % 100) / 100.0,
		  (rand () % 100) / 100.0);
  Gtk::HBox * box = new Gtk::HBox ();
  HostList * channels = HostList::getInstance ();
  Host * channel = (*channels)[channel_id];
  Listerola * l = Listerola::getInstance ();
  EventList events = channel->get_events (l->get_current_date ());
  Gtk::EventBox * hostbox = new HostBox (channel, double (row_height), 
                                         channel_width);
  box->pack_start (*hostbox, Gtk::PACK_SHRINK, 0);
  for (EventList::iterator it = events.begin (); it != events.end (); it++)
    {
      Event * event = *it;
      Listerola *l = Listerola::getInstance();
      double duration = event->get_date().get_duration();
      if (event->get_date().straddles(l->get_current_date().get_ending_date()))
        {
          double overlap = 
            event->get_date().overlap_end(l->get_current_date().get_ending_date());
          duration -= overlap;
        }
      if (event->get_date().straddles(l->get_current_date()))
        {
          double overlap = event->get_date().overlap_start(l->get_current_date());
          duration -= overlap;
        }
      Gtk::EventBox * ebox =
        new EventBox (event, double (row_height), hour_width * duration);
      box->pack_start (*ebox, Gtk::PACK_SHRINK, 0);
      if (event->get_date ().
          straddles (l->get_current_date ().get_ending_date ()))
        break;
      if (l->get_current_date().get_ending_date() < event->get_date())
        break;
    }

  channel_id++;
  if (channel_id > channels->size ())
    channel_id = 1;
  return box;
}

void
ListerolaWindow::on_row_scrolled (Gtk::HBox *row)
{
  add_row_to_scroll_box (get_next_row());
  remove_row_from_scroll_box(row);
  listerola_viewport->show_all();
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  listerola_viewport->get_vadjustment()->set_value(0);

}

bool
ListerolaWindow::on_vertical_scroll_tick ()
{
  listerola_viewport->get_vadjustment()->set_value(listerola_viewport->get_vadjustment()->get_value() + (vertical_scroll_step * 1.0));
  if (int (listerola_viewport->get_vadjustment()->get_value ()) % row_height == 0)
    {
      bool retval = Timing::CONTINUE;
      if (scroll_row_pause > 0.0)
	retval = on_vertical_scroll_pause ();
  
      Gtk::HBox *row = 
        dynamic_cast<Gtk::HBox*>(scroll_vbox->children().front().get_widget());
      on_row_scrolled(row);
      return retval;
    }
  return Timing::CONTINUE;
}

bool
ListerolaWindow::on_vertical_scroll_pause ()
{
  scroll_tick_handler.disconnect ();
  scroll_tick_handler = Timing::instance ().register_timer
    (sigc::mem_fun (*this, &ListerolaWindow::on_vertical_scroll_resume),
     int (scroll_row_pause));
  return Timing::STOP;
}

bool
ListerolaWindow::on_vertical_scroll_resume ()
{
  update_scrolling_speed (scroll_speed);
  return Timing::STOP;
}

void
ListerolaWindow::set_scrolling_speed (double tick_speed)
{
  scroll_speed = tick_speed;
  update_scrolling_speed (tick_speed);
}

void
ListerolaWindow::set_scrolling_row_pause (double row_pause)
{
  scroll_row_pause = row_pause;
}

void
ListerolaWindow::set_column_flipahead_time (double interval)
{
  flipahead_interval = interval;
}

void
ListerolaWindow::update_advertisement_image (std::string file,
					     std::string wipestyle)
{
  advertisement_image->property_pixbuf () =
    Gdk::Pixbuf::create_from_file (file);
}

void
ListerolaWindow::set_advertisement_image (std::string image_filename,
					  std::string wipestyle)
{
  advertisement_image_filename = image_filename;
  update_advertisement_image (image_filename, wipestyle);
}

void
ListerolaWindow::clear_scroll_box ()
{
  scroll_vbox->children ().erase (scroll_vbox->children ().begin (),
				  scroll_vbox->children ().end ());
}

void
ListerolaWindow::add_row_to_scroll_box (Gtk::HBox * box)
{
  scroll_vbox->pack_start (*box, Gtk::PACK_SHRINK, 0);
}

void ListerolaWindow::remove_row_from_scroll_box(Gtk::HBox *row)
{
  scroll_vbox->children().remove(*row);
}

void
ListerolaWindow::update_scroll_box_size (double rows, double columns)
{
  clear_scroll_box ();

  for (int i = 0; i < rows + 1; i++)
    {
      Gtk::HBox * box = get_next_row ();
      add_row_to_scroll_box (box);
    }
  listerola_viewport->set_size_request (channel_width +
				      (column_hours * hour_width),
				      row_height * rows);
  Gtk::Adjustment * a = listerola_viewport->get_vadjustment ();
  a->set_upper(row_height * rows);
  listerola_viewport->show_all ();
}
